%%%%% mindist_interatomic.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 11.7.2018
%%%%% 
%%%%% This function gives the distance from each moving atom to its nearest neighbour atom in configuration C.
%%%%%
%%%%% The input vectors in x are assumed to be row vectors including the
%%%%% coordinates of the moving atoms: [x_1,y_1,z_1,x_2,y_2,z_2,...].
%%%%%
%%%%% Input:   x            coordinates of the moving atoms in configurations C (n x 3*N_mov)
%%%%%          conf_info    structure array including information about the configurations necessary for the GP model
%%%%%                       - conf_info.conf_fro: coordinates of active frozen atoms (N_fro x 3)
%%%%%                       - conf_info.atomtype_mov: atomtype indices for moving atoms (1 x N_mov)
%%%%%                       - conf_info.atomtype_fro: pairtype indices for active frozen atoms (1 x N_fro)
%%%%%                       - conf_info.pairtype: pairtype indices for pairs of atomtypes (n_at x n_at)
%%%%%                       - conf_info.n_pt: number of active pairtypes
%%%%%
%%%%% Output:  dist         matrix including the minimum interatomic distances for each moving atom in configurations C (n x N_mov)

function dist = mindist_interatomic(x,conf_info)

    conf_fro = conf_info.conf_fro;
    [n,N_mov] = size(x);
    N_mov = N_mov/3;
    N_fro = size(conf_fro,1);
    dist = inf*ones(n,N_mov);
    % distances between moving atoms
    if N_mov > 1
        for i = 1:N_mov-1
            for j = (i+1):N_mov
                r_ij = sqrt(sum((x(:,(j*3-2):(j*3))-x(:,(i*3-2):(i*3))).^2,2));
                dist(:,i) = min(dist(:,i),r_ij);
                dist(:,j) = min(dist(:,j),r_ij);
            end
        end
    end
    % distances from moving atoms to active frozen atoms
    if N_fro > 0
        for j = 1:N_mov
            for i = 1:N_fro
                r_ij = sqrt(sum((bsxfun(@minus,x(:,(j*3-2):(j*3)),conf_fro(i,1:3))).^2,2));
                dist(:,j) = min(dist(:,j),r_ij);
            end
        end
    end
    
end
