%%%%% force_NEB2.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 22.2.2019
%%%%%
%%%%% This function gives the NEB force for each image on the path according to
%%%%% the NEB method.
%%%%%
%%%%% Input:   R            coordinates of the 'N_im' images on the path
%%%%%          E_R          energy at the 'N_im' images on the path
%%%%%          G_R          gradient at the 'N_im' images on the path
%%%%%          param_force  parallel spring constant
%%%%%          CI_on        1 if the climbing image option is in use, 0 if not
%%%%%
%%%%% Output:  F_R          NEB force acting on the 'N_im'-2 intermediate images
%%%%%          maxG_R_perp  maximum component of the gradient perpendicular to the path tangent at the 'N_im'-2 intermediate images
%%%%%          maxG_CI      maximum component of the gradient at the climbing image (zero if CI is off)
%%%%%          i_CI         index of the climbing image (among the 'N_im'-2 intermediate images)

function [F_R,maxG_R_perp,maxG_CI,i_CI] = force_NEB2(R,E_R,G_R,param_force,CI_on)
    k_par = param_force;
    N_im = size(R,1);
    D = size(R,2);
    Z_R = tangent(R,E_R);
    G_R_perp = G_R(2:(N_im-1),:) - repmat(sum(G_R(2:(N_im-1),:).*Z_R,2),1,D).*Z_R;
    maxG_R_perp = max(abs(G_R_perp),[],2);
    int_R_next = R(3:N_im,:) - R(2:(N_im-1),:);
    int_R_prev = R(2:(N_im-1),:) - R(1:(N_im-2),:);
    d_R_next = sqrt(sum(int_R_next.^2,2));
    d_R_prev = sqrt(sum(int_R_prev.^2,2));
    if CI_on > 0
        [~,i_CI] = max(E_R(2:(N_im-1),:));
    else
        maxG_CI = 0;
        i_CI = 0;
    end    
    F_R_spring_par = repmat(k_par*(d_R_next-d_R_prev),1,D).*Z_R;
    F_R = -G_R_perp + F_R_spring_par;
    if CI_on > 0        
        F_R(i_CI,:) = -G_R(1+i_CI,:) + 2*G_R(1+i_CI,:)*Z_R(i_CI,:)'*Z_R(i_CI,:);
        maxG_CI = max(abs(G_R(1+i_CI,:)));
    end
end
