%%%%% dist_max1Dlog.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 11.7.2018
%%%%% 
%%%%% This function gives the maximum difference in the logarithmic inter-atomic distances
%%%%% between atomic configurations C and C'.
%%%%%
%%%%% dist(C,C') = MAX_ij{|log(r_ij')-log(r_ij)|} = MAX_ij{|log(r_ij'/r_ij)|}, where r_ij and
%%%%% r_ij' are the distances between atoms i and j in configurations C and C', respectively.
%%%%%
%%%%% The input vectors x1 and x2 are assumed to be row vectors including the
%%%%% coordinates of the moving atoms: [x_1,y_1,z_1,x_2,y_2,z_2,...].
%%%%%
%%%%% Input:   x1           coordinates of the moving atoms in configurations C (n1 x 3*N_mov)
%%%%%          x2           coordinates of the moving atoms in configurations C' (n2 x 3*N_mov)
%%%%%          conf_info    structure array including information about the configurations necessary for the GP model
%%%%%                       - conf_info.conf_fro: coordinates of active frozen atoms (N_fro x 3)
%%%%%                       - conf_info.atomtype_mov: atomtype indices for moving atoms (1 x N_mov)
%%%%%                       - conf_info.atomtype_fro: pairtype indices for active frozen atoms (1 x N_fro)
%%%%%                       - conf_info.pairtype: pairtype indices for pairs of atomtypes (n_at x n_at)
%%%%%                       - conf_info.n_pt: number of active pairtypes
%%%%%
%%%%% Output:  dist         matrix including the "distances" from configurations C to configurations C' (n1 x n2)

function dist = dist_max1Dlog(x1,x2,conf_info)
    
    conf_fro = conf_info.conf_fro;
    [n1,N_mov] = size(x1);
    N_mov = N_mov/3;
    N_fro = size(conf_fro,1);
    n2 = size(x2,1);
    dist = zeros(n1,n2);
    % distances between moving atoms
    if N_mov > 1
        for i = 1:N_mov-1
            for j = (i+1):N_mov
                r_ij_1 = sqrt(sum((x1(:,(j*3-2):(j*3))-x1(:,(i*3-2):(i*3))).^2,2));
                r_ij_2 = sqrt(sum((x2(:,(j*3-2):(j*3))-x2(:,(i*3-2):(i*3))).^2,2));
                dist = max(dist,abs(log(bsxfun(@rdivide,r_ij_2',r_ij_1))));
            end
        end
    end
    % distances from moving atoms to active frozen atoms
    if N_fro > 0
        for j = 1:N_mov
            for i = 1:N_fro
                r_ij_1 = sqrt(sum((bsxfun(@minus,x1(:,(j*3-2):(j*3)),conf_fro(i,1:3))).^2,2));
                r_ij_2 = sqrt(sum((bsxfun(@minus,x2(:,(j*3-2):(j*3)),conf_fro(i,1:3))).^2,2));
                dist = max(dist,abs(log(bsxfun(@rdivide,r_ij_2',r_ij_1))));
            end
        end
    end
    
end
