%%%%% dist_at.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 11.7.2018
%%%%% 
%%%%% This function gives the distance between two atomic configurations as defined in
%%%%% the special GPstuff covariance function 'gpcf_matern32at'.
%%%%% The distance between configurations C and C' is based on the changes of
%%%%% the inter-atomic distances:
%%%%%
%%%%% dist(C,C') = sqrt(SUM_ij{[(1/r_ij-1/r_ij')/l_ij]^2}), where r_ij and
%%%%% r_ij' are the distances between atoms i and j in configurations C and
%%%%% C', respectively, and l_ij is the lengthscale of the corresponding
%%%%% atom pair type.
%%%%%
%%%%% The input vectors x1 and x2 are assumed to be row vectors including the
%%%%% coordinates of the moving atoms: [x_1,y_1,z_1,x_2,y_2,z_2,...].
%%%%%
%%%%% The parameter 'conf_info' is a structure array including necessary information about the configurations:
%%%%% conf_info.conf_fro: coordinates of active frozen atoms (N_fro x 3)
%%%%% conf_info.atomtype_mov: atomtype indices for moving atoms (1 x N_mov)
%%%%% conf_info.atomtype_fro: atomtype indices for active frozen atoms (1 x N_fro)
%%%%% Atomtypes must be indexed as 1,2,...,n_at (may include also inactive atomtypes).
%%%%% conf_info.pairtype: pairtype indices for pairs of atomtypes (n_at x n_at)
%%%%% conf_info.n_pt: number of active pairtypes
%%%%% Active pairtypes are indexed as 1,2,...,n_pt. Inactive pairtypes are given index 0.
%%%%%
%%%%% Input:   x1           coordinates of the moving atoms in configurations C (n1 x 3*N_mov)
%%%%%          x2           coordinates of the moving atoms in configurations C' (n2 x 3*N_mov)
%%%%%          conf_info    structure array including information about the configurations necessary for the GP model
%%%%%          lengthscale  lengthscales for different atom pair types (1 x n_pt)
%%%%%
%%%%% Output:  dist         matrix including the distances between configurations C and C' (n1 x n2)

function dist = dist_at(x1,x2,conf_info,lengthscale)

    conf_fro = conf_info.conf_fro; % coordinates of active frozen atoms (N_fro x 3)
    atomtype_mov = conf_info.atomtype_mov; % atomtype indices for moving atoms (1 x N_mov)
    atomtype_fro = conf_info.atomtype_fro; % atomtype indices for active frozen atoms (1 x N_fro)
    pairtype = conf_info.pairtype; % pairtype indices for pairs of atomtypes (n_at x n_at)
    n_pt = conf_info.n_pt; % number of active pairtypes  
    N_mov = size(atomtype_mov,2);
    N_fro = size(atomtype_fro,2);
    s2 = 1./lengthscale.^2;
    if size(s2)==1
        s2 = repmat(s2,1,n_pt);
    end
    n1 = size(x1,1);
    n2 = size(x2,1);
    dist = zeros(n1,n2);
    % distances between moving atoms
    if N_mov > 1
        for j = 1:N_mov-1
            for i = (j+1):N_mov
                invr_ij_1 = 1./(sqrt(sum((x1(:,(j*3-2):(j*3))-x1(:,(i*3-2):(i*3))).^2,2)));
                invr_ij_2 = 1./(sqrt(sum((x2(:,(j*3-2):(j*3))-x2(:,(i*3-2):(i*3))).^2,2)));
                dist = dist + 2*s2(pairtype(atomtype_mov(i),atomtype_mov(j))).*(bsxfun(@minus,invr_ij_1,invr_ij_2')).^2;
            end
        end
    end
    % distances from moving atoms to active frozen atoms
    if N_fro > 0
        for j = 1:N_mov
            for i = 1:N_fro
                invr_ij_1 = 1./(sqrt(sum((bsxfun(@minus,x1(:,(j*3-2):(j*3)),conf_fro(i,1:3))).^2,2)));
                invr_ij_2 = 1./(sqrt(sum((bsxfun(@minus,x2(:,(j*3-2):(j*3)),conf_fro(i,1:3))).^2,2)));
                dist = dist + 2*s2(pairtype(atomtype_fro(i),atomtype_mov(j))).*(bsxfun(@minus,invr_ij_1,invr_ij_2')).^2;
            end
        end
    end
    dist = sqrt(dist);
    
end
