%%%%% dimer.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 26.1.2020
%%%%%
%%%%% This function uses the dimer method to converge to a saddle point,
%%%%% starting from somewhere inside the convergence area.
%%%%% A rotation step rotates the dimer (a pair of images) towards its minimum energy orientation
%%%%% to find the lowest curvature mode of the potential energy.
%%%%% A translation step moves the dimer towards the saddle point by inverting the
%%%%% force component in the direction of the dimer.
%%%%%
%%%%% Input:
%%%%%   pot_general        accurate potential and gradient function
%%%%%   R_init             coordinates of the middle point of the initial dimer (1 x D)
%%%%%   orient_init        unit vector along the direction of the initial dimer (1 x D)
%%%%%   dimer_sep          dimer separation (distance from the middle point of the dimer to the two images)
%%%%%   method_rot         a function defining the rotation step
%%%%%   method_trans       a function defining the translation step
%%%%%   param_trans        parameters of the translation method (shape depends on 'method_trans')
%%%%%   T_dimer            final convergence threshold (the algorithm is stopped when
%%%%%                        all components of force acting on the middle point of the dimer are less than this)
%%%%%   T_anglerot         convergence threshold for rotation angle (the dimer is
%%%%%                        not rotated when the estimated rotation angle is less than this)
%%%%%   num_iter_rot       maximum number of rotation iterations per translation
%%%%%   num_iter           maximum number of iterations
%%%%%
%%%%% Output:
%%%%%   R                  coordinates of the middle point of the final dimer (1 x D)
%%%%%   orient             unit vector along the direction of the final dimer (1 x D)
%%%%%   E_R                energy at the middle point of the final dimer
%%%%%   G_R                gradient at the middle point of the final dimer (1 x D)
%%%%%   R_all              coordinates of all observation points (N_obs x D)
%%%%%   E_all              energies for all observation points (N_obs x 1)
%%%%%   G_all              gradients for all observation points (N_obs x D)
%%%%%   E_R_acc            energy at the middle point of the dimer for each iteration
%%%%%   maxF_R_acc         maximum component of the force at the middle point of the dimer for each iteration
%%%%%   obs_total          total number of observations

function [R,orient,E_R,G_R,R_all,E_all,G_all,E_R_acc,maxF_R_acc,obs_total] = dimer(pot_general,R_init,orient_init,dimer_sep,method_rot,method_trans,param_trans,T_dimer,T_anglerot,num_iter_rot,num_iter)


    %%%     
    %%% THIS INFORMATION IS ASSUMED TO BE KNOWN BEFORE BEGINNING
    %%%
    
    D = size(R_init,2); % dimension of the space
    if isempty(orient_init) % if 'orient_init' is empty, draw random unit vector
        orient_init = normrnd(zeros(1,D),ones(1,D));
    end
    orient_init = orient_init/sqrt(sum(orient_init.^2,2));
    
    
    %%%
    %%% THE ALGORITHM BEGINS HERE
    %%%
    
    R = R_init; % coordinates of the middle point of the dimer
    orient = orient_init; % unit vector along the direction of the dimer    
    [E_R,G_R] = pot_general(R); % energy and gradient at the middle point of the dimer
    Elevel = E_R; % set zero level of biased potential to the energy of the middle point of the initial dimer
    pot_biased = @(R) subtract_Elevel(pot_general,R,Elevel); % define biased potential with zero level at 'Elevel'
    E_R = E_R - Elevel;
    
    R_all = R; % coordinates of all observation points
    E_all = E_R; % energy for all observation points
    G_all = G_R; % gradient for all observation points
    E_R_acc = []; % vector gathering the energy at the middle point of the dimer for each iteration
    maxF_R_acc = []; % vector gathering the maximum component of the force at the middle point of the dimer for each iteration
    
    rotinfo.F_rot_old = 0; % rotational force of the previous rotation iteration
    rotinfo.F_modrot_old = 0; % modified rotational force of the previous rotation iteration (in conjugated gradients method)
    rotinfo.orient_rot_oldplane = 0; % unit vector perpendicular to 'orient' within the rotation plane of the previous rotation iteration (in conjugated gradients method)
    rotinfo.cgiter_rot = 0; % number of conjugated rotation iterations (in conjugated gradients method)
    rotinfo.num_cgiter_rot = D; % maximum number of conjugated rotation iterations before resetting the conjugate directions (in conjugated gradients method)
    rotinfo.deltaR_mem = []; % change of orientation in m previous rotation iterations (in L-BFGS)
    rotinfo.deltaF_mem = []; % change of rotational force in m previous rotation iterations excluding the last one (in L-BFGS)
    rotinfo.num_lbfgsiter_rot = D; % maximum number of previous rotation iterations kept in memory (in L-BFGS)
    rotinfo.G1 = [];
    transinfo.potential = pot_biased;
    transinfo.F_trans_old = 0; % translational force of the previous translation iteration (in conjugated gradients method)
    transinfo.F_modtrans_old = 0; % modified translational force of the previous translation iteration (in conjugated gradients method)
    transinfo.V_old = 0; % velocity of the middle point of the dimer in the previous translation iteration (in quick-min velocity-Verlet)
    transinfo.zeroV = 1; % indicator if zero velocity used (in quick-min velocity-Verlet)
    transinfo.cgiter_trans = 0; % number of conjugated transition iterations (in conjugated gradients method)
    transinfo.num_cgiter_trans = D; % maximum number of conjugated transition iterations before resetting the conjugate directions (in conjugated gradients method)
    transinfo.deltaR_mem = []; % change of location in m previous translation iterations (in L-BFGS)
    transinfo.deltaF_mem = []; % change of translational force in m previous translation iterations excluding the last one (in L-BFGS)
    transinfo.num_lbfgsiter_trans = D; % maximum number of previous translation iterations kept in memory (in L-BFGS)
    
    % in case of 2D space, plot the energy surface
    if D == 2
        [X1,X2] = meshgrid(-2:0.01:0.5,-0.4:0.01:1.8);
        E_true = zeros(size(X1,1),size(X1,2));
        for j = 1:size(X1,2)
            E_true(:,j) = pot_biased([X1(:,j),X2(:,j)]);
        end
        figure(1)
        pcolor(X1,X2,E_true),shading flat;
        colorbar;
        axis equal tight;
        hold on;
    end
    
    for iter = 0:num_iter
        
        % in case of 2D space, plot the dimer
        if D == 2
            figure(1)
            plot([R(1,1)-dimer_sep*orient(1,1);R(1,1)+dimer_sep*orient(1,1)],[R(1,2)-dimer_sep*orient(1,2);R(1,2)+dimer_sep*orient(1,2)],'y-')
        end
        
        % stop the algorithm if converged
        E_R_acc = [E_R_acc,E_R];
        maxF_R = max(abs(G_R));
        maxF_R_acc = [maxF_R_acc,maxF_R];
        if maxF_R < T_dimer
            fprintf('Stopped the algorithm: converged after %g iterations (%g image evaluations).\n', iter, size(E_all,1));
            break;
        end
        
        % stop the algorithm if maximum number of iterations reached
        if iter == num_iter
            fprintf('Stopped the algorithm: maximum number of iterations (%g) reached.\n', iter);
            break;
        end
        
        % evaluate energy and gradient at image 1
        R1 = R + dimer_sep*orient;
        [E1,G1] = pot_biased(R1);
        R_all = [R_all;R1];
        E_all = [E_all;E1];
        G_all = [G_all;G1];

        % if necessary, rotate the dimer and re-evaluate energy and gradient at image 1
        for iter_rot = 1:num_iter_rot
            orient_old = orient;
            [orient,Curv,R_obs,E_obs,G_obs,rotinfo] = method_rot(R,orient,[G_R;G1],pot_biased,dimer_sep,T_anglerot,1,rotinfo);
            if isempty(R_obs)
                break;
            else
                R_all = [R_all;R_obs];
                E_all = [E_all;E_obs];
                G_all = [G_all;G_obs];
                % in case of 2D space, plot the dimer
                if D == 2
                    figure(1)
                    plot([2*R(1,1)-R_obs(1,1);R_obs(1,1)],[2*R(1,2)-R_obs(1,2);R_obs(1,2)],'y-')
                    plot([R(1,1)-dimer_sep*orient(1,1);R(1,1)+dimer_sep*orient(1,1)],[R(1,2)-dimer_sep*orient(1,2);R(1,2)+dimer_sep*orient(1,2)],'y-')
                end
                if iter_rot == num_iter_rot || acos(orient*orient_old') < T_anglerot
                    break;
                elseif isempty(rotinfo.G1)
                    R1 = R+dimer_sep*orient;
                    [E1,G1] = pot_biased(R1);
                    R_all = [R_all;R1];
                    E_all = [E_all;E1];
                    G_all = [G_all;G1];
                else
                    G1 = rotinfo.G1;
                    rotinfo.G1 = [];
                end
            end
        end
        rotinfo.deltaR_mem = [];
        rotinfo.deltaF_mem = [];
        
        % translate the dimer and re-evaluate energy and gradient at the middle point
        if isempty(Curv)
            Curv = (-G_R(1,:)+G1(1,:))*orient'/dimer_sep;
        end
%        R
%        Curv
%        G_R
%        -G_R - 2*(-G_R*orient')*orient
%        plot(R(1,1),R(1,3),'x')
%        hold on
        [R,R_obs,E_obs,G_obs,transinfo] = method_trans(R,orient,-G_R,Curv,param_trans,transinfo);
        if ~isempty(R_obs)
            R_all = [R_all;R_obs];
            E_all = [E_all;E_obs];
            G_all = [G_all;G_obs];
            % in case of 2D space, plot the test dimer
            if D == 2
                figure(1)
                plot([R_obs(1,1)-dimer_sep*orient(1,1);R_obs(1,1)+dimer_sep*orient(1,1)],[R_obs(1,2)-dimer_sep*orient(1,2);R_obs(1,2)+dimer_sep*orient(1,2)],'y-')
            end
        end
        % in case of 2D space, plot the dimer
        if D == 2
            figure(1)
            plot([R(1,1)-dimer_sep*orient(1,1);R(1,1)+dimer_sep*orient(1,1)],[R(1,2)-dimer_sep*orient(1,2);R(1,2)+dimer_sep*orient(1,2)],'y-')
        end
        [E_R,G_R] = pot_biased(R);
        R_all = [R_all;R];
        E_all = [E_all;E_R];
        G_all = [G_all;G_R];

    end
    
    % in case of 2D space, emphasize the final dimer and plot all observation points
    if D == 2
        figure(1)
        plot([R(1,1)-dimer_sep*orient(1,1);R(1,1)+dimer_sep*orient(1,1)],[R(1,2)-dimer_sep*orient(1,2);R(1,2)+dimer_sep*orient(1,2)],'r-','lineWidth',2)
        plot(R_all(:,1),R_all(:,2),'r+')
    end
    
    obs_total = size(E_all,1);
    
end

