%%%%% demo_CuH2__atomic_GP_NEB_OIE__atomic_GP_dimer.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 25.1.2020
%%%%% 
%%%%% This script shows how to combine 'atomic_GP_NEB_OIE2.m' and 'atomic_GP_dimer' in a CuH2 example.
%%%%% Note that usually it makes no sense to combine GP-NEB with GP-dimer like this, but instead
%%%%% control the OIE evaluation rules by modifying the convergence thresholds of the climbing image
%%%%% (T_CI) and other images (T_MEP).

load('CuH2_idpp10.mat')
conf = conf_0;
R_init_NEB = R_init_0;

% 'pot_general' gives the potential energy and its gradient vector as a function of the atomic configuration.
% Each row of 'R' represents one configuration including the coordinates of the moving atoms:
% [x_1,y_1,z_1,x_2,y_2,z_2,...]
pot_general = @(R) ECuH2(R,conf);
min1 = R_init_NEB(1,:);
min2 = R_init_NEB(end,:);
N_mov = size(min1,2)/3; % number of moving atoms

% 'conf_info' is a structure array including information about the configurations necessary for the GP model:
conf_info = struct;
% 'conf_info.conf_fro': coordinates of active frozen atoms (N_fro x 3)
% In the beginning, list none of the frozen atoms as active:
conf_info.conf_fro = [];
% 'conf_info.atomtype_mov': atomtype indices for moving atoms (1 x N_mov)
conf_info.atomtype_mov = [1,1]; % H,H
% 'conf_info.atomtype_fro': atomtype indices for active frozen atoms (1 x N_fro)
conf_info.atomtype_fro = [];
% The atomtypes must be indexed as 1,2,...,n_at.
n_at = 2; % number of atomtypes (including also the types of inactive frozen atoms)
% 'conf_info.pairtype': pairtype indices for pairs of atomtypes (n_at x n_at)
% Active pairtypes are indexed as 1,2,...,n_pt. Inactive pairtypes are given index 0.
conf_info.pairtype = zeros(n_at);
% 'conf_info.n_pt': number of active pairtypes
conf_info.n_pt = 0;
% Set pairtype indices for moving+moving atom pairs (and update number of active pairtypes):
[conf_info.pairtype,conf_info.n_pt] = set_pairtype_mov(conf_info.atomtype_mov,conf_info.pairtype,conf_info.n_pt);

% 'conf_info_inactive' is a structure array including information about inactive frozen atoms:
conf_info_inactive = struct;
% 'conf_info_inactive.conf_ifro': coordinates of inactive frozen atoms (N_ifro x 3)
% In the beginning, list all frozen atoms as inactive:
conf_info_inactive.conf_ifro = conf(conf(:,4)==1,1:3);
% 'conf_info_inactive.atomtype_ifro': atomtype indices for inactive frozen atoms (1 x N_ifro)
conf_info_inactive.atomtype_ifro = 2*ones(1,size(conf_info_inactive.conf_ifro,1)); % all frozen atoms are Cu

% Only active frozen atoms are taken into account in the covariance function.
% A frozen atom is activated, when it is within the radius of 'actdist_fro'
% from some moving atom in some configuration on the path.
% Once a frozen atom is activated, it stays active from then on.
% If 'actdist_fro' is set to infinity, all frozen atoms are taken into account.
% When a frozen atom is activated, its coordinates and atomtype index are added
% to 'conf_info.conf_fro' and 'conf_info.atomtype_fro', respectively,
% and removed from 'conf_info_inactive.conf_fro' and 'conf_info_inactive.atomtype_fro'.
% If the new frozen atom activates new pairtypes, also 'conf_info.pairtype'
% and 'conf_info.n_pt' are updated.
actdist_fro = 5;
% Activate frozen atoms within activation distance:
[conf_info,conf_info_inactive,~] = update_active_fro(conf_info,conf_info_inactive,[min1;min2],actdist_fro);

N_im = size(R_init,1); % define the number of images on the path
%R_init_NEB = initialize_path_linear(min1,min2,N_im); % define the initial path
method_step = @(R,F_R,param_step,F_R_old,V_old,zeroV) step_QMVelocityVerlet(R,F_R,param_step,F_R_old,V_old,zeroV); % use quick-min Velocity Verlet to define the steps during path relaxation
param_step = 0.1; % define the time step for the quick-min Velocity Verlet algorithm
method_force = @(R,E_R,G_R,param_force,CI_on) force_NEB2(R,E_R,G_R,param_force,CI_on); % use regular NEB force
param_force = 1; % define parallel spring constant
%method_force = @(R,E_R,G_R,param_force,CI_on) force_sNEB2(R,E_R,G_R,param_force,CI_on); % use sNEB force
%param_force = [1,1]; % define parallel and perpendicular spring constants

% 'T_MEP' defines the final convergence threshold for 'maxmaxG_R_perp', which is the accurate maximum component
% of gradient perpendicular to the path tangent at any of the 'N_im'-2 intermediate images
% (i.e., the algorithm is stopped when the maximum component of perpendicular gradient is below 'T_MEP' for all images).
T_MEP = 1;

% 'T_CI' defines an additional final convergence threshold for 'maxG_CI', which is the accurate maximum component
% of gradient at the climbing image, if the climbing image option is used.
T_CI = 0.1;

% 'T_CIon_gp' defines a preliminary GP convergence threshold for turning the climbing image mode on during each relaxation phase:
% When the estimated 'maxmaxF_R', which is the maximum component of NEB force at any of the intermediate images,
% is below 'T_CIon_gp', the climbing image mode is turned on.
% If you don't want to use climbing image at all, set 'T_CIon_gp' to zero.
T_CIon_gp = 1;

% If 'divisor_T_MEP_gp' is set to zero, the default GP convergence threshold for
% the estimated 'maxmaxF_R' is 1/10 of the lowest final threshold.
% To save inner iterations during the first relaxation phases, one can set
% a positive value for 'divisor_T_MEP_gp', so that the GP convergence threshold will be
% 1/'divisor_T_MEP_gp' of the smallest accurate 'maxG_R_perp' obtained so far on any of the
% 'N_im'-2 intermediate images, but not less than 1/10 of the lowest final threshold.
% If the estimation error is assumed to not decrease more than that during one outer iteration,
% there is no need for more accurate relaxation on an estimated surface.
divisor_T_MEP_gp = 0;

% 'disp_max' defines the maximum displacement of image from the nearest observed data point
% relative to the length of the initial path.
% Thus, the last inner step is rejected and the relaxation phase stopped if, for any image, the distance
% to the nearest observed data point is larger than 'disp_max' times the length of the initial path.
disp_max = 0.5;

% 'ratio_at_limit' defines the limit for the ratio (< 1) of inter-atomic distances between image
% and its "nearest" observed data point.
% More precisely, the last inner step is rejected and the relaxation phase stopped if the following
% does not hold for some of the current images:
% There is an observed data point so that all inter-atomic distances of the current image are more
% than 'ratio_at_limit' (by default 2/3) but less than 1/'ratio_at_limit' (3/2) times the corresponding
% inter-atomic distance of the observed data point.
ratio_at_limit = 2/3;

% 'num_bigiter_initpath' defines the number of outer iterations started from the initial path 'R_init'
% - Until 'num_bigiter_initpath' is reached, each relaxation phase is started from the initial path 'R_init'.
%     (If climbing image is used, the CI phase is continued from the "preliminarily converged" evenly spaced path.)
% - After that, each relaxation phase is started from the latest converged path.
%     (If climbing image is used, each relaxation phase is started from the latest "preliminarily converged" evenly spaced path,
%      and the CI phase started from the latest converged CI-path if CI is unchanged
%      (otherwise continued from the current "preliminarily converged" evenly spaced path).)
% Starting each round from the initial path may improve stability (and decrease outer iterations),
% but starting from the latest path may decrease the amount of inner iterations during the relaxation phases.
num_bigiter_initpath = inf;

% 'num_bigiter_initparam' defines the number of outer iterations where the hyperparameter
% optimization is started from values initialized based on the range of current data.
% After that, the optimization is started from the values of the previous round.
num_bigiter_initparam_NEB = inf;

num_bigiter_NEB = 300; % define the maximum number of outer iterations (new sets of observations)
num_iter_NEB = 10000; % define the maximum number of inner iterations (steps during a relaxation phase)

% 'islarge_num_iter' indicates if 'num_iter' is assumed to be much larger than required
% for NEB convergence on accurate energy surface. If not (0), the next relaxation phase is
% continued from the current path in case 'num_iter' is reached.
islarge_num_iter_NEB = 1;

% 'num_bigiter_hess' defines the number of outer iterations using the "virtual Hessian",
% i.e., additional observations around the minimum points. The "virtual Hessian"
% may slow down the GP computations especially in high-dimensional cases,
% but they may give useful information in the beginning.
% They usually don't bring gain after 4 outer iterations (but in some cases do).
% By setting 'num_bigiter_hess' to zero, the "virtual Hessian" is set off.
num_bigiter_hess = inf;
eps_hess = 0.001; % defines the distance of the additional points from the minimum points

load_file = []; % start normally from beginning
save_file = []; % no saves after each outer iteration

quatern = 0; % no quaternion trick used

% Call the atomic GP-NEB algorithm
[R_NEB,E_R_NEB,G_R_NEB,i_CI,gp,R_all_NEB,E_all_NEB,G_all_NEB,Elevel,obs_at_NEB,E_R_NEB_ae,E_R_NEB_gp,maxF_R_NEB_ae,maxF_R_NEB_gp,maxFCI_ae,maxFCI_gp,param_gp_NEB] = ...
             atomic_GP_NEB_OIE2(pot_general,conf_info,conf_info_inactive,actdist_fro,R_init_NEB,method_step,param_step,method_force,param_force,T_MEP,T_CI, ...
             T_CIon_gp,divisor_T_MEP_gp,disp_max,ratio_at_limit,num_bigiter_initpath,num_bigiter_initparam_NEB,num_bigiter_NEB,num_iter_NEB,islarge_num_iter_NEB, ...
             num_bigiter_hess,eps_hess,load_file,save_file,quatern);

% Plot the behaviour
figure
subplot(4,1,1)
title('Mean of maximum components of NEB forces (GP approximation)')
xlabel('iteration')
hold
plot(mean(maxF_R_NEB_gp,1))
plot(obs_at_NEB,mean(maxF_R_NEB_ae,1),'o')
subplot(4,1,2)
title('Maximum of maximum components of NEB force (GP approximation)')
xlabel('iteration')
hold
plot(max(maxF_R_NEB_gp))
plot(obs_at_NEB,max(maxF_R_NEB_ae),'o')
subplot(4,1,3)
title('Maximum component of NEB force on the climbing image (GP approximation)')
xlabel('iteration')
hold
plot(maxFCI_gp)
plot(obs_at_NEB,maxFCI_ae,'o')
subplot(4,1,4)
title('Mean of energies (GP approximation)')
xlabel('iteration')
hold
plot(mean(E_R_NEB_gp,1))  
plot(obs_at_NEB,mean(E_R_NEB_ae,1),'o')

R_all_init = R_all_NEB; % define initial data points (got from GP-NEB)
E_all_init = E_all_NEB+Elevel; % define energy at the initial data points (got from GP-NEB)
G_all_init = G_all_NEB; % define gradient at the initial data points (got from GP-NEB)
R_init = R_NEB(1+i_CI,:); % define the initial middle point of the dimer as the climbing image of the final path (got from GP-NEB)
E_init = E_R_NEB(1+i_CI,1)+Elevel; % define the energy at the initial middle point of the dimer (got from GP-NEB)
G_init = G_R_NEB(1+i_CI,:); % define the gradient at the initial middle point of the dimer (got from GP-NEB)
Z = tangent(R_NEB,E_R_NEB);
orient_init = Z(i_CI,:); % define the initial orientation of the dimer as the tangent of the final path at the climbing image (got from GP-NEB)
D = size(R_init,2); % number of dimensions
N_mov = size(R_init,2)/3; % number of moving atoms

% 'conf_info' is a structure array including information about the configurations necessary for the GP model:
conf_info = struct;
% 'conf_info.conf_fro': coordinates of active frozen atoms (N_fro x 3)
% In the beginning, list none of the frozen atoms as active:
conf_info.conf_fro = [];
% 'conf_info.atomtype_mov': atomtype indices for moving atoms (1 x N_mov)
conf_info.atomtype_mov = [1,1]; % H,H
% 'conf_info.atomtype_fro': atomtype indices for active frozen atoms (1 x N_fro)
conf_info.atomtype_fro = [];
% The atomtypes must be indexed as 1,2,...,n_at.
n_at = 2; % number of atomtypes (including also the types of inactive frozen atoms)
% 'conf_info.pairtype': pairtype indices for pairs of atomtypes (n_at x n_at)
% Active pairtypes are indexed as 1,2,...,n_pt. Inactive pairtypes are given index 0.
conf_info.pairtype = zeros(n_at);
% 'conf_info.n_pt': number of active pairtypes
conf_info.n_pt = 0;
% Set pairtype indices for moving+moving atom pairs (and update number of active pairtypes):
[conf_info.pairtype,conf_info.n_pt] = set_pairtype_mov(conf_info.atomtype_mov,conf_info.pairtype,conf_info.n_pt);

% 'conf_info_inactive' is a structure array including information about inactive frozen atoms:
conf_info_inactive = struct;
% 'conf_info_inactive.conf_ifro': coordinates of inactive frozen atoms (N_ifro x 3)
% In the beginning, list all frozen atoms as inactive:
conf_info_inactive.conf_ifro = conf(conf(:,4)==1,1:3);
% 'conf_info_inactive.atomtype_ifro': atomtype indices for inactive frozen atoms (1 x N_ifro)
conf_info_inactive.atomtype_ifro = 2*ones(1,size(conf_info_inactive.conf_ifro,1)); % all frozen atoms are Cu

% Only active frozen atoms are taken into account in the covariance function.
% A frozen atom is activated, when it is within the radius of 'actdist_fro'
% from some moving atom in some configuration on the path.
% Once a frozen atom is activated, it stays active from then on.
% If 'actdist_fro' is set to infinity, all frozen atoms are taken into account.
% When a frozen atom is activated, its coordinates and atomtype index are added
% to 'conf_info.conf_fro' and 'conf_info.atomtype_fro', respectively,
% and removed from 'conf_info_inactive.conf_fro' and 'conf_info_inactive.atomtype_fro'.
% If the new frozen atom activates new pairtypes, also 'conf_info.pairtype'
% and 'conf_info.n_pt' are updated.
actdist_fro = 5;
% Activate frozen atoms within activation distance:
[conf_info,conf_info_inactive,~] = update_active_fro(conf_info,conf_info_inactive,[R_all_init;R_init],actdist_fro);

dimer_sep = 0.01; % define the dimer separation (distance from the middle point of the dimer to the two images)
method_rot = @rot_iter_lbfgs; % use the conjugate gradient method for rotations
method_trans = @trans_iter_lbfgs; % use the conjugate gradient method for translations
param_trans = [0.1, 0.1]; % define a step length for convex regions and maximum step length

% 'eval_image1' indicates if image 1 of the dimer is evaluted (1) or not (0)
% after each relaxation phase in addition to the middle point of the dimer
eval_image1 = 0;

% 'T_dimer' defines the final convergence threshold for 'maxF_R', which is
% the maximum component of the force acting on the middle point of the dimer (i.e., the
% algorithm is stopped when all components of the accurate force are below 'T_dimer').
T_dimer = 0.01;

% 'initrot_nogp' indicates if the initial rotations are performed without GP (1) or with GP (0).
initrot_nogp = 0;

% 'T_anglerot_init' defines a convergence threshold for the rotation angle
% in the initial rotations performed in the beginning of the algorithm
% (the dimer is not rotated when the estimated rotation angle is less than this).
T_anglerot_init = 0.0873;

% 'num_iter_initrot' defines the maximum number of initial rotations (0 if initial rotations skipped).
num_iter_initrot = D;

% 'inittrans_nogp' is an indicator if an initial test translation step is taken without GP (1)
% or if GP is used right after initial rotations (0).
inittrans_nogp = 0;

% 'T_anglerot_gp' defines a convergence threshold for the rotation angle
% during a relaxation phase (the dimer is not rotated when the estimated
% rotation angle is less than this).
T_anglerot_gp = 0.01;

% 'num_iter_rot_gp' defines a maximum number of rotation iterations per
% translation during a relaxation phase.
num_iter_rot_gp = 1;

% If 'divisor_T_dimer_gp' is set to zero, the default convergence threshold
% for each relaxation phase for the approximated 'maxF_R' on the
% approximated energy surface is 1/10 of the 'T_dimer'. To save inner
% iterations during the first relaxation phases, one can set a positive
% value for 'divisor_T_dimer_gp', so that the GP convergence threshold will
% be 1/'divisor_T_dimer_gp' of the smallest accurate 'maxF_R' obtained so
% far, but not less than 1/10 of the 'T_dimer'. If the approximation error
% is assumed to not decrease more than that during one outer iteration,
% there is no need for more accurate relaxation on an approximated surface.
divisor_T_dimer_gp = 10;

% 'disp_max' defines the maximum displacement of the middle point of the
% dimer from the nearest observed data point. Thus, the last inner step is
% rejected and the relaxation phase stopped, if the distance to the nearest
% observed data point is larger than 'disp_max'.
%disp_max = 0.5;

% 'ratio_at_limit' defines the limit for the ratio (< 1) of inter-atomic
% distances between image and its "nearest" observed data point.
% More precisely, the last inner step is rejected and the relaxation phase
% stopped if the following does not hold:
% There is an observed data point so that all inter-atomic distances of the
% current image are more than 'ratio_at_limit' (by default 2/3) but less
% than 1/'ratio_at_limit' (3/2) times the corresponding inter-atomic
% distance of the observed data point.
%ratio_at_limit = 2/3;

% 'num_bigiter_initloc' defines the number of outer iterations started from the initial location 'R_init'.
% After that, each relaxation phase is started from the latest converged dimer.
% Starting each round from the initial location may improve stability (and decrease outer iterations),
% but starting from the latest dimer may decrease the number of inner iterations during the relaxation phases.
num_bigiter_initloc = inf;

% 'num_bigiter_initparam' defines the number of outer iterations where the hyperparameter
% optimization is started from values initialized based on the range of current data.
% After that, the optimization is started from the values of the previous round.
num_bigiter_initparam = inf;

num_bigiter = 300; % define the maximum number of outer iterations (new sets of observations)
num_iter = 10000; % define the maximum number of inner iterations (steps during a relaxation phase)

% 'islarge_num_iter' indicates if 'num_iter' is assumed to be much larger than required
% for dimer convergence on accurate energy surface. If not (0), the next relaxation phase is
% continued from the current dimer in case 'num_iter' is reached.
islarge_num_iter = 1;

load_file = []; % start normally from beginning
save_file = []; % no saves after each outer iteration

% Call the atomic GP-dimer algorithm
[R,orient,E_R,G_R,gp,R_all,E_all,G_all,obs_at,E_R_acc,E_R_gp,maxF_R_acc,maxF_R_gp,param_gp_initrot,param_gp,obs_initrot,obs_total,num_esmax,num_es1,num_es2] = ...
             atomic_GP_dimer(pot_general,conf_info,conf_info_inactive,actdist_fro,R_all_init,E_all_init,G_all_init,R_init,E_init,G_init,orient_init,dimer_sep,method_rot,method_trans, ...
             param_trans,eval_image1,T_dimer,initrot_nogp,T_anglerot_init,num_iter_initrot,inittrans_nogp,T_anglerot_gp,num_iter_rot_gp,divisor_T_dimer_gp, ...
             disp_max,ratio_at_limit,num_bigiter_initloc,num_bigiter_initparam,num_bigiter,num_iter,islarge_num_iter,load_file,save_file);

% Plot the behaviour
figure
subplot(2,1,1)
title('Maximum component of force on the middle point (GP approximation)')
xlabel('iteration')
hold
plot(maxF_R_gp)
plot(obs_at,maxF_R_acc,'o')
subplot(2,1,2)
title('Energy at the middle point (GP approximation)')
xlabel('iteration')
hold
plot(E_R_gp)  
plot(obs_at,E_R_acc,'o')
