%%%%% demo_CuH2_GP_sNEB_AIE.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 18.9.2017
%%%%%
%%%%% This script shows how to use 'GP_sNEB_AIE.m' in a CuH2 example.

%load('CuH2_idpp8_villi.mat')
% define the potential energy function
pot_general = @(R) ECuH2(R,conf);

min1 = [8.6822,9.9470,11.7330,7.9422,9.9470,11.7330]; % define the first minimum point
min2 = [9.8914,9.9470,7.7599,6.7330,9.9470,7.7599]; % define the second minimum point
D = size(min1,2);
N_im = 10; % define the number of images on the path
R_init = initialize_path_linear(min1,min2,N_im); % define the initial path
method_step = @(R,F_R,param_step,F_R_old,V_old,zeroV) step_QMVelocityVerlet(R,F_R,param_step,F_R_old,V_old,zeroV); % use quick-min Velocity Verlet to define the steps path relaxation
param_step = 0.1; % define the time step for the quick-min Velocity Verlet algorithm
k_par = 1; % define the parallel spring constant
k_perp = 1; % define the perpendicular spring constant

% 'T_MEP' defines the final convergence threshold for the 'maxnormF', which is
% the maximum of the accurate norms of the NEB forces acting on the 'N_im'-2 intermediate
% images (i.e., the algorithm is stopped when the accurate NEB force is below 'T_MEP' for all images).
T_MEP = 0.01;

% 'T_CI' defines an additional final convergence threshold for the
% climbing image, if the climbing image option is used.
% If you don't want to use a tighter convergence threshold for the climbing
% image, set 'T_CI' equal to 'T_MEP' (or larger, because
% the general threshold 'T_MEP' concerns also the climbing image).
T_CI = 0.01;

% 'T_CIon_gp' defines a preliminary convergence threshold for each relaxation phase:
% When the approximated 'maxnormF' is below 'T_CIon_gp', the climbing
% image mode is turned on.
% If you don't want to use climbing image at all, set 'T_CIon_gp' to zero.
T_CIon_gp = 0.01;

% If 'divisor_T_MEP_gp' is set to zero, the default convergence threshold for each relaxation
% phase for the approximated 'maxnormF' on the approximated energy surface is 1/10 of the lowest final threshold.
% To save inner iterations during the first relaxation phases, one can set
% a positive value for 'divisor_T_MEP_gp', so that the GP convergence threshold will be
% 1/'divisor_T_MEP_gp' of the smallest accurate norm of NEB force obtained so far
% on any of the 'N_im'-2 intermediate images, but not less than 1/10 of the lowest final threshold.
% If the approximation error is assumed to not decrease more than that during one outer iteration,
% there is no need for more accurate relaxation on an approximated surface.
divisor_T_MEP_gp = 10;

% 'disp_max' defines the maximum displacement of image from the nearest observed data point
% relative to the length of the initial path.
% Thus, the last inner step is rejected and the relaxation phase stopped if, for any image, the distance
% to the nearest observed data point is larger than 'disp_max' times the length of the initial path.
disp_max = 0.5;

% 'num_bigiter_init' defines the number of outer iterations started from the initial path 'R_init'
% - Until 'num_bigiter_init' is reached, each relaxation phase is started from the initial path 'R_init'.
%     (If climbing image is used, the CI phase is continued from the "preliminarily converged" evenly spaced path.)
% - After that, each relaxation phase is started from the latest converged path.
%     (If climbing image is used, each relaxation phase is started from the latest "preliminarily converged" evenly spaced path,
%      and the CI phase started from the latest converged CI-path if CI is unchanged
%      (otherwise continued from the current "preliminarily converged" evenly spaced path).)
% Starting each round from the initial path may improve stability (and decrease outer iterations),
% but starting from the latest path may decrease the amount of inner iterations during the relaxation phases.
num_bigiter_init = inf;

num_bigiter = 300; % define the maximum number of outer iterations (new sets of observations)
num_iter = 10000; % define the maximum number of inner iterations (steps during a relaxation phase)

% 'num_bigiter_hess' defines the number of outer iterations using the "virtual Hessian",
% i.e., additional observations around the minimum points. The "virtual Hessian"
% may slow down the GP computations especially in high-dimensional cases,
% but they may give useful information in the beginning.
% They usually don't bring gain after 4 outer iterations (but in some cases do).
% By setting 'num_bigiter_hess' to zero, the "virtual Hessian" is set off.
num_bigiter_hess = 0;
eps_hess = 0.001; % defines the distance of the additional points from the minimum points

% Define here the GP model by choosing the covariance function family and the
% observation model (likelihood). The hyperparameters of the GP model will
% be optimized before each relaxation phase, unless 'prior_fixed' is set
% (like here for the observation noise 'sigma2'). If the prior is left out,
% an uninformative default prior is used.

% 'gpcf_constant' defines a constant covariance function, which has a
% similar effect as integration over an unknown constant mean function
% having a Gaussian prior distribution with variance 'constSigma2'.
% Thus, adding this to the covariance function gives the mean level of the GP
% more space to vary.
cfc = gpcf_constant('constSigma2',100,'constSigma2_prior',prior_fixed);

% 'gpcf_sexp' defines a squared exponential covariance function which favours
% smooth functions. The hyperparameter 'magnSigma2' controls the magnitude
% of the overall variation of the function, and 'lengthScale' defines how fast the function
% can chance. 'lengthScale' can be the same for all dimensions (when the
% initial value is scalar) or a vector including an independent value for
% each dimension. In this 2D example, independent values are reasonable,
% but in high-dimensional cases a shared 'lengthScale' may be better identifiable.
%cfse = gpcf_sexp('lengthScale',ones(size(min1)));
cfse = gpcf_sexp('lengthScale',1);

% 'lik_gaussian' defines a Gaussian noise model. Here, we assume that the observations
% are accurate, so 'sigma2' should be set small. However, if it is set too
% small, numerical issues may arise.
lik = lik_gaussian('sigma2', 1e-8, 'sigma2_prior', prior_fixed);

% 'gp_set' defines the GP model. Here, the covariance function 'cf' is
% defined as the sum of 'cfc' and 'cfse' defined above. To be able to use
% derivative observations, 'derivobs' needs to be set to 'on'.
gp = gp_set('cf', {cfc, cfse}, 'lik', lik, 'deriv', D+1, 'jitterSigma2', 0);

% 'optimset' defines the tolerances used in optimization (of the GP hyperparameters).
opt = optimset('TolFun',1e-4,'TolX',1e-4,'display','off');

% Define the optimization function for the GP hyperparameter optimization.
% '@fminscg' may be more stable, but slower than '@fminlbfgs'.
optimf = @fminscg;

% Call the GP-sNEB algorithm
[R,E_R,G_R,i_CI,gp,R_all,E_all,G_all,obs_at,E_R_acc,E_R_gp,normF_R_acc,normF_R_gp,normFCI_acc,normFCI_gp,param_gp] = ...
             GP_sNEB_AIE(pot_general,R_init,method_step,param_step,k_par,k_perp,T_MEP,T_CI, ...
             T_CIon_gp,divisor_T_MEP_gp,disp_max,num_bigiter_init,num_bigiter,num_iter,num_bigiter_hess,eps_hess,gp,opt,optimf);

% Plot the behaviour
figure
subplot(4,1,1)
title('Mean of norms of NEB forces (GP approximation)')
xlabel('iteration')
hold
plot(mean(normF_R_gp,1))
plot(obs_at,mean(normF_R_acc,1),'o')
subplot(4,1,2)
title('Maximum norm of NEB force (GP approximation)')
xlabel('iteration')
hold
plot(max(normF_R_gp))
plot(obs_at,max(normF_R_acc),'o')
subplot(4,1,3)
title('Norm of NEB force on the climbing image (GP approximation)')
xlabel('iteration')
hold
plot(normFCI_gp)
plot(obs_at,normFCI_acc,'o')
subplot(4,1,4)
title('Mean of energies (GP approximation)')
xlabel('iteration')
hold
plot(mean(E_R_gp,1))  
plot(obs_at,mean(E_R_acc,1),'o')

