%%%%% pot_magnetic.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 10.10.2018
%%%%%
%%%%% This function gives the potential energy and its gradient of a
%%%%% magnetic system in N_im locations.
%%%%% 
%%%%% Input:   R    matrix (N_im x D) including the coordinates of N_im locations
%%%%% Output:  E_R  vector (N_im x 1) including the potential in the N_im locations
%%%%%          G_R  matrix (N_im x D) including the gradient for the N_im locations

function [E_R,G_R] = pot_magnetic(R)

    N_im = size(R,1);
    E_R = zeros(N_im,1);
    G_R = zeros(N_im,2);

    c1 = -0.2;
    c2 = -0.1;
    c3 = -0.17;
    c4 = 0.015;
    F1 = [-2, 0; 0, -20];
    F2 = [-2, 0; 0, -20];
    F3 = [-13, 11; 11, -13];
    F4 = [7/5, 3/5; 3/5, 7/5];
    q1 = [1, 0]';
    q2 = [0, 1/2]';
    q3 = [-1/2, 3/2]';
    q4 = [-1, 1]';
    
    for i = 1:N_im
        
        E1 = c1*exp(0.5*(R(i,:)-q1')*F1*(R(i,:)'-q1));
        E2 = c2*exp(0.5*(R(i,:)-q2')*F2*(R(i,:)'-q2));
        E3 = c3*exp(0.5*(R(i,:)-q3')*F3*(R(i,:)'-q3));
        E4 = c4*exp(0.5*(R(i,:)-q4')*F4*(R(i,:)'-q4));
        G1_1 = E1*[F1(1,1),(F1(1,2)+F1(2,1))/2]*(R(i,:)'-q1);
        G1_2 = E1*[(F1(1,2)+F1(2,1))/2,F1(2,2)]*(R(i,:)'-q1);
        G2_1 = E2*[F2(1,1),(F2(1,2)+F2(2,1))/2]*(R(i,:)'-q2);
        G2_2 = E2*[(F2(1,2)+F2(2,1))/2,F2(2,2)]*(R(i,:)'-q2);
        G3_1 = E3*[F3(1,1),(F3(1,2)+F3(2,1))/2]*(R(i,:)'-q3);
        G3_2 = E3*[(F3(1,2)+F3(2,1))/2,F3(2,2)]*(R(i,:)'-q3);
        G4_1 = E4*[F4(1,1),(F4(1,2)+F4(2,1))/2]*(R(i,:)'-q4);
        G4_2 = E4*[(F4(1,2)+F4(2,1))/2,F4(2,2)]*(R(i,:)'-q4);
        
        E_R(i,1) = E1 + E2 + E3 + E4;
        G_R(i,1) = G1_1 + G2_1 + G3_1 + G4_1;
        G_R(i,2) = G1_2 + G2_2 + G3_2 + G4_2;
           
    end
    
end