%%%%% atomic_GP_NEB_AIE2.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 10.2.2020
%%%%%
%%%%% This is the all-images-evaluated (AIE) version of the
%%%%% atomic GP-NEB algorithm for finding a minimum energy path and a saddle
%%%%% point between two minimum energy configurations.
%%%%% The relaxation of the path on the estimated energy surface
%%%%% is done according to the nudged elastic band (NEB) method with
%%%%% a climbing image option for the highest-energy image.
%%%%% After each relaxation phase, the energy and gradient are acquired
%%%%% for all the intermediate images, and the GP hyperparameters are
%%%%% reoptimized.
%%%%%
%%%%% The atomic version of GP-NEB uses a special GPstuff covariance function 'gpcf_sexpat.m'
%%%%% (not included in the GPstuff installation) where the distance between configurations C and C'
%%%%% is based on the changes of the inter-atomic distances. The distance function is implemented
%%%%% in 'dist_at.m', which is needed when defining the prior distributions for the lengthscales
%%%%% during the algorithm. When defining the stopping criteria for the path relaxation phases,
%%%%% also another distance function 'dist_max1Dlog.m' is needed. One more auxiliary function required
%%%%% is 'mindist_interatomic.m', which gives the minimum inter-atomic distance in a configuration.
%%%%% To update the set of active frozen atoms, also an auxiliary function 'update_active_fro.m' is needed.
%%%%%
%%%%% Using the function requires that the develop version of GPstuff is installed and added to the path in Matlab.
%%%%%
%%%%% Input:
%%%%%   pot_general            accurate potential and gradient function
%%%%%   conf_info              structure array including information about the configurations necessary for the GP model
%%%%%                          - conf_info.conf_fro: coordinates of active frozen atoms (N_fro x 3)
%%%%%                          - conf_info.atomtype_mov: atomtype indices for moving atoms (1 x N_mov)
%%%%%                          - conf_info.atomtype_fro: pairtype indices for active frozen atoms (1 x N_fro)
%%%%%                          - conf_info.pairtype: pairtype indices for pairs of atomtypes (n_at x n_at)
%%%%%                          - conf_info.n_pt: number of active pairtypes
%%%%%   conf_info_inactive     structure array including information about inactive frozen atoms
%%%%%                          - conf_info_inactive.conf_ifro: coordinates of inactive frozen atoms (N_ifro x 3)
%%%%%                          - conf_info_inactive.atomtype_ifro: atomtype indices for inactive frozen atoms (1 x N_ifro)
%%%%%   actdist_fro            activation distance for moving+frozen atom pairs (inf if all active)
%%%%%   R_init                 coordinates for the images on the initial path (N_im x D)
%%%%%   method_step            a function defining the following step during path relaxation
%%%%%   param_step             parameters of the path relaxation method (shape depends on 'method_step')
%%%%%   method_force           a function defining the NEB force
%%%%%   param_force            parameters of the NEB force method (shape depends on 'method_force')
%%%%%   T_MEP                  final convergence threshold for the accurate 'maxmaxG_R_perp', which is the maximum component
%%%%%                            of gradient perpendicular to the path tangent at any of the intermediate images
%%%%%                            (the algorithm is stopped when maximum component of perpendicular gradient is less than 'T_MEP' for all images)
%%%%%   T_CI                   additional final convergence threshold for the accurate 'maxG_CI', which is the maximum component
%%%%%                            of gradient at the climbing image
%%%%%   T_CIon_gp              premilinary GP convergence threshold for the estimated 'maxmaxF_R', which is the maximum component
%%%%%                            of NEB force at any of the intermediate images
%%%%%                            (the climbing image is turned on when 'T_CIon_gp' is reached during relaxation phase)
%%%%%                            (use 0 if CI not used at all)
%%%%%   divisor_T_MEP_gp       if this option is set on (> 0), the GP convergence threshold for the estimated 'maxmaxF_R'
%%%%%                            is 1/'divisor_T_MEP_gp' of the smallest accurate 'maxG_R_perp' obtained so far
%%%%%                            for any of the intermediate images, but not less than 1/10 of the lowest final threshold
%%%%%                            (otherwise the GP convergence threshold is always 1/10 of the lowest final threshold)
%%%%%   disp_max               maximum displacement of image from the nearest observed data point
%%%%%                            relative to the length of the initial path
%%%%%                            (the relaxation phase is stopped if 'disp_max' is reached for any image)
%%%%%   ratio_at_limit         limit for the ratio (< 1) of inter-atomic distances between image and its "nearest" observed data point
%%%%%                            (the relaxation phase is stopped if 'ratio_at_limit' is reached for any image)
%%%%%   num_bigiter_initpath   number of outer iterations started from the initial path 'R_init'
%%%%%                          - Until 'num_bigiter_initpath' is reached, each relaxation phase is started from the initial path 'R_init'
%%%%%                              (if climbing image is used, the CI phase is continued from the "preliminarily converged" evenly spaced path)
%%%%%                          - After that, each relaxation phase is started from the latest converged path
%%%%%                              (if climbing image is used, each relaxation phase is started
%%%%%                               from the latest "preliminarily converged" evenly spaced path,
%%%%%                               and the CI phase is started from the latest converged CI-path if CI is unchanged
%%%%%                               (otherwise continued from the current "preliminarily converged" evenly spaced path))
%%%%%   num_bigiter_initparam  number of outer iterations where the hyperparameter optimization is started
%%%%%                            from values initialized based on the range of the current data
%%%%%                            (after that, the optimization is started from the values of the previous round)
%%%%%   num_bigiter            maximum number of outer iterations (new sets of evaluations)
%%%%%   num_iter               maximum number of inner iterations (steps during a relaxation phase)
%%%%%   islarge_num_iter       indicator if 'num_iter' is assumed to be much larger than required for NEB convergence on accurate energy surface
%%%%%                            (if not, the next relaxation phase is continued from the current path if 'num_iter' is reached)
%%%%%   num_bigiter_hess       number of outer iterations using the "virtual Hessian" around the minimum points
%%%%%   eps_hess               epsilon for the "virtual Hessian"
%%%%%   load_file              path to the data file required to continue from a cancelled run (empty if started normally from the beginning)
%%%%%   save_file              path to the data file where data is saved (empty if not saved)
%%%%%   quatern                indicator if quaternion trick used to remove rotation/translation of system
%%%%%
%%%%% Output:
%%%%%   R                      coordinates for the images on the final path (N_im x D)
%%%%%   E_R                    energy at the images on the final path (N_im x 1)
%%%%%   G_R                    gradient at the images on the final path (N_im x D)
%%%%%   i_CI                   index of the climbing image among the intermediate images of the final path
%%%%%   gp                     the final GP model
%%%%%   R_all                  coordinates for all image observations
%%%%%   E_all                  energy for all image observations
%%%%%   G_all                  gradient for all image observations
%%%%%   Elevel                 level of zero energy in terms of the input potential
%%%%%   obs_at                 total numbers of inner iterations before new observations were taken
%%%%%   E_R_acc                accurate energies of the images for each outer iteration
%%%%%   E_R_gp                 estimated energies of the images for each inner iteration
%%%%%   maxG_R_perp_acc        accurate maximum component of gradient perpendicular to the path tangent at each image after each evaluation
%%%%%   maxF_R_gp              estimated maximum component of NEB force acting on each intermediate image for each inner iteration
%%%%%   maxG_CI_acc            accurate maximum component of gradient at the climbing image after each evaluation (0 if CI is off)
%%%%%   maxG_CI_gp             estimated maximum component of gradient at the climbing image for each inner iteration (0 if CI is off)
%%%%%   param_gp               optimized GP hyperparameters for each outer iteration (in logarithmic scale)

function [R,E_R,G_R,i_CI,gp,R_all,E_all,G_all,Elevel,obs_at,E_R_acc,E_R_gp,maxG_R_perp_acc,maxF_R_gp,maxG_CI_acc,maxG_CI_gp,param_gp] = ...
             atomic_GP_NEB_AIE2(pot_general,conf_info,conf_info_inactive,actdist_fro,R_init,method_step,param_step,method_force,param_force,T_MEP,T_CI, ...
             T_CIon_gp,divisor_T_MEP_gp,disp_max,ratio_at_limit,num_bigiter_initpath,num_bigiter_initparam,num_bigiter,num_iter,islarge_num_iter, ...
             num_bigiter_hess,eps_hess,load_file,save_file,quatern)


    if isempty(load_file)    
        
        %%%     
        %%% THIS INFORMATION IS ASSUMED TO BE KNOWN BEFORE BEGINNING
        %%%
        
        N_im = size(R_init,1); % number of images on the path
        D = size(R_init,2); % dimension of the space
        min1 = R_init(1,:); % minimum point 1
        min2 = R_init(end,:); % minimum point 2
        scale = 0;
        for im = 1:N_im-1
            scale = scale + sqrt(sum((R_init(im+1,:)-R_init(im,:)).^2)); % length of the initial path
        end
        [E_min1,G_min1] = pot_general(min1); % energy and gradient at minimum point 1
        [E_min2,G_min2] = pot_general(min2); % energy and gradient at minimum point 2
        % Elevel = min([E_min1,E_min2]); % zero level of energy is set to the lower minimum
        Elevel = E_min1; % zero level of energy is set to minimum point 1
        E_min1 = E_min1 - Elevel;
        E_min2 = E_min2 - Elevel;
        if num_bigiter_hess > 0
            R_h = get_hessian_points(R_init,eps_hess); % define the "virtual Hessian" points if used
            [E_h,G_h] = pot_general(R_h); % energy and gradient at the "virtual Hessian" points
            E_h = E_h - Elevel;
        else
            R_h = [];
            E_h = [];
            G_h = [];
        end
        R_all = [R_h;min1;min2]; % coordinates of all observation points
        E_all = [E_h;E_min1;E_min2]; % energy for all observation points
        G_all = [G_h;G_min1;G_min2]; % gradient for all observation points
        
        % initialize the GP model
        cfc = gpcf_constant('constSigma2_prior',prior_fixed);
        [conf_info,conf_info_inactive,~] = update_active_fro(conf_info,conf_info_inactive,R_all,actdist_fro);
        fprintf('%g active and %g inactive frozen atoms in the beginning.\n', size(conf_info.conf_fro,1), size(conf_info_inactive.conf_ifro,1));
        cfat = gpcf_sexpat('magnSigma2_prior',prior_sqrtt('nu',20),'lengthScale_prior',prior_gaussian(),'conf_info',conf_info);
        lik = lik_gaussian('sigma2', 1e-8, 'sigma2_prior', prior_fixed);
        gp = gp_set('cf', {cfc,cfat}, 'lik', lik, 'deriv', D+1, 'jitterSigma2', 0);
        opt = optimset('TolFun',1e-4,'TolX',1e-4,'display','off');
        optimf = @fminscg;
        
        
        %%%
        %%% THE ALGORITHM BEGINS HERE
        %%%
        
        R = R_init; % coordinates of the images
        R_latest_equal = []; % latest converged evenly spaced path (no climbing image)
        R_previous_equal = []; % previous evenly spaced path (no climbing image) if maximum number of inner iterations reached
        if T_CIon_gp > 0
            R_latest_climb = []; % latest converged CI-path
            i_CI_latest = 0; % climbing image index among the intermediate images for the latest converged CI-path
            R_previous_climb = []; % previous CI-path if maximum number of inner iterations reached
            i_CI_previous = 0; % climbing image index among the intermediate images for the previous CI-path
        end 
        E_R_acc = []; % matrix gathering accurate energies of the images for each outer iteration
        E_R_gp = []; % matrix gathering estimated energies of the images for each inner iteration
        maxG_R_perp_acc = []; % matrix gathering accurate maximum component of gradient perpendicular to the path tangent at each intermediate image for each outer iteration
        maxG_CI_acc = []; % vector gathering accurate maximum component of gradient at the climbing image for each outer iteration (0 if CI is off)
        maxF_R_gp = []; % matrix gathering estimated maximum component of the NEB force acting on each intermediate image for each inner iteration
        maxG_CI_gp = []; % vector gathering estimated maximum component of gradient at the climbing image for each inner iteration (0 if CI is off)
        obs_at = []; % vector gathering the total numbers of inner iterations before new observations were taken    
        param_gp = []; % optimized GP hyperparameters for each relaxation phase (in logarithmic scale)
        bigiter_init = 0;
 
    else

        load(load_file);
        bigiter_init = bigiter + 1;

    end
       
    for bigiter = bigiter_init:num_bigiter % OUTER ITERATION LOOP
        
        % acquire the accurate energy and gradient on the relaxed path and add them to the data
        E_R = [E_min1;zeros(N_im-2,1);E_min2]; % energy at the images
        G_R = [G_min1;zeros(N_im-2,D);G_min2]; % gradient at the images
        R_all = [R_all;R(2:(N_im-1),:)];
        [E_R(2:(N_im-1),:),G_R(2:(N_im-1),:)] = pot_general(R(2:(N_im-1),:));
        E_R(2:(N_im-1),:) = E_R(2:(N_im-1),:) - Elevel;
        [F_R,maxG_R_perp,maxG_CI,i_CI] = method_force(R,E_R,G_R,param_force,T_CIon_gp);
        E_all = [E_all;E_R(2:(N_im-1),:)];
        G_all = [G_all;G_R(2:(N_im-1),:)];
        E_R_acc = [E_R_acc,E_R];
        maxG_R_perp_acc = [maxG_R_perp_acc,maxG_R_perp];
        maxG_CI_acc = [maxG_CI_acc,maxG_CI];
        obs_at = [obs_at; size(E_R_gp,2)];
        if T_CIon_gp > 0
            fprintf('Accurate values: maxE_R = %1.3g, maxmaxG_R_perp = %1.3g, minmaxG_R_perp = %1.3g, maxG_CI = %1.3g (image %g) \n\n',max(E_R_acc(:,end)),max(maxG_R_perp_acc(:,end)),min(maxG_R_perp_acc(:,end)),maxG_CI_acc(end),i_CI+1);
        else
            fprintf('Accurate values: maxE_R = %1.3g, maxmaxG_R_perp = %1.3g, minmaxG_R_perp = %1.3g \n\n',max(E_R_acc(:,end)),max(maxG_R_perp_acc(:,end)),min(maxG_R_perp_acc(:,end))); 
        end
        
        % stop the algorithm if final convergence is obtained
        if max(maxG_R_perp_acc(:,end)) < T_MEP && maxG_CI_acc(end) < T_CI
            fprintf('Final convergence obtained after %g relaxation phases (%g image evaluations).\n', bigiter, (N_im-2)*(bigiter+1));
            break;
        end
        
        % stop the algorithm if maximum number of outer iterations is reached
        if bigiter == num_bigiter
            fprintf('Stopped the algorithm: Maximum number of outer iterations (%g) reached.\n', bigiter);
            break;
        end
        
        % remove the "virtual Hessian" observations if needed
        if num_bigiter_hess > 0 && bigiter == num_bigiter_hess
            R_all(1:2*D,:) = [];
            E_all(1:2*D,:) = [];
            G_all(1:2*D,:) = [];
        end
        
        % visualize the true energy along the spline interpolation of the path
        %figure(10)
        %hold on
        %visualize_path_true
        %hold off
        
        % optimize the GP hyperparameters and calculate some variables unchanged during the relaxation
        if actdist_fro < inf
            [conf_info,conf_info_inactive,new_act] = update_active_fro(conf_info,conf_info_inactive,R(2:(N_im-1),:),actdist_fro);
            if new_act > 0
                fprintf('More frozen atoms activated. Now %g active and %g inactive frozen atoms.\n', size(conf_info.conf_fro,1), size(conf_info_inactive.conf_ifro,1));
                gp.cf{2}.conf_info = conf_info;
            end
        end
        if bigiter < num_bigiter_hess
            mean_y = mean(E_all(2*D+1:end,:));
            range_y = max(E_all(2*D+1:end,:))-min(E_all(2*D+1:end,:));
            range_x = max(max(dist_at(R_all(2*D+1:end,:),R_all(2*D+1:end,:),conf_info,1)));
        else
            mean_y = mean(E_all);
            range_y = max(E_all)-min(E_all);
            range_x = max(max(dist_at(R_all,R_all,conf_info,1)));
        end
        gp.cf{1}.constSigma2 = mean_y^2;
        gp.cf{2}.p.magnSigma2.s2 = (range_y/3)^2;
        gp.cf{2}.p.lengthScale.s2 = (range_x/3)^2;
        if bigiter < 1 || bigiter+1 <= num_bigiter_initparam || conf_info.n_pt > size(gp.cf{2}.lengthScale,2)
            gp.cf{2}.magnSigma2 = norminv(0.75,0,range_y/3)^2;
            gp.cf{2}.lengthScale = repmat(norminv(0.75,0,range_x/3),1,conf_info.n_pt);
        end
        R_all2 = [repmat(R_all,D+1,1),reshape(repmat(0:D,size(R_all,1),1),[],1)];
        gp = gp_optim(gp,R_all2,[E_all;G_all(:)],'opt',opt,'optimf',optimf);
        param_gp = [param_gp;gp_pak(gp)]
        [~, C] = gp_trcov(gp, R_all2);
        L = chol(C,'lower');
        a = L'\(L\[E_all;G_all(:)]);
        
        % define the convergence threshold for the relaxation phase
        if divisor_T_MEP_gp > 0
            % if this option is set on, the GP convergence threshold is 1/'divisor_T_MEP_gp'
            % of the smallest accurate 'maxG_R_perp' obtained so far for any of the intermediate images,
            % but not less than 1/10 of the lowest final threshold
            T_MEP_gp = max([min(min(maxG_R_perp_acc))/divisor_T_MEP_gp,min([T_MEP/10,T_CI/10])]);
        else
            % otherwise the GP convergence threshold is always 1/10 of the lowest final threshold
            T_MEP_gp = min([T_MEP,T_CI])/10;
        end
        
        % define the start path for the relaxation phase
        if islarge_num_iter > 0 || isempty(R_previous_equal)
            if bigiter >= num_bigiter_initpath && ~isempty(R_latest_equal)
                R = R_latest_equal;
                if T_CIon_gp > 0
                    fprintf('Started relaxation phase %g from the latest "preliminarily converged" evenly spaced path (no climbing image).\n', bigiter+1);
                else
                    fprintf('Started relaxation phase %g from the latest converged path.\n', bigiter+1);
                end
            else
                R = R_init;
                fprintf('Started relaxation phase %g from the initial path.\n', bigiter+1);
            end
        else
            R = R_previous_equal;
            fprintf('Started relaxation phase %g where the previous one stopped.\n', bigiter+1);
            R_previous_equal = [];
        end
        
        iters = 0;
        CI_on = 0; % set climbing image mode off in the beginning
        V_old = zeros(N_im-2,D); % velocities of the intermediate images (given as an output of the previous step)
        F_R_old = 0; % NEB forces on the intermediate images of the previous path
        zeroV = 1; % indicator if zero velocity used (for the first iteration)
        
        for iter = 0:num_iter % INNER ITERATION LOOP
            
            % calculate estimated energy and gradient on the path
            KK = gp_cov(gp,R_all2,[repmat(R,D+1,1),reshape(repmat(0:D,N_im,1),[],1)]);
            EG_R = KK'*a;
            E_R = EG_R(1:N_im,1);
            G_R = reshape(EG_R(N_im+1:end,1),N_im,D);
            [F_R,~,maxG_CI,i_CI] = method_force(R,E_R,G_R,param_force,T_CIon_gp);
            maxF_R = max(abs(F_R),[],2);
            
            % turn climbing image mode on and correct the NEB force accordingly if sufficiently relaxed
            if CI_on <= 0 && max(maxF_R) < T_CIon_gp
                R_latest_equal = R;
                CI_on = 1;
                [~,i_CI_test] = max(E_R(2:(N_im-1),:));
                fprintf('Climbing image (image %g) turned on after %g inner iterations. \n', i_CI_test+1, iters);                  
                if islarge_num_iter > 0 || isempty(R_previous_climb)
                    if bigiter+1 > num_bigiter_initpath && ~isempty(R_latest_climb)
                        R_start_climb = R_latest_climb;
                        R_start_climb_text = 'latest converged';
                        i_CI_start = i_CI_latest;
                    else
                        R_start_climb = [];
                        i_CI_start = 0;
                    end
                else
                    R_start_climb = R_previous_climb;
                    R_start_climb_text = 'previous';
                    R_previous_climb = [];
                    i_CI_start = i_CI_previous;
                    i_CI_previous = 0;
                end
                if i_CI_test == i_CI_start
                    KK_test2 = gp_cov(gp,R_all2,[repmat(R_start_climb,D+1,1),reshape(repmat(0:D,N_im,1),[],1)]);
                    EG_R_test2 = KK_test2'*a;
                    E_R_test2 = EG_R_test2(1:N_im,1);
                    [~,i_CI_test2] = max(E_R_test2(2:(N_im-1),:));
                    if i_CI_test2 == i_CI_start
                        R = R_start_climb;
                        E_R = E_R_test2;
                        G_R = reshape(EG_R_test2(N_im+1:end,1),N_im,D);
                        fprintf('CI unchanged: continued from the %s CI-path. \n', R_start_climb_text);
                    end
                end
                [F_R,~,maxG_CI,i_CI] = method_force(R,E_R,G_R,param_force,CI_on);
                maxF_R = max(abs(F_R),[],2);
                zeroV = 1;
            end
            
            E_R_gp = [E_R_gp,E_R];
            maxF_R_gp = [maxF_R_gp,maxF_R];
            maxG_CI_gp = [maxG_CI_gp,maxG_CI];
            
            % stop the relaxation phase if converged
            if ( T_CIon_gp <= 0 || CI_on > 0 ) && max(maxF_R) < T_MEP_gp && iters > 0
                if CI_on > 0
                    R_latest_climb = R;
                    i_CI_latest = i_CI;
                    fprintf('Stopped the relaxation phase: converged after %g inner iterations (CI: image %g).\n', iters, i_CI+1);
                    R_previous_climb = [];
                    i_CI_previous = 0;
                else
                    R_latest_equal = R;
                    fprintf('Stopped the relaxation phase: converged after %g inner iterations.\n', iters);
                end
                break;
            end
            
            % stop the relaxation phase if maximum number of inner iterations reached
            if iters == num_iter
                if islarge_num_iter <= 0
                    if CI_on > 0
                        R_previous_climb = R;
                        i_CI_previous = i_CI;
                    else
                        R_previous_equal = R;
                        R_previous_climb = [];
                        i_CI_previous = 0;
                    end
                end
                fprintf('Stopped the relaxation phase: maximum number of inner iterations (%g) reached.\n', iters)
                break;
            end
            
            % move the path one step along the NEB force according to the chosen method
            [R_new,V_old] = method_step(R,F_R,param_step,F_R_old,V_old,zeroV);
            zeroV = 0;
            
            if actdist_fro < inf
                % check if new active frozen atoms and update 'conf_info' and 'conf_info_inactive'
                [conf_info,conf_info_inactive,new_act] = update_active_fro(conf_info,conf_info_inactive,R_new(2:(N_im-1),:),actdist_fro);
                % if new active frozen atoms, update the GP model and reoptimize hyperparameters
                if new_act > 0
                    fprintf('More frozen atoms activated. Now %g active and %g inactive frozen atoms.\n', size(conf_info.conf_fro,1), size(conf_info_inactive.conf_ifro,1));
                    gp.cf{2}.conf_info = conf_info;
                    if bigiter < num_bigiter_hess
                        range_x = max(max(dist_at(R_all(2*D+1:end,:),R_all(2*D+1:end,:),conf_info,1)));
                    else
                        range_x = max(max(dist_at(R_all,R_all,conf_info,1)));
                    end
                    gp.cf{2}.p.lengthScale.s2 = (range_x/3)^2;
                    if bigiter+1 <= num_bigiter_initparam || conf_info.n_pt > size(gp.cf{2}.lengthScale,2)
                        gp.cf{2}.lengthScale = repmat(norminv(0.75,0,range_x/3),1,conf_info.n_pt);    
                    end
                    R_all2 = [repmat(R_all,D+1,1),reshape(repmat(0:D,size(R_all,1),1),[],1)];
                    gp = gp_optim(gp,R_all2,[E_all;G_all(:)],'opt',opt,'optimf',optimf);
                    [~, C] = gp_trcov(gp, R_all2);
                    L = chol(C,'lower');
                    a = L'\(L\[E_all;G_all(:)]);
                    zeroV = 1;
                end
            end
            
            % limit the move if step length is larger than 99 % of 'disp_max' times the length of the initial path or
            % if any atom-wise step length is larger than 99 % of 0.5*(1-'ratio_at_limit') times the minimum inter-atomic distance
            steplength = sqrt(sum((R_new(2:N_im-1,:)-R(2:N_im-1,:)).^2,2));
            steplength_atomwise = sqrt((R_new(2:N_im-1,1:3:end)-R(2:N_im-1,1:3:end)).^2+(R_new(2:N_im-1,2:3:end)-R(2:N_im-1,2:3:end)).^2+(R_new(2:N_im-1,3:3:end)-R(2:N_im-1,3:3:end)).^2); % atom-wise step lengths of the intermediate images
            steplength_atomwise_limit = 0.5*(1-ratio_at_limit)*mindist_interatomic(R(2:N_im-1,:),conf_info);
            if any(steplength > 0.99*disp_max*scale) || any(any(steplength_atomwise > 0.99*steplength_atomwise_limit))
                step_coeff = min(ones(N_im-2,1),0.99*disp_max*scale./steplength);
                step_coeff = min(step_coeff,0.99*min(steplength_atomwise_limit./steplength_atomwise,[],2));
                fprintf('Warning: the step length of inner iteration %g limited.\n', iters+1)
                R_new(2:N_im-1,:) = R(2:N_im-1,:) + bsxfun(@times,step_coeff,R_new(2:N_im-1,:)-R(2:N_im-1,:));
                zeroV = 1;
            end
            
            % STOPPING CRITERION FOR INTER-ATOMIC DISTANCES
            % reject the step and stop the relaxation phase if the following does not hold for some of the current images:
            % there is an observed data point so that all inter-atomic distances of the current image are more than 'ratio_at_limit'
            % (by default 2/3) but less than 1/'ratio_at_limit' (3/2) times the corresponding inter-atomic distance of the observed data point,
            % i.e., |log(r_im/r_nearobs)| < |log(ratio_at_limit)| ( = |log(2/3)| = 0.4055 )
            if iters > 0
			    [disp1D_nearest,obs_nearest] = min(dist_max1Dlog(R_new(2:N_im-1,:),R_all,conf_info),[],2);
                [maxdisp1D_nearest,i_maxdisp1D_nearest] = max(disp1D_nearest);
                if maxdisp1D_nearest > abs(log(ratio_at_limit))
                    fprintf('Stopped the relaxation phase after %g inner iterations: inter-atomic distance in image %g changes too much compared to "nearest" observed data point.\n', iters, i_maxdisp1D_nearest+1)
                    if T_CIon_gp > 0
                        R_previous_climb = [];
                        i_CI_previous = 0;
                    end
                    break;
                end
            end
            
            %{
            % STOPPING CRITERION FOR JOINT MOVEMENT OF ATOMS (OPTIONAL)
            % reject the step and stop the relaxation phase if, for some of the current images, there does not exist
            % an observed data point that fulfils the following requirement:
            % for all moving atoms, the change in the position of the atom between the current image and
            % the observed data point is not more than 1/2 of the distance from the atom to its nearest
            % neighbour atom in the current image or the observed data point           
            if iters > 0
                [dispmaxrel_nearest,obs_nearest] = min(dist_maxrel_atomwise3(R_new(2:N_im-1,:),R_all,conf_info),[],2);
                [maxdispmaxrel_nearest,i_maxdispmaxrel_nearest] = max(dispmaxrel_nearest);
                if maxdispmaxrel_nearest > 1/2
                    fprintf('Stopped the relaxation phase after %g inner iterations: atom position in image %g changes too much compared to "nearest" observed data point.\n', iters, i_maxdispmaxrel_nearest+1)
                    if T_CIon_gp > 0
                        R_previous_climb = [];
                        i_CI_previous = 0;
                    end
                    break;
                end
            end
            %}
            
            %{
            % ALTERNATIVE STOPPING CRITERION FOR JOINT MOVEMENT OF ATOMS (OPTIONAL)
            % reject the step and stop the relaxation phase if, for some of the current images, there does not exist
            % an observed data point that fulfils the following requirement:
            % for all moving atoms, the change in the position of the atom between the current image and
            % the observed data point is not more than 1/4 of the distance from the atom to its nearest
            % neighbour atom in the observed data point               
            if iters > 0
                [dispmaxrel_nearest,obs_nearest] = min(dist_maxrel_atomwise2(R_new(2:N_im-1,:),R_all,conf_info),[],2);
                [maxdispmaxrel_nearest,i_maxdispmaxrel_nearest] = max(dispmaxrel_nearest);
                if maxdispmaxrel_nearest > 1/4
                    fprintf('Stopped the relaxation phase after %g inner iterations: atom position in image %g changes too much compared to "nearest" observed data point.\n', iters, i_maxdispmaxrel_nearest+1)
                    if T_CIon_gp > 0
                        R_previous_climb = [];
                        i_CI_previous = 0;
                    end
                    break;
               end
            end
            %}
            
            % THE OLD STOPPING CRITERION FOR RAW DISPLACEMENT
            % reject the step and stop the relaxation phase if the distance from any current image to the
            % nearest observed data point is larger than 'disp_max' times the length of the initial path
            if iters > 0
                disp_nearest = zeros(N_im-2,1);
                for im = 2:N_im-1
                    disp_nearest(im-1,1) = sqrt(min(sum((repmat(R_new(im,:),size(R_all,1),1)-R_all).^2,2)));
                end
                [maxdisp_nearest,i_maxdisp_nearest] = max(disp_nearest);
                if maxdisp_nearest > disp_max*scale
                    fprintf('Stopped the relaxation phase after %g inner iterations: image %g too far from the nearest observed data point.\n', iters, i_maxdisp_nearest+1)
                    if T_CIon_gp > 0
                        R_previous_climb = [];
                        i_CI_previous = 0;
                    end
                    break;
                end
            end
            
            % otherwise accept the step and continue the relaxation
            iters = iters + 1;
            R = R_new;
            if quatern > 0
                for im = 2:N_im-1
                    [R(im,:),~] = doRotation(R(im,:),R(im-1,:));
                end
            end
            F_R_old = F_R;
            
        end % END OF INNER ITERATION LOOP
        
        %figure(11);
        %hold on;
        %plot([R(:,1);R(:,4)],[R(:,3);R(:,6)],'yo','markerFaceColor','y');  
        %plot([R_all(:,1);R_all(:,4)],[R_all(:,3);R_all(:,6)],'r+');
        
        if ~isempty(save_file)
            save(save_file);
        end

    end % END OF OUTER ITERATION LOOP
    
    % visualize the true energy along the spline interpolation of the final path
    %figure(10)
    %hold on
    %N_im = size(R,1);
    %R_spline = spline((0:N_im-1)/(N_im-1),R',(0:0.1:N_im-1)/(N_im-1))';
    %[E_spline,G_spline] = pot_general(R_spline);
    %E_spline = E_spline - Elevel;
    %[E_images,G_images] = pot_general(R);
    %E_images = E_images - Elevel;
    %plot(1:0.1:N_im,E_spline,'b','LineWidth',2);
    %plot(1:N_im,E_images,'o','MarkerEdgeColor','b','MarkerFaceColor','b')
    %hold off
    
end


