%%%%% appr_hessian.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 15.9.2017
%%%%%
%%%%% This function approximates the Hessian at the saddle point on the GP surface using finite differences.
%%%%%
%%%%% Input:   R_sp          coordinates of the saddle point (1 x D)
%%%%%          eps_Hess      epsilon for the finite difference
%%%%%          gp            gp model
%%%%%          R_all         coordinates for all observations
%%%%%          L             variable unchanged during the Newton-Rapshon phase
%%%%%          a             variable unchanged during the Newton-Rapshon phase
%%%%%
%%%%% Output:  H_R_sp_gp     approximated Hessian at the saddle point
%%%%%          stdH_R_sp_gp  standard deviation of the Hessian at the saddle point

function [H_R_sp_gp,stdH_R_sp_gp] = appr_hessian(R_sp,eps_Hess,gp,R_all,L,a)
    D = size(R_sp,2);
    H_R_sp_gp = zeros(D,D);
    stdH_R_sp_gp = zeros(D,D);
    R_all2 = [repmat(R_all,D+1,1),reshape(repmat(0:D,size(R_all,1),1),[],1)];
    for i = 1:D
        R_sp_h = [R_sp;R_sp];
        R_sp_h(1,i) = R_sp(1,i) - eps_Hess;
        R_sp_h(2,i) = R_sp(1,i) + eps_Hess;
        R_sp_h2 = [repmat(R_sp_h,D+1,1),reshape(repmat(0:D,2,1),[],1)];
        KKh = gp_cov(gp,R_all2,R_sp_h2);
        EG_R_sp_h_gp = KKh'*a;
        G_R_sp_h_gp = reshape(EG_R_sp_h_gp(3:end,1),2,D);
        H_R_sp_gp(i,:) = (G_R_sp_h_gp(2,:)-G_R_sp_h_gp(1,:))/(2*eps_Hess);
        Cov_prior = gp_trcov(gp,R_sp_h2);
        v = L\KKh;
        Cov_post = Cov_prior - v'*v;
        VarG_R_sp_plus_gp = diag(Cov_post(2+2*(1:D),2+2*(1:D)))';
        VarG_R_sp_minus_gp = diag(Cov_post(1+2*(1:D),1+2*(1:D)))';
        CovG_R_sp_plus_minus_gp = diag(Cov_post(2+2*(1:D),1+2*(1:D)))';
        stdH_R_sp_gp(i,:) = sqrt(VarG_R_sp_plus_gp+VarG_R_sp_minus_gp-2*CovG_R_sp_plus_minus_gp)/(2*eps_Hess);
    end
end
