%%%%% NEB2.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 22.2.2019
%%%%%
%%%%% This function uses the nudged elastic band (NEB)
%%%%% method with a climbing image option to find a minimum energy path
%%%%% and a saddle point between two minimum points.
%%%%%
%%%%% Input:
%%%%%   pot_general        accurate potential and gradient function
%%%%%   R_init             coordinates for the images on the initial path (N_im x D)
%%%%%   method_step        a function defining the following step during path relaxation
%%%%%   param_step         parameters of the path relaxation method (shape depends on 'method_step')
%%%%%   method_force       a function defining the NEB force
%%%%%   param_force        parameters of the NEB force method (shape depends on 'method_force')
%%%%%   T_MEP              convergence threshold (the algorithm is stopped when the maximum component
%%%%%                        of gradient perpendicular to the path tangent is less than this for all images)
%%%%%   T_CI               additional convergence threshold for the maximum component of gradient at the climbing image
%%%%%   T_CIon             preliminary convergence threshold after which the climbing image
%%%%%                        mode is turned on (use 0 if CI not used at all)
%%%%%   num_iter           maximum number of iterations
%%%%%   quatern            indicator if quaternion trick used to remove rotation/translation of system
%%%%%
%%%%% Output:
%%%%%   R                  coordinates for the images on the final path (N_im x D)
%%%%%   E_R                energy at the images on the final path (N_im x 1)
%%%%%   G_R                gradient at the images on the final path (N_im x D)
%%%%%   i_CI               index of the climbing image among the intermediate images of the final path
%%%%%   E_R_acc            energies of the images for each iteration
%%%%%   maxG_R_perp_acc    maximum component of gradient perpendicular to the path tangent at each intermediate image for each iteration
%%%%%   maxG_CI_acc        maximum component of gradient at the climbing image for each iteration (0 if CI is off)

function [R,E_R,G_R,i_CI,E_R_acc,maxG_R_perp_acc,maxG_CI_acc] = NEB2(pot_general,R_init,method_step,param_step,method_force,param_force,T_MEP,T_CI,T_CIon,num_iter,quatern)


    %%%     
    %%% THIS INFORMATION IS ASSUMED TO BE KNOWN BEFORE BEGINNING
    %%%
    
    N_im = size(R_init,1); % number of images on the path
    D = size(R_init,2); % dimension of the space
    min1 = R_init(1,:); % minimum point 1
    [E_min1,G_min1] = pot_general(min1); % energy and gradient at minimum point 1
    min2 = R_init(end,:); % minimum point 2
    [E_min2,G_min2] = pot_general(min2); % energy and gradient at minimum point 2
    % Elevel = min([E_min1,E_min2]); % zero level of energy is set to the lower minimum
    Elevel = E_min1; % zero level of energy is set to minimum point 1
    E_min1 = E_min1 - Elevel;
    E_min2 = E_min2 - Elevel;
    
    
    %%%
    %%% THE ALGORITHM BEGINS HERE
    %%%
    
    R = R_init; % coordinates of the images
    E_R = [E_min1;zeros(N_im-2,1);E_min2]; % energy at the images
    G_R = [G_min1;zeros(N_im-2,D);G_min2]; % gradient at the images
    
    % in case of 2D space, plot the energy surface
    if D == 2
        scale1 = abs(min2(1)-min1(1))/4;
        scale2 = abs(min2(2)-min1(2))/4;
        [X1,X2] = meshgrid(min(min1(1),min2(1))-2*scale1:scale1/20:max(min1(1),min2(1))+scale1,min(min1(2),min2(2))-scale2:scale2/20:max(min1(2),min2(2))+scale2);
        E_true = zeros(size(X1,1),size(X1,2));
        for j = 1:size(X1,2)
            E_true(:,j) = pot_general([X1(:,j),X2(:,j)])-Elevel;
        end
        figure(1)
        pcolor(X1,X2,E_true),shading flat;
        colorbar;
        axis equal tight;
        hold on;
    end
         
    R_CIon = [];
    E_R_acc = []; % matrix gathering energies of the images for each iteration
    maxG_R_perp_acc = []; % matrix gathering the maximum component of gradient perpendicular to the path tangent at each intermediate image for each iteration
    maxG_CI_acc = []; % vector gathering the maximum component of gradient at the climbing image for each iteration (0 if CI is off)
    
    iters = 0;
    CI_on = 0; % set climbing image mode off in the beginning
    V_old = zeros(N_im-2,D); % velocities of the intermediate images (given as an output of the previous step)
    F_R_old = 0; % NEB forces on the intermediate images on the previous path
    zeroV = 1; % indicator if zero velocity used (for the first iteration)
    
    for iter = 0:num_iter
        
        % calculate energy and gradient on the new path
        [E_R(2:N_im-1,:),G_R(2:N_im-1,:)] = pot_general(R(2:N_im-1,:)); % evaluate the energy and gradient at the intermediate images
        E_R(2:N_im-1,:) = E_R(2:N_im-1,:)-Elevel;
        [F_R,maxG_R_perp,maxG_CI,i_CI] = method_force(R,E_R,G_R,param_force,CI_on);
        
        % turn climbing image mode on and correct the NEB force accordingly if sufficiently relaxed
        if CI_on <= 0 && max(maxG_R_perp) < T_CIon
            CI_on = 1;
            R_CIon = R;
            E_R_CIon = E_R;
            [F_R,maxG_R_perp,maxG_CI,i_CI] = method_force(R,E_R,G_R,param_force,CI_on);
            zeroV = 1;
            fprintf('Climbing image turned on after %g iterations\n', iters);
        end
        
        E_R_acc = [E_R_acc,E_R];
        maxG_R_perp_acc = [maxG_R_perp_acc,maxG_R_perp];
        maxG_CI_acc = [maxG_CI_acc,maxG_CI];
        
        % stop relaxation if converged
        if ( T_CIon <= 0 || CI_on > 0 ) && max(maxG_R_perp) < T_MEP && maxG_CI < T_CI
            fprintf('Stopped relaxation: converged after %g iterations (%g image evaluations).\n', iters, (N_im-2)*(iters+1));
            break;
        end
        
        % stop relaxation if maximum number of iterations reached
        if iters == num_iter
            fprintf('Stopped relaxation: maximum number of iterations (%g) reached.\n', iters)
            break;
        end
        
        % in case of 2D space, plot the path
        if D == 2
            figure(1)
            plot(R(:,1),R(:,2),'yo','markerFaceColor','y')  
        end
        
        % move the path one step along the NEB force according to the chosen method
        [R_new,V_old] = method_step(R,F_R,param_step,F_R_old,V_old,zeroV);
        zeroV = 0;
                    
        % here one can set some condition when the step is rejected (and relaxation stopped)
        % if 'condition'
        %     disp('Stopped relaxation: because of condition')
        %     break;
        % end
        
        % (otherwise) accept the step and continue relaxation
        iters = iters + 1;
        R = R_new;
        if quatern > 0
            for im = 2:N_im-1
                [R(im,:),~] = doRotation(R(im,:),R(im-1,:));
            end
        end
        F_R_old = F_R;

    end
 
    % in case of 2D space, plot the relaxed path
    % (and the path when the climbing image option was set on)
    if D == 2
        figure(1)
        if CI_on > 0
            plot(R_CIon(:,1),R_CIon(:,2),'ro','markerFaceColor','r')
            plot(R(:,1),R(:,2),'ko','markerFaceColor','k')
        else
            plot(R(:,1),R(:,2),'ro','markerFaceColor','r')
        end
    end
    
    % visualize the true energy along the spline interpolation of the final
    % path (and the path when the climbing image mode was set on)
    %figure(10)
    %hold on
    %N_im = size(R,1);
    %R_spline_final = spline((0:N_im-1)/(N_im-1),R',(0:0.1:N_im-1)/(N_im-1))';
    %[E_spline_final,G_spline_final] = pot_general(R_spline_final);
    %E_spline_final = E_spline_final - Elevel;
    %if CI_on > 0
    %    R_spline_CIon = spline((0:N_im-1)/(N_im-1),R_CIon',(0:0.1:N_im-1)/(N_im-1))';
    %    [E_spline_CIon,G_spline_CIon] = pot_general(R_spline_CIon);
    %    E_spline_CIon = E_spline_CIon - Elevel;
    %    plot(1:0.1:N_im,E_spline_CIon,'r','LineWidth',2);
    %    plot(1:N_im,E_R_CIon,'o','MarkerEdgeColor','r','MarkerFaceColor','r');
    %    plot(1:0.1:N_im,E_spline_final,'k','LineWidth',2);
    %    plot(1:N_im,E_R,'o','MarkerEdgeColor','k','MarkerFaceColor','k');
    %else
    %    plot(1:0.1:N_im,E_spline_final,'r','LineWidth',2);
    %    plot(1:N_im,E_R,'o','MarkerEdgeColor','r','MarkerFaceColor','r');
    %end
    %hold off
    
end

