%%%%% GP_newton_rapshon.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 15.9.2017
%%%%%
%%%%% This function finds a saddle point iteratively by using the Newton-Rapshon method
%%%%% on the approximated GP surface, evaluating the energy and gradient,
%%%%% and repeating the Newton-Rapshon phase on the updated GP surface.
%%%%%
%%%%% Input:
%%%%%   pot_general   true potential energy function
%%%%%   Elevel        zero energy level (this will be subtracted from the energy given by 'pot_general')
%%%%%   R_init        coordinates of the initial guess (1 x dim)
%%%%%   E_R_init      energy at the current guess
%%%%%   G_R_init      energy gradient at the current guess (1 x dim)
%%%%%   gp            initial GP model (GPStuff structure)
%%%%%   R_all         coordinates of the N initial observations (N x dim)
%%%%%   E_all         energy at the N initial observations (N x 1)
%%%%%   G_all         gradient at the N initial observations (N x dim)
%%%%%   normG_conv    convergence limit for the norm of the gradient
%%%%%   num_bigiter   maximum number of outer iterations (new observations)
%%%%%   num_iter      maximum number of inner iterations (steps during Newton-Rapshon phase)
%%%%%   eps_Hess      epsilon for the finite difference Hessian
%%%%%   opt           optimization settings for the GP hyperparameters defined using 'optimset'
%%%%%   optimf        optimization function for the GP hyperparameters
%%%%%
%%%%% Output:
%%%%%   R_sp          coordinates of the saddle point
%%%%%   E_R_sp        energy at the saddle point
%%%%%   G_R_sp        gradient at the saddle point
%%%%%   H_R_sp_gp     approximated Hessian at the saddle point
%%%%%   stdH_R_sp_gp  standard deviation of the Hessian at the saddle point

function [R_sp,E_R_sp,G_R_sp,H_R_sp_gp,stdH_R_sp_gp] = GP_newton_rapshon(pot_general,Elevel,...
    R_init,E_R_init,G_R_init,gp,R_all,E_all,G_all,normG_conv,num_bigiter,num_iter,eps_Hess,opt,optimf)
    
    D = size(R_init,2); % dimension of the space  
    R_sp = R_init;
    R_sp2 = [repmat(R_sp,D+1,1),(0:D)'];
    E_R_sp = E_R_init;
    G_R_sp = G_R_init;
    normG = sqrt(sum(G_R_sp.^2,2));
    fprintf('Initial normG: %1.3g\n\n', normG);
    R_all2 = [repmat(R_all,D+1,1),reshape(repmat(0:D,size(R_all,1),1),[],1)];
    
    for bigiter = 0:num_bigiter
        
        % calculate some variables unchanged during the Newton-Rapshon phase
        [~, C] = gp_trcov(gp,R_all2);
        L = chol(C,'lower');
        a = L'\(L\[E_all;G_all(:)]);
        
        % calculate approximated gradient for the current guess
        KK = gp_cov(gp,R_all2,R_sp2);
        EG_R = KK'*a;
        G_R_sp_gp = EG_R(2:end,1)';

        % approximate the Hessian using finite differences
        [H_R_sp_gp,stdH_R_sp_gp] = appr_hessian(R_sp,eps_Hess,gp,R_all,L,a);

        % stop the algorithm if converged
        if normG < normG_conv
            fprintf('Stopped the algorithm: converged after %g outer iterations.\n', bigiter);
            return;
        end
        
        % stop the algorithm if maximum number of outer iterations is reached
        if bigiter == num_bigiter
            fprintf('Stopped the algorithm: Maximum number of outer iterations (%g) reached.\n', bigiter);
            break;
        end
        
        % set the GP convergence limit to 1/10 of the final convergence limit
        normG_conv_gp = normG_conv/10;
        fprintf('Started Newton-Rapshon phase on the GP surface.\n');
        
        for iter = 1:num_iter
                        
            % Newton-Rapshon step
            R_sp = R_sp - (H_R_sp_gp\G_R_sp_gp')';
            R_sp2 = [repmat(R_sp,D+1,1),(0:D)'];
            
            % calculate approximated gradient for the current guess
            KK = gp_cov(gp,R_all2,R_sp2);
            EG_R_sp_gp = KK'*a;
            G_R_sp_gp = EG_R_sp_gp(2:end,1)';
            normG_gp = sqrt(sum(G_R_sp_gp.^2,2));
            fprintf('Approximated normG after %g inner iterations: %1.3g\n', iter, normG_gp);
            
            if sqrt(sum(G_R_sp_gp.^2,2)) < normG_conv_gp
                fprintf('Stopped Newton-Rapshon phase on the GP surface: converged after %g inner iterations.\n', iter);
                break;
            end

            % stop the algorithm if maximum number of outer iterations is reached
            if iter == num_iter
                fprintf('Stopped Newton-Rapshon phase on the GP surface: maximum number of inner iterations (%g) reached.\n', iter);
                break;
            end
            
        % approximate the Hessian using finite differences
        [H_R_sp_gp,stdH_R_sp_gp] = appr_hessian(R_sp,eps_Hess,gp,R_all,L,a);
                     
        end
        
        % acquire true energy and gradient at the current guess and add them to data
        [E_R_sp,G_R_sp] = pot_general(R_sp);
        E_R_sp = E_R_sp - Elevel;
        R_all = [R_all;R_sp];
        R_all2 = [repmat(R_all,D+1,1),reshape(repmat(0:D,size(R_all,1),1),[],1)];
        E_all = [E_all;E_R_sp];
        G_all = [G_all;G_R_sp];
        normG = sqrt(sum(G_R_sp.^2,2));
        fprintf('Accurate normG after %g outer iterations: %1.3g\n\n', bigiter+1, normG);
        
        % optimize the GP hyperparameters
        gp = gp_optim(gp,R_all2,[E_all;G_all(:)],'opt',opt,'optimf',optimf);
        
    end
    
end
