%% Test the example implementations
%
% An example of GP and TP regression with the Matern covariance function,
% and hyperparameter ML optimization. This example demonstartes how to 
% call the TP and GP regression implementations done for the `naive'
% implementations (gp_solve and tp_solve) and the state space
% implementations (gf_solve and tf_solve).
%
% This code acts as proof-of-concept, and this Matlab implementation is 
% optimized for clarity, not speed. 
%
% Copyright:
%   2014-2015   Arno Solin
%
%  This software is distributed under the GNU General Public
%  License (version 3 or later); please refer to the file
%  License.txt, included with the software, for details.

%%  
  
  % Load (x,y,xt)
  load('example-data.mat')
  
  % Covariance function (Matern, nu=3/2)
  k = @(r,p) p(1)*(1+sqrt(3)*abs(r)/p(2)).*exp(-sqrt(3)*abs(r)/p(2));
  
  % Derivatives of covariance function (Matern, nu=3/2)
  dk{1} = @(r,p) (1+sqrt(3)*abs(r)/p(2)).*exp(-sqrt(3)*abs(r)/p(2));
  dk{2} = @(r,p) p(1)*3*r.^2/p(2)^3.*exp(-sqrt(3)*abs(r)/p(2));
  
  % Set up the corresponding state space model (Matern, nu=3/2):
  ss = @(x,p) cf_matern32_to_ss(p(1), p(2));   
    
  % Concatenate initial parameters (nu, sigma2, magnSigma2, lengthScale)
  param = [2.1 .1 .1 .1];
  
  % Optimization options
  opts = optimset('GradObj','on','display','iter');
    
  % Figure
  figure(1); clf
  
  
  % ## GP (naive)
  
  % Optimize hyperparameters w.r.t. log marginal likelihood
  [w,ll] = fminunc(@(w) gp_solve(w,x,y,k,dk), ...
      log(param(2:end)),opts);

  % Assign optimized params
  param_gp = exp(w);

  % Predict using the optimized GP
  [Eft,Varft,Covft,lb,ub] = gp_solve(log(param_gp),x,y,k,xt);

  % Visualize
  subplot(221);
    plot(x,y,'+k',xt,Eft,'-b',xt,lb,'--r',xt,ub,'--r')
    xlabel('Input, t'); ylabel('Output, y')
    title('Naive GP solution')
    drawnow

    
  % ## GP (state space)
  
  % Optimize hyperparameters w.r.t. log marginal likelihood
  [w,ll] = fminunc(@(w) gf_solve(w,x,y,ss), ...
      log(param(2:end)),opts);

  % Assign optimized params
  param_gp = exp(w);

  % Predict using the optimized GP
  [Eft,Varft,Covft,lb,ub] = gf_solve(log(param_gp),x,y,ss,xt);

  % Visualize
  subplot(222);
    plot(x,y,'+k',xt,Eft,'-b',xt,lb,'--r',xt,ub,'--r')
    xlabel('Input, t'); ylabel('Output, y')
    title('State space GP solution')
    drawnow
    
    
  % ## TP (naive)
  
  % Optimize hyperparameters w.r.t. log marginal likelihood
  [w,ll] = fminunc(@(w) tp_solve(w,x,y,k,dk), ...
      log(param),opts);

  % Assign optimized params
  param_tp = exp(w);

  % Predict using the optimized TP
  [Eft,Varft,Covft,lb,ub] = tp_solve(log(param_tp),x,y,k,xt);

  % Visualize
  subplot(223);
    plot(x,y,'+k',xt,Eft,'-b',xt,lb,'--r',xt,ub,'--r')
    xlabel('Input, t'); ylabel('Output, y')
    title('Naive TP solution')
    drawnow

    
  % ## TP (state space)
  
  % Optimize hyperparameters w.r.t. log marginal likelihood
  [w,ll] = fminunc(@(w) tf_solve(w,x,y,ss), ...
      log(param),opts);

  % Assign optimized params
  param_tp = exp(w);

  % Predict using the optimized TP
  [Eft,Varft,Covft,lb,ub] = tf_solve(log(param_tp),x,y,ss,xt);

  % Visualize
  subplot(224);
    plot(x,y,'+k',xt,Eft,'-b',xt,lb,'--r',xt,ub,'--r')
    xlabel('Input, t'); ylabel('Output, y')
    title('State space TP solution')
    drawnow
    
