function [meanf,Varf] = ss_predict(varargin)
% SS_PREDICT - Predicts a set of test points from state space form
%
% Syntax:
%   [meanf,Varf,Covf] = ss_predict(varargin)
%
% In:
%   model       - Stucture of state space model
%   xt          - Vector of test points (model.x)
%   components  - If true, predict ss models component-wise (false)
%
% Out:
%   meanf       - Predicted mean
%   Varf        - Predicted variance
%
% The model structure:
%
%   Required:
%
%   model.x			- Measurement points (required)
%   model.y			- Measurement values (required)
%   
%   Optional:
%
%   model.sigma2	- Measurement noise variance (default: 1)
%   model.opt		- sigma2 is optimized if true or does not exist, 
%                     otherwise sigma2 not optimized (optional)
%   model.ss{i}		- i:th state space model, one model structure can have 
%                     multiple ss models
%   model.ss{i}.make_ss - Function handle, function which forms 
%                     the state space model (default @cf_se_to_ss)
%   model.ss{i}."name of parameter" - Value of parameter, one ss model can 
%                     have multiple parameters (defaults defined at
%                     model.ss{i}.make_ss)
%   model.ss{i}.opt	- Cell of parameter names, which are optimized, other 
%                     parameters are assumed constant. If the opt cell does 
%                     not exist, all proper parameters all optimized. Which
%                     parameters are proper, is defined in 
%                     model.ss{i}.make_ss
%
%     The state space model is given as follows in terms of a stochastic 
%   differential equation
%
%      df(t)/dt = F f(t) + L w(t),
%
%   where w(t) is a white noise process with spectral denisty Qc. F and L
%   are defined matrices. Observation model for discrete observation y_k 
%   of f(t_k) at step k is as follows,
%
%      y_k = H f(t_k) + r_k, r_k ~ N(0, R) 
%
%   where r_k is the Gaussian measurement noise wit covariance R and H is
%   defined vector. Syntax of model.ss{i}.make_ss is as follows, 
%   
%     [F,L,Qc,H,Pinf,dF,dQc,params] = model.ss{i}.make_ss(...),
%
%   where Pinf is the stationary covariance, dF and dQc are derivatives of
%   F and Qc w.r.t. the parameters. params is a structure of input and 
%   output parameter information (see, e.g., CF_EXP_TO_SS for more 
%   details).
%   
% See also:
%   SS_PREDICT, KF_LIKELIHOOD_EG, SS_STACK, SS_SET, SS_PAK, SS_UNPAK
%
% Copyright:
%   2012-2014 Arno Solin
%   2013-2014 Jukka Koskenranta
%
% This software is distributed under the GNU General Public
% License (version 3 or later); please refer to the file
% License.txt, included with the software, for details.


%% Parse input

  % Set up input parser
  P = inputParser;
  
  % Add covariance function parameters
  P.addRequired('model');            % The model
  P.addOptional('xt',[]);            % Length scale parameter
  P.addOptional('components',false); % Output component-wise
  P.addOptional('filteronly',false); % Only run filter
  
  % Parse given inputs
  P.parse(varargin{:});
  P = P.Results;  

%% Assign variable values and check defaults
  
  % Extarct the model
  model = P.model;
  components = P.components;
  
  % Check the rigth form of data in model.x and model.y
  if isfield(model,'x') && isfield(model,'y') && ...
     ~isempty(model.x) && ~isempty(model.y)   
 
     % Set data
     x = model.x;
     y = model.y;
 
  else
     
     % Throw error if no data supplied
     error('No data supplied.')
     
  end
  
  % Set the optional parameters
  model = ss_set(model);
  
  % If no test points given, use training points
  if isempty(P.xt)
    xt = model.x;
  else
    xt = P.xt;
  end

  % Combine observations and test points
  xall = [x(:); xt(:)];
  yall = [y(:); nan(numel(xt),1)];
    
  % Make sure the points are unique and in ascending order
  [xall,sort_ind,return_ind] = unique(xall,'first');
  yall = yall(sort_ind);
    
  % Only return test indices;
  return_ind = return_ind(end-numel(xt)+1:end);
            
  % Extract the noise magnitude from the GP likelihood model
  R = model.sigma2;
    
  % Make the state space model
  if components==true
    [F,L,Qc,H,Pinf,dF,dQc,dPinf,dR,Hs] = ss_stack(model);
  else  
    [F,L,Qc,H,Pinf] = ss_stack(model);
    Hs = H;
  end
  
  % Balance matrices for numerical stability
  %[F,L,Qc,H,Pinf] = ss_balance(F,L,Qc,H,Pinf);
          
  % Set initial state
  m = zeros(size(F,1),1);
  P = Pinf;
    
  % Allocate space for results
  MS = zeros(size(m,1),size(yall,1));
  PS = zeros(size(m,1),size(m,1),size(yall,1));
  A = zeros(size(F,1),size(F,2),size(yall,1));
  Q = zeros(size(P,1),size(P,2),size(yall,1));
  
  % Initial dt
  dt = inf;
  
  % Kalman filter
  for k=1:numel(yall)
      
      % Solve A using the method by Davison
      if (k>1)
          
          % Discrete-time solution (only for stable systems)
          dt_old = dt;
          dt = xall(k)-xall(k-1);
          
          % Should we calculate a new discretization?
          if abs(dt-dt_old) < 1e-9
              A(:,:,k) = A(:,:,k-1);
              Q(:,:,k) = Q(:,:,k-1);
          else
              A(:,:,k) = expm(F*dt);
              Q(:,:,k) = Pinf - A(:,:,k)*Pinf*A(:,:,k)';
          end
          
          % Prediction step
          m = A(:,:,k) * m;
          P = A(:,:,k) * P * A(:,:,k)' + Q(:,:,k);
          
      end
      
      % Update step
      if ~isnan(yall(k))
          S = H*P*H'+R;
          K = P*H'/S;
          v = yall(k,:)'-H*m;
          m = m + K*v;
          P = P - K*H*P;
      end
      
      % Store estimate
      MS(:,k)   = m;
      PS(:,:,k) = P;
      
  end
  
  % Rauch-Tung-Striebel smoother
  for k=size(MS,2)-1:-1:1
      
      % Smoothing step (using Cholesky for stability)
      PSk = PS(:,:,k);
      
      % Solve the Cholesky factorization
      [L,notposdef] = chol(A(:,:,k+1)*PSk*A(:,:,k+1)'+Q(:,:,k+1),'lower');
      
      % Numerical problems in Cheloseky, retry with jitter
      if notposdef>0
          jitter = gp.jitterSigma2*diag(rand(size(A,1),1));
          L = chol(A(:,:,k+1)*PSk*A(:,:,k+1)'+Q(:,:,k+1)+jitter,'lower');
      end
      
      % Continue smoothing step
      G = PSk*A(:,:,k+1)'/L'/L;
      m = MS(:,k) + G*(m-A(:,:,k+1)*MS(:,k));
      P = PSk + G*(P-A(:,:,k+1)*PSk*A(:,:,k+1)'-Q(:,:,k+1))*G';
      
      % Store estimate
      MS(:,k)   = m;
      PS(:,:,k) = P;
      
  end
  
  % These indices shall remain
  MS = MS(:,return_ind);
  PS = PS(:,:,return_ind);
  
  % Return mean
  meanf = Hs*MS;
  
  % Return variance
  if nargout > 1
    if components==false
      Varf = arrayfun(@(k) Hs*PS(:,:,k)*Hs',1:size(PS,3));  
    else
      Varf = zeros(size(Hs,1),size(Hs,1),size(MS,2));
      for k=1:size(MS,2)
        Varf(:,:,k)  = Hs*PS(:,:,k)*Hs';
      end
    end
  end
  
  