function [model,loglik] = ss_optimize(varargin)
% SS_OPTIMIZE - State space hyperparameter optimization
%
% Syntax:
%   [model, loglik] = ss_optimize(varargin)
%
% In:
%   model           - Stucture of the state space model (required)
%   optimizer       - Function handle, used optimizer (@fminunc)
%   options         - Options to be passed to the optimizer function
%   use_derivatives - 'off' if derivatives not used and 'on' otherwise ('off')
%
% Out:
%   model           - Updated state space model
%   loglik          - Negative log marginal likelihood
%
% The model structure:
%
%   Required:
%
%   model.x			- Measurement points (required)
%   model.y			- Measurement values (required)
%   
%   Optional:
%
%   model.sigma2	- Measurement noise variance (default: 1)
%   model.opt		- sigma2 is optimized if true or does not exist, 
%                     otherwise sigma2 not optimized (optional)
%   model.ss{i}		- i:th state space model, one model structure can have 
%                     multiple ss models
%   model.ss{i}.make_ss - Function handle, function which forms 
%                     the state space model (default @cf_se_to_ss)
%   model.ss{i}."name of parameter" - Value of parameter, one ss model can 
%                     have multiple parameters (defaults defined at
%                     model.ss{i}.make_ss)
%   model.ss{i}.opt	- Cell of parameter names, which are optimized, other 
%                     parameters are assumed constant. If the opt cell does 
%                     not exist, all proper parameters all optimized. Which
%                     parameters are proper, is defined in 
%                     model.ss{i}.make_ss
%
%     The state space model is given as follows in terms of a stochastic 
%   differential equation
%
%      df(t)/dt = F f(t) + L w(t),
%
%   where w(t) is a white noise process with spectral denisty Qc. F and L
%   are defined matrices. Observation model for discrete observation y_k 
%   of f(t_k) at step k is as follows,
%
%      y_k = H f(t_k) + r_k, r_k ~ N(0, R) 
%
%   where r_k is the Gaussian measurement noise wit covariance R and H is
%   defined vector. Syntax of model.ss{i}.make_ss is as follows, 
%   
%     [F,L,Qc,H,Pinf,dF,dQc,params] = model.ss{i}.make_ss(...),
%
%   where Pinf is the stationary covariance, dF and dQc are derivatives of
%   F and Qc w.r.t. the parameters. params is a structure of input and 
%   output parameter information (see, e.g., CF_EXP_TO_SS for more 
%   details).
%   
% See also:
%   SS_PREDICT, KF_LIKELIHOOD_EG, SS_STACK, SS_SET, SS_PAK, SS_UNPAK
%
% Copyright:
%   2012-2014 Arno Solin
%   2013-2014 Jukka Koskenranta
%
% This software is distributed under the GNU General Public
% License (version 3 or later); please refer to the file
% License.txt, included with the software, for details.

%% Defaults

  % Default options for the optimizer
  options = optimset('GradObj','on');
  options = optimset(options,'TolX', 1e-3);
  options = optimset(options,'LargeScale', 'off');
  options = optimset(options,'Display', 'off');
  options = optimset(options,'DerivativeCheck', 'off');
  

%% Parse input

  % Set up input parser
  P = inputParser;
  
  % Add covariance function parameters
  P.addRequired('model');                       % The model
  P.addOptional('use_derivatives','off');       % Are we using derivatives
  P.addOptional('optimizer',@fminunc);          % The optimizer
  P.addOptional('options',options);             % The optimizer options
  
  % Parse given inputs
  P.parse(varargin{:});
  P = P.Results;  
  
  
%% Assign variable values and check defaults
  
  % Extract the model
  model = P.model;
  
  % Check the rigth form of data in model.x and model.y
  if isfield(model,'x') && isfield(model,'y') && ...
     ~isempty(model.x) && ~isempty(model.y) && ...
     numel(model.x)==numel(model.y)
 
     % Make sure we deal with row vectors
     model.x = model.x(:)';
     model.y = model.y(:)';
     
  else
     
     % Throw error if no data supplied
     error('There''s something wrong with the data you supplied.')
     
  end
  
  % Set the optional parameters
  model = ss_set(model);
  
  % The optimizer
  optimizer = P.optimizer;

  % The optimizer options
  options = optimset(options,P.options);
  
  % Should we use derivatives
  if ~isfield(options,'GradObj')
    options = optimset(options,'GradObj',P.use_derivatives); 
  end
  
  % Make sure the model is right form
  [w0,pnames] = ss_pak(model);

  
%% Optimization
  
  % Target function
  fun = @(x) optfun(model,x,pnames);

  % Do ML optimization
  if ~isempty(w0)
      
    % Run the optimization
    [x,loglik] = optimizer(fun,log(w0(:)),options);
  
    % Return the model
    model = ss_unpak(model, exp(x), pnames);

  else
    
    % This version of this function requires parameters to be optimized
    error('No parameters to optimize.')
    
  end
   
  
end

% The optimization target function:
%
% Run filter for evaluating the marginal likelihood:
% (this is for stable models; see the Matrix fraction decomposition 
% version for other models. However, this should do for now. ~Arno)  
%
function [edata, gdata, model] = optfun(model, w, pnames)
  
  % Set parameters in w to right place in model
  model = ss_unpak(model, exp(w), pnames);
  
  % Make combined state space model
  [F,L,Qc,H,Pinf,dF,dQc,dPinf,dR] = ...
      ss_stack(model, pnames);
    
  % Balance matrices for numerical stability
  [F,L,Qc,H,Pinf,dF,dQc,dPinf] = ...
      ss_balance(F,L,Qc,H,Pinf,dF,dQc,dPinf);
  
  % The data and measurement noise model
  x = model.x(:)';
  y = model.y(:)';
  R = model.sigma2;
  
  % Evaluate
  if nargout == 1
  
    % Evaluate energy
    [edata] = kf_likelihood_eg(F,L,Qc,H,R,Pinf,dF,dQc,dR,dPinf,y,x);
  
  else
      
    % Evaluate energy and its gradient
    [edata, gdata] = kf_likelihood_eg(F,L,Qc,H,R,Pinf,dF,dQc,dR,dPinf,y,x);
    
    % Account for log-transform of the parameters
    gdata = gdata.*exp(w(:)');
    
  end
  
end
