#include "blksdp.h"

/* ************************************************************
   PROCEDURE symproj -- Y = (X+X')/2
   INPUT x, n - full n x n matrix x.
   OUTPUT y - on output, contains (x+x')/2
   ************************************************************ */
void symproj(double *y, const double *x, const int n)
{
  int colp,i,j;
  double yij;

  /* ------------------------------------------------------------
     x points to x(:,i);     x+colp = x(:,j).
     ------------------------------------------------------------ */
  for(i = 0; i < n; x += n, y += n, i++){
    y[i] = x[i];                         /* diagonal entry */
    for(colp = n + i, j=i+1; j<n; j++, colp += n){
      yij = (x[j] + x[colp]) / 2;         /* x(i,j)+x(j,i) */
      y[j] = yij;
      y[colp] = yij;
    }
  }
}

/* ************************************************************
   PROCEDURE skewproj -- Y = (X-X')/2
   INPUT x, n - full n x n matrix x.
   OUTPUT y - on output, contains (x-x')/2
   ************************************************************ */
void skewproj(double *y, const double *x, const int n)
{
  int colp,i,j;
  double yij;

  /* ------------------------------------------------------------
     x points to x(:,i);     x+colp = x(:,j).
     ------------------------------------------------------------ */
  for(i = 0; i < n; x += n, y += n, i++){
    y[i] = 0.0;                         /* diagonal entry */
    for(colp = n + i, j=i+1; j<n; j++, colp += n){
      yij = (x[j] - x[colp]) / 2;         /* x(j,i) - x(i,j) */
      y[j] = yij;
      y[colp] = -yij;                   /* conjugate */
    }
  }
}

/* ************************************************************
   PROCEDURE vecsymPSD - Let y = (x+x')/2.
   INPUT
     x     - length sum(K.s.^2) vector, to be symmetrized.
     rsdpN - number of real PSD blocks
     sdpN  - total number of PSD blocks, length(K.s)
     sdpNL - K.s, length sdpN array listing the orders of the PSD blocks.
   OUTPUT
     y - length sum(K.s.^2) vector, y = vecsym(x,K).
   ************************************************************ */
void vecsymPSD(double *y, const double *x,const int rsdpN,const int sdpN,
               const double *sdpNL)
{
  int k,nk,nksqr;
/* ------------------------------------------------------------
   Make real PSD blocks symmetric
   ------------------------------------------------------------ */
  for(k = 0; k < rsdpN; k++){
    nk = sdpNL[k]; nksqr = SQR(nk);
    symproj(y,x,nk);
    x += nksqr; y += nksqr;
  }
/* ------------------------------------------------------------
   Make complex PSD blocks Hermitian
   ------------------------------------------------------------ */
  for(; k < sdpN; k++){
    nk = sdpNL[k]; nksqr = SQR(nk);
    symproj(y,x,nk);
    x += nksqr; y += nksqr;
    skewproj(y,x,nk);
    x += nksqr; y += nksqr;
  }
}
