/*
    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include "mex.h"
#include "blksdp.h"

/* ========================= L O R E N T Z ========================= */

/* ************************************************************
   PROCEDURE spqlmul -- Lorentz scale D(x^2)y, with y sparse
     Restrict to subvector x(first:first+n-1). on output,
     D(x^2)y = z + (x'*y)*x.
     VIZ:  D(x^2)y = (x'*y) * x + det(x) * [-y(1); y(2:n)].
   INPUT
     first     - First possible subscript for x,y,z.
     x,detx,n  - x[first:first+n-1] is in interior of Lorentz cone,
	         and its determinant (not its sqrt) is detx.
     y         - sparse vector, with subscripts in [first:first+n-1]
	         The caller has to supply the correct y.jc[0:1].
   OUTPUT
     z         - full length n vector,
	         z(first:first+n-1) = detx * [-y(0); y(1:n-1)]
		 just as sparse as y but stored full.
   RETURNS x'*y
   ************************************************************ */
double spqlmul(double *z, const double *x, const double detx,
               const int first, const int n, const jcir y)
{
  double xdoty;
  int inz;
/* ------------------------------------------------------------
   Let xdoty = x'*y
   ------------------------------------------------------------ */
  for(inz = y.jc[0], xdoty = 0.0; inz < y.jc[1]; inz++)
    xdoty += y.pr[inz] * x[y.ir[inz]];
/* ------------------------------------------------------------
   z[0]     = -detx * y[first],
   z[1:n-1] = detx * y[1:n-1]
   ------------------------------------------------------------ */
  inz = y.jc[0];
  if(y.ir[inz] == first)
    z[first] = - detx * y.pr[inz++];
  for(; inz < y.jc[1]; inz++)
    z[y.ir[inz]] = detx * y.pr[inz];
  return xdoty;
}


/* =========================  S D P :  ========================= */

/* ************************************************************
   PROCEDURE spsymuxu -- R = diag(U*X*U) + 2*tril(U*X*U,-1)
     with X sparse symmetric. Notice that tr(YR) = tr(YUXU).
     Restrict to subvector u(first:first+n^2-1).
     Subscripts X in [first:first+n^2-1]
   INPUT
     u     - full n x n, range [0:(n^2 - 1)]
     x     - sparse nxn symmetric, stored as x = vec(diag(X) + 2*tril(X,-1)).
             x(x.jc[0]:x.jc[1]) in range [first:first+n^2-1]
     first - start counting "vec()" indices at first
             i.e. "[0:n^2-1]" = [first:first+n^2-1]
     n     - order of u, x and r.
     rsubir - Length rsubnnz integer array, subscripts for which r[i] must
             be computed. Range is [first:first+n^2-1] (important !).
     rsubnnz - length of rsubir.
   OUTPUT
     r     - r[first:first+n^2-1] is full n x n,
             On OUTPUT, contains diag(U*X*U) + 2*tril(U*X*U,-1).
   WORKING ARRAYS
     fwork - n x n matrix of doubles, used for storing U*TRIL(X).
     iwork - n vector of integers, for storing nonzero colno's of X
  (Initial contents of working arrays is irrelevant.)
  ************************************************************ */
void spsymuxu(double *r, const double *u, const jcir x,
              const int first, const int n,
              const int *rsubir, const int rsubnnz, double *fwork, int *iwork)
{
  int i,j,k, inz,jnz,jstart,jend,firstj,lastj,lastsub;
  double *uxj;
  const double *uwork;
  double ukj;

/* ------------------------------------------------------------
   STRATEGY: compute "fwork = u*TRIL(x)" and then
   "uxu = (fwork*u) + (u*fwork')".  Typically, x is highly sparse,
   like x=(e_i*e_j'+e_je_i' ), for which fwork has only 1 nonzero
   column, whereas u*x has 2 nonzero columns.  Furthermore, the
   contents of x is already TRIL(X) anyway.
   ------------------------------------------------------------ */
  jnz = 0;
  lastsub = x.ir[x.jc[1]-1];                      /* last nz in vec(x) */
  for(jstart = x.jc[0]; jstart < x.jc[1]; jstart = jend){
/* -------------------------------------------------------------
   NOTE: SPEED-UP POSSIBLE BY 1ST PARTITIONING (INTO NZ-COLS).
   Let j = column number of next nonzero, and iwork[jnz] = j * n,
   firstj:firstj+n-1 is index range column j.
   ------------------------------------------------------------- */
    j = (x.ir[jstart] - first) / n;
    firstj = j * n;
    uxj = fwork + firstj;
    iwork[jnz++] = firstj;
    firstj += first;
/* -------------------------------------------------------------
   Index range is firstj : lastj-1.
   Let x(jstart:jend-1) be the nonzeros in x(:,j).
   ------------------------------------------------------------- */
    if( (lastj = firstj + n) > lastsub)
      jend = x.jc[1];
    else{
      jend = jstart;
      mxAssert(x.ir[jend] < lastj,"");
      while(x.ir[++jend] < lastj);
    }
/* ------------------------------------------------------------
   Compute fwork(:,j) = U' * x(:,j),
   We have uwork[firstj:firstj+n] = uk[0:n] and firstj<=x.ir[inz]<firstj+n
   ------------------------------------------------------------ */
    for(uwork = u - firstj, k=0; k < n; uwork += n, k++){
      for(ukj = 0.0, inz = jstart; inz < jend; inz++)
        ukj +=  uwork[x.ir[inz]] * x.pr[inz];
      uxj[k] = ukj;
    }
  }
/* -------------------------------------------------------------
   Now jnz = |iwork|.
   For each nonzero (i,j) in rsub, compute rij.
   (e.g. in "maxcut", R is diagonal)
   Let k be the last column in r; first compute R(:,0:k-1)
   ------------------------------------------------------------- */
  lastsub = rsubir[rsubnnz-1];
  k = (lastsub - first) / n;        /* k = #last_column, e.g. _[(n^2-1)/n]_ */
  inz = 0;
  firstj = first;
/* ------------------------------------------------------------
   2 * U*X*U = (U*TRIL(X))*U + U*(TRIL(X)'*U) 
   = (fwork     *U) +(U*fwork')
   ------------------------------------------------------------ */
  for(j = 0; j < k; j++){
/* ------------------------------------------------------------
   r(j,j) = fwork(j,:)*u(j,:)'
   ------------------------------------------------------------ */
    if( (i = rsubir[inz]) ==  firstj + j){
      r[i] = selrealdot(fwork + j, u+j, iwork, jnz);
      i = rsubir[++inz];
    }
/* ------------------------------------------------------------
   i>j: r(i,j) = (fwork(i,:)*u(j,:)' + u(i,:) * fwork(j,:)')/2
   ------------------------------------------------------------ */
    lastj = firstj + n;
    for(; i < lastj; i = rsubir[++inz]){
      r[i] = (selrealdot(u+(i-firstj), fwork + j, iwork, jnz)
              + selrealdot(fwork + (i-firstj), u+j, iwork, jnz)) / 2;
    }
    firstj = lastj;     /* point to next column */
  }
/* -------------------------------------------------------------
   Compute R(:,k)
   ------------------------------------------------------------- */
  if( (i = rsubir[inz]) == firstj + k){
    r[i] = selrealdot(fwork + k, u+k, iwork, jnz);
    ++inz;
  }
/* ------------------------------------------------------------
   i>k: r(i,k) = (fwork(i,:)*u(k,:)' + u(i,:) * fwork(k,:)')/2
   ------------------------------------------------------------ */
  for(; inz < rsubnnz; inz++){
    i = rsubir[inz];
    r[i] = (selrealdot(u+(i-firstj), fwork + k, iwork, jnz)
            + selrealdot(fwork + (i-firstj), u+k, iwork, jnz)) / 2;
  }
}

/* ************************************************************
   PROCEDURE sphermuxu -- R = diag(U*X*U) + 2*tril(U*X*U,1)
     with X sparse Hermitian. Notice that tr(YR) = tr(YUXU).
     The complex R,U,X are stored in the format [Re(X) Im(X)].
   INPUT
     u - full n x n Hermitian (2*n^2 floats), starts ar u[0], u[n^2]
     x - sparse nxn Hermitian, stored as x = vec(diag(X) + 2*tril(X,-1)).
	 (RE, IM) nonzeros xjc[0:1].
         index RE x(0,0) is "first" and index IM x(0,0) is "first"+n^2.
     first - start counting "vec()" indices at first
     n     - order of u, x and r.
     rsubir - lists the lower-triangular subscripts for which r_ij
             must be computed. Is subset of [first:first+2*n^2-1].
     rsubnnz - length(rsubir)
   OUTPUT
     r - full 2*(n x n), r = diag(U*X*U) + 2*tril(U*X*U,1).
	 starts at r+first.
   WORKING ARRAYS
     fwork - 2*(n x n) matrix of doubles, used for storing U*TRIL(X).
     iwork - n vector of integers, for storing nonzero colno's of U*X
   (Initial contents of working arrays is irrelevant.)
   ************************************************************ */
void sphermuxu(double *r, const double *u, const jcir x,
               const int first, const int n,
               const int *rsubir, const int rsubnnz, double *fwork, int *iwork)
{
  int i,j,k, inz,jnz,jstart,jstartIM, jend, nsqr, firstIM, firstj,firstjIM,
    lastj,lastcol,lastcolIM, xjc0IM,rsubjc0IM;
  double *fworkIM,*uxj, *uxjIM;
  const double *uIM, *uwork, *uworkIM;
  double ukj, ukjIM;
  char nzcol;

  nsqr = SQR(n);
  fworkIM = fwork + nsqr;
  uIM = u + nsqr;
/* ------------------------------------------------------------
   Let firstIM be 1st usable index in imaginary block,
     xjc0IM (rsubjc0IM) location of 1st imaginary nonzero in x (rsubir).
   ------------------------------------------------------------ */
  firstIM = first + nsqr;
  xjc0IM = x.jc[0];                               /* start imag(x) */
  intbsearch(&xjc0IM, x.ir, x.jc[1], firstIM);
  rsubjc0IM = 0;                                  /* start imag(r) */
  intbsearch(&rsubjc0IM, rsubir, rsubnnz, firstIM);
  /* ------------------------------------------------------------
     STRATEGY: compute "fwork = u*TRIL(x)" and then
     "uxu = (fwork*u) + (u*fwork')".  Typically, x is highly sparse,
     like x=(e_i*e_j'+e_je_i' ), for which fwork has only 1 nonzero
     column, whereas u*x has 2 nonzero columns. Furthermore, the
     contents of x is already TRIL(X) anyway.
     WARNING: we will get the rows of U by reading its columns (which
     is more efficient), and hence the sign of IM U is always reversed
     (because U is Hermitian).
     ------------------------------------------------------------ */
  jnz = 0;
  uxj = fwork;
  uxjIM = fworkIM;
  if(x.jc[0] < xjc0IM)
    lastcol = (x.ir[xjc0IM-1] - first) / n;
  else
    lastcol = 0;
  if(xjc0IM < x.jc[1])
    lastcolIM = (x.ir[x.jc[1]-1] - firstIM) / n;
  else
    lastcolIM = 0;
  jstart = x.jc[0];
  jstartIM = xjc0IM;
  /* ------------------------------------------------------------
     >>> REAL PART COLUMNS OF X: <<<
     For each column j, compute uxj(:) = U' * TRIL(RE X)(:j),
     where TRIL(x) = 2*tril(x,-1) + diag(x).  Hence, 2*x=TRIL(x)+TRIL(x)'.
     ------------------------------------------------------------ */
  firstj = first; firstjIM = firstIM;
  for(j = 0; j < n; uxj += n, uxjIM += n, j++){
    nzcol = 0;
    /* ------------------------------------------------------------
       RE x(:,j) CONTRIBUTION:
       ------------------------------------------------------------ */
    lastj = firstj + n;
    if(j < lastcol){
      for(jend = jstart; x.ir[jend] < lastj; jend++);
    }
    else
      jend = xjc0IM;
    if( jstart < jend ){
      nzcol = 1;            /* Flag that this is gonna be nonzero col */
      /* ------------------------------------------------------------
         Compute fwork(:,j) = U * RE x(:,j), using that U is Hermitian
       We have uwork[firstj:firstj+n] = uk[0:n] and firstj<=x.ir[inz]<firstj+n
         ------------------------------------------------------------ */
      uwork = u - firstj;
      uworkIM = uIM - firstj;
      for(k=0; k < n; uwork += n, uworkIM += n, k++){
        for(ukj = 0.0, ukjIM = 0.0, inz = jstart; inz < jend; inz ++){
          i = x.ir[inz];
          ukj += uwork[i] * x.pr[inz];
          ukjIM += uworkIM[i] * x.pr[inz];
        }
        uxj[k] = ukj;              /* uxj(:)   =  REu*REx(:,j) */
        uxjIM[k] = -ukjIM;         /* uxjIM(:) = -IMu*REx(:,j) */
      }
      jstart = jend;
    }
    firstj = lastj;          /* firstj for nextcolumn, Real part */
    /* ------------------------------------------------------------
       IM x(:,j) CONTRIBUTION:
       ------------------------------------------------------------ */
    lastj = firstjIM + n;    /* "lastj" ("jend") temp storage for "lastjIM" ("jendIM")*/
    if(j < lastcolIM){
      for(jend = jstartIM; x.ir[jend] < lastj; jend++);
    }
    else
      jend = x.jc[1];
    if( jstartIM < jend ){
      if(!nzcol){          /* If RE TRIL(X)(:,j)==0 then initialize uxj */
        nzcol = 1;         /* and flag that this is gonna be nonzero col */
        fzeros(uxj,n); fzeros(uxjIM,n);
      }
/* ------------------------------------------------------------
   Compute fwork(:,j) = U * IM x(:,j), using that U is Hermitian
   We have uwork[firstjIM:firstjIM+n] = uk[0:n]
   and firstjIM<=x.ir[inz]<firstjIM+n
   ------------------------------------------------------------ */
      uwork = u - firstjIM;
      uworkIM = uIM - firstjIM;
      for(k = 0; k < n; uwork += n, uworkIM += n, k++){
        for(ukj = 0.0, ukjIM = 0.0, inz = jstartIM; inz < jend; inz ++){
          i = x.ir[inz];
          ukj += uwork[i] * x.pr[inz];
          ukjIM += uworkIM[i] * x.pr[inz];
        }
        uxj[k] += ukjIM;              /* uxj(:) += IMu*IMx(:,j) */
        uxjIM[k] += ukj;              /* uxjIM(:) += REu*IMx(:,j) */
      }
      jstartIM = jend;      /* jend is temp storage for "jendIM" */
    }
    firstjIM = lastj;      /* lastj is temp storage for "lastjIM" */
    if(nzcol)
      iwork[jnz++] = j*n;    /* done with this nonzero column of TRIL(x) */
  }
  /* -------------------------------------------------------------
     Now jnz = |iwork|.
     For each nonzero (i,j) in rsub, compute rij.
     Let k be the last column in RE r; first compute RE R(:,0:k-1)
     ------------------------------------------------------------- */
  if(rsubjc0IM > 0)
    k = (rsubir[rsubjc0IM-1] - first) / n;     /* k=last_column_RE */
  else
    k = 0;
  inz = 0;
  firstj = first;
/* ------------------------------------------------------------
   2 * U*X*U = (U*TRIL(X))*U + U*(TRIL(X)'*U) 
   = (fwork     *U) +(U*fwork')
   ------------------------------------------------------------ */
  for(j = 0; j < k; j++){
/* ------------------------------------------------------------
   real diagonal: r(j,j) = fwork(j,:)*u(j,:)'
   ------------------------------------------------------------ */
    if( (i = rsubir[inz]) ==  firstj + j){
      r[i] = selrealdot(fwork + j, u+j, iwork, jnz)
        + selrealdot(fworkIM + j, uIM+j, iwork, jnz);
      i = rsubir[++inz];
    }
/* ------------------------------------------------------------
   i>j: RE r(i,j) = {fwork(j,:)*u(i,:)' + u(j,:) * fwork(i,:)'}/2
   ------------------------------------------------------------ */
    lastj = firstj + n;
    for(; i < lastj; i = rsubir[++inz]){
      i -= firstj;
      mxAssert(i > j && i < n,"");
      r[firstj + i] = (selrealdot(u+i, fwork + j, iwork, jnz)
                       + selrealdot(fwork + i, u+j, iwork, jnz)
                       + selrealdot(uIM+i, fworkIM + j, iwork, jnz)
                       + selrealdot(fworkIM + i, uIM+j, iwork, jnz)) / 2;
    }
    firstj = lastj;     /* point to next column */
  }
/* -------------------------------------------------------------
   Compute R(:,k)
   ------------------------------------------------------------- */
  if( (i = rsubir[inz]) == firstj + k){
    r[i] = selrealdot(fwork + k, u+k, iwork, jnz)
      + selrealdot(fworkIM + k, uIM+k, iwork, jnz);
    ++inz;
  }
/* ------------------------------------------------------------
   i>k: RE r(i,k) = {fwork(k,:)*u(i,:)' + u(k,:) * fwork(i,:)'}/2
   ------------------------------------------------------------ */
  for(; inz < rsubjc0IM; inz++){
    i = rsubir[inz];
    i -= firstj;
    r[firstj + i] = (selrealdot(u+i, fwork + k, iwork, jnz)
                     + selrealdot(fwork + i, u+k, iwork, jnz)
                     + selrealdot(uIM+i, fworkIM + k, iwork, jnz)
                     + selrealdot(fworkIM + i, uIM+k, iwork, jnz)) / 2;
  }
/* ------------------------------------------------------------
   Let k be the last column in IM r; compute IM R(:,0:k-1)
   ------------------------------------------------------------- */
  if(rsubjc0IM < rsubnnz)
    k = (rsubir[rsubnnz-1] - firstIM) / n;   /* k = last_column_IM */
  else
    k = 0;
  inz = rsubjc0IM;
  firstjIM = firstIM;
  for(j = 0; j < k; j++){
/* ------------------------------------------------------------
   imaginary -- only strict below diagonal needed:
   i > j: IM r(i,j) = {fwork(j,:)*u(i,:)' + u(j,:) * fwork(i,:)'} / 2
   ------------------------------------------------------------ */
    lastj = firstjIM + n;
    for(i = rsubir[inz]; i < lastj; i = rsubir[++inz]){
      i -= firstjIM;
      mxAssert(i > j && i < n,"");
      r[firstjIM + i] = (selrealdot(uIM+i, fwork + j, iwork, jnz)
                         - selrealdot(fwork + i, uIM+j, iwork, jnz)
                         - selrealdot(u+i, fworkIM + j, iwork, jnz)
                         + selrealdot(fworkIM + i, u+j, iwork, jnz)) / 2;
    }
    firstjIM = lastj;     /* point to next column */
  }
/* -------------------------------------------------------------
   Compute IM R(:,k)
   ------------------------------------------------------------- */
/* ------------------------------------------------------------
   i>k: IM r(i,k) = {fwork(k,:)*u(i,:)' + u(k,:) * fwork(i,:)'}/2
   ------------------------------------------------------------ */
  for(; inz < rsubnnz; inz++){
    i = rsubir[inz]-firstjIM;
    mxAssert(i > k && i < n,"");
    r[firstjIM + i] = (selrealdot(uIM+i, fwork + k, iwork, jnz)
                       - selrealdot(fwork + i, uIM+k, iwork, jnz)
                       - selrealdot(u+i, fworkIM + k, iwork, jnz)
                       + selrealdot(fworkIM + i, u+k, iwork, jnz)) / 2;
  }
}

/* ============================================================
   MAIN ROUTINE: spscaleK
   ============================================================ */
/* ************************************************************
   PROCEDURE spscaleK -- z = D(x)y with y sparse, having nonzeros
      only in blocks yblk(:).
      (NOTATION below: 'N' means lenfull)
   INPUT
     x       - FULL x >_K 0, the scaling vector. Only its Lorentz part is used.
     xsqr    - x(1:K.l).^2
     detx,uxsqr - if [qdetx,ux] = factorK(x,K), then detx = qdetx.^2,
              uxsqr = invchol(ux).
     y       - *input* vector, stored as sparse N x yblknnz matrix.
             CAUTION: the row-indices are for vec(y), i.e. [0:N],
             yet, the columns point to the nonzero-block-positions in y.
     yblk    - lists the block numbers where y has nonzeros
     yblknnz - length(yblk), number of nonzero blocks in y.
     blkstart - blkstart[k] yields 1st subscript to be used in block k
                when vectorizing (as in  y.ir and dzir).
     dzir     - nonzero pattern of PSD-part of *output* vector z,
               as SPARSE N x 1; dzir tells us which PSD entries of D(x)y
               we must compute (since we may not use them all).
     dzjc     - Induces block partition of dzir, viz. dzjc[k] is start of
              kth PSD block listed in yblk. Length #PSD-blocks(yblk) + 1.
     cK      - the cone structure, describing the cone K.
     blkNL   - length nblk array, blkNL = [K.l, K.q, K.s], listing the
              orders of the cone primitives.
   OUTPUT
     z     - FULL N-vector z = D(x)*y. For PSD blocks, it has nonzeros only
             at z(dzir(:)), and 0.0 elsewhere.
     xdotyIr, Pr - sparse vector, nnz = number of nonzero Lorentz blocks in y.
             On output,
                xdoty.pr(knz) = x_k'*y_k,    k = xdoty.ir[knz]
             for each nonzero Lorentz block k.
   WORK
     fwork - max(K.s)^2 working array of doubles.
     iwork - max(K.s) working array of ints.
   RETURNS
     nnz(xdoty)
   ************************************************************ */
int spscaleK(double *z, const int *dzjc, const int *dzir,
             jcir y, const int *yblkir, const int yblknnz,
             const int *blkstart,
             const double *x, const double *xsqr, const double *detx,
             const double *uxsqr,
             const coneK cK, const int *blkNL,
             double *fwork, int *iwork,
             int *xdotyIr, double *xdotyPr)
{
  int k,blknz,lastblk,lornz, inz,isub,lorNp1, dznnzk;
  double xdotyk, alphak;

  blknz = 0;
  lastblk = yblkir[yblknnz-1];
  lornz = 0;             /* location for next Lorentz x'*y. */
/* ------------------------------------------------------------
   LP-PART: z = D(x^2)y = xsqr.*y    (x full, y sparse)
   ------------------------------------------------------------ */
  if((k = yblkir[blknz]) == 0){
    for(inz = y.jc[0]; inz < y.jc[1]; inz++){
      isub = y.ir[inz];
      z[isub] = xsqr[isub] * y.pr[inz];
    }
    blknz++;               /* to next block */
    y.jc++;
  }
/* ------------------------------------------------------------
   LORENTZ PART:  z = D(x^2)y = (x'*y) * x + det(x) * [-y(1); y(2:nk)]
   ------------------------------------------------------------ */
  if(lastblk <= cK.lorN)              /* Lorentz blocks until end of column */
    for(; blknz < yblknnz; blknz++){
      k = yblkir[blknz];
      xdotyPr[lornz] = spqlmul(z,x,detx[k-1], blkstart[k],blkNL[k],y);
      xdotyIr[lornz++] = k-1;
      y.jc++;
    }
  else{                              /* Proceed until 1st PSD block */
    for(k = yblkir[blknz]; k <= cK.lorN; k = yblkir[++blknz]){
      xdotyPr[lornz] = spqlmul(z,x,detx[k-1], blkstart[k],blkNL[k],y);
      xdotyIr[lornz++] = k-1;
      y.jc++;
    }
/* ------------------------------------------------------------
   Update uxsqr, dzjc so that for k >= lorN + 1:
   uxsqr_k starts at uxsqr+blkstart[k],  and DZ_k at dzjc[k]
   ------------------------------------------------------------ */
    lorNp1 = cK.lorN + 1;
    uxsqr -= blkstart[lorNp1];        /* Now, blkstart[k] is correct index */
    dzjc -= blknz;                    /* Now dzjc[blknz] is correct */
/* ------------------------------------------------------------
   REAL SYM PSD PART: z = D(x)y = UXsqr * Y * UXsqr.
   ------------------------------------------------------------ */
    if(lastblk <= cK.lorN + cK.rsdpN)          /* real PSD blocks until end */
      for(; blknz < yblknnz; blknz++){
        k = yblkir[blknz];
        dznnzk = dzjc[blknz];                     /* find end of dzir-block */
        intbsearch(&dznnzk, dzir, dzjc[blknz+1], blkstart[k+1]);
        spsymuxu(z, uxsqr + blkstart[k], y, blkstart[k], blkNL[k],
                dzir+dzjc[blknz],dznnzk - dzjc[blknz], fwork, iwork);
        y.jc++;
      }
    else{                              /* Proceed until 1st HermPSD block */
      for(k = yblkir[blknz]; k <= cK.lorN + cK.rsdpN; k = yblkir[++blknz]){
        dznnzk = dzjc[blknz];                     /* find end of dzir-block */
        intbsearch(&dznnzk, dzir, dzjc[blknz+1], blkstart[k+1]);
        spsymuxu(z, uxsqr + blkstart[k], y, blkstart[k], blkNL[k],
                dzir+dzjc[blknz],dznnzk - dzjc[blknz], fwork, iwork);
        y.jc++;
      }
/* ------------------------------------------------------------
   COMPLEX HERMITIAN PSD PART: Actually the same, except now we call
   the function "sphermuxu" to take the imaginary parts into account.
   ------------------------------------------------------------ */
      for(; blknz < yblknnz; blknz++){
        k = yblkir[blknz];
        dznnzk = dzjc[blknz];                     /* find end of dzir-block */
        intbsearch(&dznnzk, dzir, dzjc[blknz+1], blkstart[k+1]);
        sphermuxu(z, uxsqr + blkstart[k], y, blkstart[k], blkNL[k],
                  dzir+dzjc[blknz],dznnzk - dzjc[blknz], fwork, iwork);
        y.jc++;
      }
    } /* real + herm psd */
  } /* lorentz + psd */
/* ------------------------------------------------------------
   Return nnz in sparse column xdoty(:), viz. lornz
   ------------------------------------------------------------ */
  return lornz;
}
