% SPARCHOL Fast block sparse Cholesky factorization.
%    L = sparchol(L,X,pars) uses only tril(X); triu(X) is assumed
%    to be its transpose. The sparse Cholesky factor will be
%    placed in the fields L.L, L.d; the symbolic factorization fields
%    remain unchanged.
%    Sparchol also writes the fields L.U12, L.U22, L.ph2nodes, L.ph2perm,
%    which contains a dense pivoted Cholesky factorization on nodes that
%    were unstable in Phase I.
%    Finally, L.dep contains a list of nodes that are linearly dependent
%    on their predecessors.
%    On input, L should be the symbolic factorization structure of X,
%    as created by SYMBCHOL.
%    Performs LDL' factorization of X(L.perm,L.perm).
%
%   There are important differences with CHOL(X(L.perm,L.perm))':
%
%   -  SPARCHOL uses the supernodal partition L.XSUPER,
%    to use dense linear algebra on dense subblocks.
%    This explains the performance benefit of SPARCHOL over CHOL.
%
%   -  SPARCHOL never fails. If during factorization, L.d(i) underflows
%    (as controled by pars.reltol), then - for scaled
%    least squares - it's rebuilt from the original (unscaled) data. If a
%    row happens to be linearly dependent on its predecessors, then it's
%    discarded and listed in L.dep.
%
%   - To solve "X*y = b", use
%    >> L = sparchol(symbchol(X),X);
%    >> L.d(L.dep) = inf;
%    >> y = sparbwslv(L, sparfwslv(L,b) ./ L.d);
%
% SEE ALSO symbchol, sparfwslv, sparbwslv, [symbfact, symmmd, chol].
function L = sparchol(L,X,pars, adotd,A,scld,sclud,K)

 %  
 %   This file is part of SeDuMi 1.03BETA
 %   Copyright (C) 1999 Jos F. Sturm
 %   Dept. Quantitative Economics, Maastricht University, the Netherlands.
 %   Affiliations up to SeDuMi 1.02 (AUG1998):
 %     CRL, McMaster University, Canada.
 %     Supported by the Netherlands Organization for Scientific Research (NWO).
 % 
 %   This program is free software; you can redistribute it and/or modify
 %   it under the terms of the GNU General Public License as published by
 %   the Free Software Foundation; either version 2 of the License, or
 %   (at your option) any later version.
 % 
 %   This program is distributed in the hope that it will be useful,
 %   but WITHOUT ANY WARRANTY; without even the implied warranty of
 %   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 %   GNU General Public License for more details.
 % 
 %   You should have received a copy of the GNU General Public License
 %   along with this program; if not, write to the Free Software
 %   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 %

% ------------------------------------------------------------
% INITIALIZE: check the numerical tolerance parameters in "pars",
%   and supply with defaults if necessary.
% ------------------------------------------------------------
 if nargin < 3
   pars.deptol = 1e-25;              % Threshold for discarding nodes
   pars.maxuden = 10;                % Stability of product-form factors
   pars.numlvl = 0;                % No numerical reordering on sparse chol.
   pars.ph1tol = 1e-6;
   pars.maxu = 10;
   pars.reltol = 1e-12;
 else
   if ~isfield(pars,'deptol')
     pars.deptol = 1e-25;              % Threshold for discarding nodes
   end
   if ~isfield(pars,'maxuden')
     pars.maxuden = 10;                % Stability of product-form factors
   end
   if ~isfield(pars,'numlvl')
     pars.numlvl = 0;                % 0=No numerical reordering
   elseif pars.numlvl > 0
     if ~isfield(pars,'ph1tol')
       pars.ph1tol = 1e-6;             % When to check stability of pivot
     end
     if ~isfield(pars,'maxu')
       pars.maxu = 10;                 % What is considered stable
     end
     if ~isfield(pars,'reltol')
       pars.reltol = 1e-12;             % Currently NOT used.
     end
   end
 end
% ----------------------------------------
% Numerical block sparse Cholesky of X
% ----------------------------------------
 [L.L,L.d,L.U12,L22,L.ph2nodes,L.dep] = blkchol(L,X,pars);
 L.d(L.dep) = 0.0;
% ------------------------------------------------------------
% For numerical reasons, "blkchol" may postpone instable pivots,
% which are then listed in L.ph2nodes.
% FACTORIZE these Phase II nodes using dense diagonal pivoting:
% ------------------------------------------------------------
 [L.U22,ph2d,L.ph2perm,dep2] = cholp(L22,pars.deptol);
 if length(L.ph2nodes) > 0
   L.d(L.ph2nodes) = ph2d;
   if length(dep2) > 0
     L.dep = sort([L.dep(:,1); L.ph2nodes(dep2)]);  % NB: merging 'd be faster'
   end
 end
% ------------------------------------------------------------
% Factor dense columns
% ------------------------------------------------------------
 if nargin > 3
   if nargin < 8
     error('Format: L = sparchol(L,X,pars, adotd,A,scld,sclud,K)');
   end
   [L.den,L.d,L.dep] = deninfac(L,A,adotd,scld,sclud.qdet,pars.deptol,...
           pars.maxuden);
 end
