%                                          [x,y,info] = sedumi(A,b,c,K,pars)
%
% SEDUMI  Self-Dual-Minimization/ Optimization over self-dual homogeneous
%         cones.
%
% >  X = SEDUMI(A,b,c) yields an optimal solution to the linear program
%      MINIMIZE c'*x SUCH THAT A*x = b, x >= 0
%      x is a vector of decision variables.
%      If size(A,2)==length(b), then it solves the linear program
%      MINIMIZE c'*x SUCH THAT A'*x = b, x >= 0
%
% >  [X,Y,INFO] = SEDUMI(A,b,c) also yields a vector of dual multipliers Y,
%      and a structure INFO, with the fields INFO.pinf, INFO.dinf and
%      INFO.numerr.
%
%    (1) INFO.pinf=info.dinf=0: x is an optimal solution (as above)
%      and y certifies optimality, viz.\ b'*y = c'*x and c - A'*y >= 0.
%      Stated otherwise, y is an optimal solution to
%      MAXIMIZE b'*y SUCH THAT c-A'*y >= 0.
%      If size(A,2)==length(b), then y solves the linear program
%      MAXIMIZE b'*y SUCH THAT c-A*y >= 0.
%
%    (2) INFO.pinf=1: there cannot be x>=0 with A*x=b, and this is certified
%      by y, viz. b'*y > 0 and A'*y <= 0. Thus y is a Farkas solution.
%
%    (3) INFO.dinf=1: there cannot be y such that c-A'*y >= 0, and this is
%      certified by x, viz. c'*x <0, A*x = 0, x >= 0. Thus x is a Farkas
%      solution.
%
%    (I)   INFO.numerr = 0: desired accuracy achieved (see PARS.eps).  
%    (II)  INFO.numerr = 1: numerical problems warning. Results are accurate
%          merely to the level of PARS.bigeps.
%    (III) INFO.numerr = 2: complete failure due to numerical problems.
%
%    INFO.feasratio is the final value of the feasibility indicator. This
%    indicator converges to 1 for problems with a complementary solution, and
%    to -1 for strongly infeasible problems. If feasratio in somewhere in
%    between, the problem may be nasty (e.g. the optimum is not attained),
%    if the problem is NOT purely linear (see below).
%
% >  [X,Y,INFO] = SEDUMI(A,b,0) solves the feasibility problem
%    FIND x>=0 such that A*x = b
%
% >  [X,Y,INFO] = SEDUMI(A,0,c) solves the feasibility problem
%    FIND y such that A'*y <= c
%
% >  [X,Y,INFO] = SEDUMI(A,b,c,K) instead of the constraint "x>=0", this
%      restricts x to a self-dual homogeneous cone that you describe in the
%      structure K. Up to 4 fields can be used, called K.l, K.q, K.r and K.s,
%      for Linear, Quadratic, Rotated quadratic and Semi-definite.
%
%    (1) K.l is the number of NONNEGATIVE components. E.g. if K.l=10 then
%      x(1:10) >=0.  These are ALWAYS the first components in x.
%
%    (2) K.q lists the dimensions of LORENTZ (quadratic, second-order cone)
%      constraints. E.g. if K.l=10 and K.q = [3 7] then
%          x(11) >= norm(x(12:13)),
%          x(14) >= norm(x(15:20)).
%      These components ALWAYS immediately follow the K.l nonnegative ones.
%      If the entries in A and/or c are COMPLEX, then the x-components in
%      "norm(x(#,#))" take complex-values, whenever that is beneficial.
%       Use K.ycomplex to impose constraints on the imaginary part of A*x.
%
%    (2) K.r lists the dimensions of Rotated LORENTZ
%      constraints. E.g. if K.l=10, K.q = [3 7] and K.r = [4 6], then
%          x(21)x(22) >= norm(x(23:24)),
%          x(25)x(26) >= norm(x(27:30)).
%      These components ALWAYS immediately follow the K.q ones.
%      Just as for the K.q-variables, the variables in "norm(x(#,#))" are
%      allowed to be complex, if you provide complex data. Use K.ycomplex
%      to impose constraints on the imaginary part of A*x.
%
%    (3) K.s lists the dimensions of POSITIVE SEMI-DEFINITE (PSD) constraints.
%      E.g. if K.l=10, K.q = [3 7] and K.s = [4 3], then
%          mat( x(21:36),4 ) is PSD,
%          mat( x(37:45),3 ) is PSD.
%      These components are ALWAYS the last entries in x.
%      If corresponding entries in A and/or c are COMPLEX or link with complex
%      values elsewhere in the model, then these x-components are HERMITIAN
%      PSD. Use K.ycomplex to impose constraints on the imaginary part of A*x.
%
%    The dual multipliers y have analogous meaning as in the "x>=0" case,
%    except that instead of "c-A'*y>=0" resp. "-A'*y>=0", one should read that
%    c-A'*y resp. -A'*y are in the cone that is described by K.l, K.q and K.s.
%    In the above example, if z = c-A'*y and mat(z(21:36),4) is not symmetric/
%    Hermitian, then positive semi-definiteness reflects the symmetric/
%    Hermitian parts, i.e. Z + Z' is PSD.
%
%    If the model contains COMPLEX data, then you may provide a list
%    K.ycomplex, with the following meaning:
%      y(i) is complex if  imag(b(i)) ~= 0 or ismember(i,K.ycomplex) 
%      y(i) is real otherwise
%    The equality constraints in the primal are then as follows:
%          A(i,:)*x = b(i)      if imag(b(i)) ~= 0 or ismember(i,K.ycomplex) 
%          real(A(i,:)*x) = b(i)  otherwise. 
%    Thus, equality constraints on both the real and imaginary part
%    of A(i,:)*x should be listed in the field K.ycomplex.
%    CAUTION: any nonzero value in imag(b(i)) imposes a constraint on
%    imag(A(i,:)*x). Be sure to replace b(i) by real(b(i)) if b(i) is
%    supposed to be real, but may be complex due to rounding errors.
%
%    You may use EIGK(x,K) and EIGK(c-A'*y,K) to check that x and c-A'*y
%    are in the cone K.
%
% >  [X,Y,INFO] = SEDUMI(A,b,c,K,pars) allows you to override the default
%      parameter settings, using fields in the structure `pars'.
%
%    (1) pars.fid     By default, fid=1. If fid=0, then SeDuMi runs quietly,
%      i.e. no screen output. If fid=FILENAME, then output is written to
%      a file called FILENAME.
%
%    (2) pars.alg     By default, alg=1 for nonlinear cones, and alg=2 if
%      the problem is completely linear. If alg=0, then a first-order wide
%      region algorithm is used, not recommended. If alg=1, then SeDuMi uses
%      the centering-predictor-corrector algorithm with v-linearization.
%      If alg=2, then xz-linearization is used in the corrector, similar
%      to Mehrotra's algorithm. The wide-region centering-predictor-corrector
%      algorithm was proposed in Chapter 7 of
%        J.F. Sturm, Primal-Dual Interior Point Approach to Semidefinite Pro-
%        gramming, TIR 156, Thesis Publishers Amsterdam, 1997.
%
%    (3) pars.theta, pars.beta   By default, theta=0.25 and beta=0.5. These
%      are the wide region and neighborhood parameters. Valid choices are
%      0 < theta <= 1 and 0 < beta < 1.
%
%    (4) pars.vplot  If this field is 1, then SeDuMi produces a fancy
%      v-plot, for research purposes. Default: vplot = 0.
%
%    (5) pars.eps     The desired accuracy.
%
%    (6) pars.bigeps  In case the desired accuracy canoot be achieved, the
%     solution is tagged as info.numerr=1 if it is accurate to pars.bigeps,
%     otherwise it yields info.numerr=2.
%
%    (7) pars.maxiter Maximum number of iterations, before termination.
%
%    (8) pars.stopat  SeDuMi enters debugging mode at this iteration number
%
%    (9) pars.numtol  Tolerance to observed numerical problems. SeDuMi refines
%     the search directions if this accuracy is not achieved.
%
%    (10) pars.chol   Various parameters for controling the Cholesky solve.
%     Subfields of the structure pars.chol are:
%    (a) chol.numlvl: level of numerical control. You may set chol.numlvl=1
%      if SeDuMi fails numerically otherwise.
%    (b) chol.deptol: controls when a node is discarded from the system
%    (c) chol.ph1tol: Relative tolerance to determine which nodes have to
%      be checked on numerical stability
%    (d) chol.maxu:   Produces LDL' with max(abs(L)) <= chol.maxu.
%    (e) chol.maxuden: Same for L-factors in product form (for dense columns)
%
% Bug reports can be send to j.sturm@ke.unimaas.nl
%
% SEE ALSO mat, vec, eyeK AND eigK.

function [xsol,ysol,info] = sedumi(A,b,c,K,pars)

 %  
 %   This file is part of SeDuMi 1.03 (AUG1999)
 %   Copyright (C) 1999 Jos F. Sturm
 %   Dept. Quantitative Economics, Maastricht University, the Netherlands.
 %   Affiliations up to SeDuMi 1.02 (AUG1998):
 %     CRL, McMaster University, Canada.
 %     Supported by the Netherlands Organization for Scientific Research (NWO).
 % 
 %   This program is free software; you can redistribute it and/or modify
 %   it under the terms of the GNU General Public License as published by
 %   the Free Software Foundation; either version 2 of the License, or
 %   (at your option) any later version.
 % 
 %   This program is distributed in the hope that it will be useful,
 %   but WITHOUT ANY WARRANTY; without even the implied warranty of
 %   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 %   GNU General Public License for more details.
 % 
 %   You should have received a copy of the GNU General Public License
 %   along with this program; if not, write to the Free Software
 %   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 %

% ----------------------------------------
% Check input
% ----------------------------------------
 if (nargin < 5) 
   pars.fid = 1;
 end
 if (nargin < 4)
   if (nargin < 3)
     if (nargin < 2)
       error('Should have at least A,b,c arguments')
     end
     c = 0;           % given (A,b): default to feasibility problem
   end
   if (isstruct(c))
     K = c; c = 0;        % given (A,b,K): cone feasibility problem
   else
     K.l = max(size(A));    % given (A,b,c): default to LP problem.
   end
 end
% ----------------------------------------
% Symbolic analysis, Initialize solution, and check parameters.
% ----------------------------------------
 [A,b,c,K,prep] = pretransfo(A,b,c,K);
 [A,c,K,L,d,ud, v,vfrm, x,y,y0, R] = sdinit(A,b,c,K,pars);
 z = x;
 R.norm = max([max(abs(R.b)) max(abs(R.c)) abs(R.sd)]);
 n = K.l + 2*length(K.q) + sum(K.s);                         % order of K
 lponly = (K.l == length(c));
 pars = checkpars(pars,lponly);
% ----------------------------------------
% Print welcome and statistics of cone-problem
% ----------------------------------------
 fprintf(pars.fid,'SeDuMi 1.03 by Jos F. Sturm, 1998, 1999.\n');
 if pars.alg == 0
   fprintf(pars.fid,'Alg = 0: No corrector, ');
 elseif pars.alg == 1
   fprintf(pars.fid,'Alg = 1: v-corrector, ');
 else
   fprintf(pars.fid,'Alg = 2: xz-corrector, ');
 end
 fprintf(pars.fid,'theta = %5.3f, beta = %5.3f\n',pars.theta,pars.beta);
 fprintf(pars.fid,'eqs m = %g, order n = %g, dim = %g, blocks = %g\n',...
       length(b),n,length(x),1 + length(K.q) + length(K.s));
 if length(A.dense.cols) > 0
   fprintf(pars.fid,'Handling %d + %d dense columns.\n',...
    length(A.dense.cols),length(A.dense.qs));
 end
 fprintf(pars.fid,' it :     b*y       gap    delta  rate  t/maxt   feas\n');
 fprintf(pars.fid,'  0 :            %8.2E %5.3f\n',R.b0*y0,0);
% ----------------------------------------
% Initialize iterative statistics
% ----------------------------------------
 STOP = 0;
 numerr = 0.0;                  % Estimates the need to recompute residuals
 newresid = 0;                  % -1 = Needs updating, +1 = Just updated
 iter = 0;
 if pars.vplot == 1
   vlist = [vfrm.lab];
   ratelist = [];
 end
 delta = 0.0; vTAR = vfrm.lab; rate = 1.0; by = b'*y;   %'
 x0 = d(1)*v(1); z0 = v(1)/d(1);
 feasratio = 0.0;
 cputime0 = cputime;
% ----------------------------------------
% Main predictor-corrector loop
% ----------------------------------------
 while ~STOP
   iter = iter+1;
   if iter == pars.stopat
     keyboard
   end
% ----------------------------------------
% RESIDUALS (compute from 1st iterate where numerical problems arise)
% ----------------------------------------
   if newresid < 0
      x = scaleK(d,ud, v,K,1);
      z = invscaleK(d,ud, v,K,0);
      Rold = R;
      R.b = full(Amul(A,x,0)            - x(1)*b) / y0;
      if iter > 1
        cumerb(iter-1) = y0 * norm(R.b - Rold.b);            % for debugging
      end
      R.c = (vecsym(full(Amul(A,y,1)),K) + z  - x(1)*c) / y0;   R.c(1) = 0.0;
      R.sd = (z(1) + c'*x - by) / y0;
      R.b0 = R.sd*x(1) + R.c'*x - R.b'*y; %'
      newresid = 1;
      fprintf(pars.fid,'*')
   else
      newresid = 0;
      fprintf(pars.fid,' ')
   end
% ----------------------------------------
% FACTORIZATION of normal equations
% ----------------------------------------
   [L,Lsd,Rscl] = factorneq(L,d,ud,A,b,c,K,R, y0,v,by,pars);
% ----------------------------------------
% INITIAL CENTERING
% ----------------------------------------
   y0Old = y0;
   if delta > 0.0           % initial centering needed ?
     pv = 2*(vTAR-vfrm.lab);
     [dx, dy, dz, dy0, deltaz] = sddir(L,Lsd,pv, Rscl, ...
          d,ud,v,vfrm,A,c,R, K,pars,y,y0);
% ------------------------------------------------------------
% Refine initial centering step if needed
% ------------------------------------------------------------
     [dx,dy,dz,deltax,deltaz,Lsd,Rscl] = refine(dy,deltaz,dy0,dx,dz,d,ud,...
                     A,b,c,K,L,Lsd,R,y,pars, x0,y0,v,by,Rscl);
% ------------------------------------------------------------
% Take initial centering step
% ------------------------------------------------------------
     xc = v + dx;              %scaled
     zc = v + dz;
     yc = y + dy;
     y0c = y0+dy0;
     [uxc.qdet,uxc.u,xispos] = factorK(xc,K);
     [uzc.qdet,uzc.u,zispos] = factorK(zc,K);
     if (~xispos) | (~zispos)
       STOP = -1; iter = iter - 1;   % Reject and terminate
       fprintf(pars.fid,'Run into numerical problems during initial centering step.\n');
     end
     pv = -vTAR;
   else
     xc = v;                    % No initial centering needed.
     [uxc.qdet,uxc.u,xispos] = factorK(xc,K);
     zc = v; uzc = uxc;          %zscl=xscl=v
     yc = y;
     y0c = y0;
     pv = [];              % Means pv = -v, the predictor direction.
   end
% ----------------------------------------
% PREDICTOR
% ----------------------------------------
   if STOP ~= -1
     [dx, dy, dz, dy0, deltaz] = sddir(L,Lsd,pv, Rscl, ...
          d,ud,v,vfrm,A,c,R, K,pars,y,y0, b);
     dxmdz = dx-dz;
     newfeasratio = dxmdz(1) / v(1);         % (deltax0/x0) - (deltaz0/z0)
     if (pars.alg ~= 0)
       gd1 = invjmulK(vTAR,vfrm,dxmdz, K);
       maxt1 = maxstep(dx,dz, xc,uxc,zc,uzc,K);
% ----------------------------------------
% 2ND ORDER CORRECTOR
% alg == 1 :  2nd order expansion of v = sqrt(eig(D(x)z)), like Sturm-Zhang.
% alg == 2 :  2nd order expansion of vsqr = eig(D(x)z), like Mehrotra.
% ----------------------------------------
       if pars.alg == 1                 % alg 1: v-expansion
         tTAR = 1-(1-maxt1);
         pv = (tTAR*tTAR)*jmulK(gd1,dxmdz,K);
         pv2 = 2*tTAR*(1-tTAR)*((sum(vTAR)/n)*ones(n,1) - vTAR) -(2*tTAR)*vTAR;
         pv = pv + frameit(pv2, vfrm.q, vfrm.s, K);
       else                             % alg 2: v^2, like Mehrotra.
         tTAR = 1-(1-maxt1)^3;
         pv = (tTAR / 4) * jmulK(gd1,dxmdz,K);
         pv2 = ((1-tTAR)*tTAR*R.b0*y0/n)./vTAR - (1+tTAR/4)*vTAR;
         pv = pv + frameit(pv2, vfrm.q, vfrm.s, K);
       end
       [dx, dy, dz, dy0, deltaz] = sddir(L,Lsd,pv, Rscl, ...
          d,ud,v,vfrm,A,c,R, K,pars,y,y0);
     end
% ----------------------------------------
% Compute unscaled primal step "deltax" and
% refine if needed
% ----------------------------------------
     [dx,dy,dz,deltax,deltaz,Lsd,Rscl,erbi,failure] = refine(dy,deltaz,dy0,dx,dz,d,...
                     ud,A,b,c,K,L,Lsd,R,y,pars, x0,y0,v,by,Rscl);
     itnumerr = abs(dx'*dz) / (R.b0*y0);               %'
% ----------------------------------------
% Estimate numerical error, and progress in residuals y0*[R.b,R.c.R.sd].
% If erbi:= ||A*dx - dx0*b-dy0*Rb|| > y0*||Rb||, then reject this step:
% ==> recompute residuals OR terminate.
% ----------------------------------------
     numerr = numerr + itnumerr;
     if failure
       if newresid <= 0                % Reject, but try again
         newresid = -1;                % force computing new resids
         iter = iter - 1;
       elseif pars.chol.numlvl == 0          % Reject, but try again
         pars.chol.numlvl = 1;           % do numerical reordering in chol
         newresid = -1;
         iter = iter - 1;
       else
         STOP = -1; iter = iter - 1;   % Reject and terminate
         fprintf(pars.fid,'Run into numerical problems.\n');
       end
     else
       if numerr > pars.numtol
          if newresid > 0
             pars.chol.numlvl = 1;           % do numerical reordering in chol
          end
          newresid = -1;
       end
% ----------------------------------------
% Compute boundary step length "maxt".
% ----------------------------------------
       maxt = maxstep(dx,dz, xc,uxc,zc,uzc,K);
% ----------------------------------------
% CREGION NBRHD STEP OF PRED+COR
% ----------------------------------------
       [t,h,alpha,delta,w,uxscl] = widelen(xc,zc,y0c, dx,dz,dy0, maxt,pars,K);
% ----------------------------------------
% Take step (in scaled space)
% ----------------------------------------
       xscl = xc + t*dx;
       y    = yc + t*dy;
       zscl = zc + t*dz;
       y0   = y0c + t*dy0;
       by = full(b'*y);
       x0 = d(1) * xscl(1);
       z0 = zscl(1) / d(1);
       rate = y0 / y0Old;
       if rate < 0.1
         newresid = -1;             % Update residuals if we're going far.
       end
% -------------------------------------------------- 
% Primal-Dual transformation
% --------------------------------------------------
       [d,ud,vfrm] = newtransfo(xscl,uxscl,zscl,w,d,ud,K);
       v = frameit(vfrm.lab,vfrm.q,vfrm.s,K);
       vTAR = (1-alpha) * max(h,vfrm.lab);
       feasratio = newfeasratio;
% ------------------------------------------------------------
% Compute iterative errors (for debugging analysis)
% ------------------------------------------------------------
       if erbi > pars.numtol * y0 * (1+R.norm)
         newresid = -1;                % force computing new resids
       else
         cumerb(iter) = 0;
       end
       erb(iter) = erbi;
% ----------------------------------------
% SHOW ITERATION STATISTICS
% ----------------------------------------
       if(x0 > y0)
         byobj = by / (x0-y0);
       else
         byobj = by / z0;
       end
       rate = y0 / y0Old;
       fprintf(pars.fid,'%2.0f : %10.2E %8.2E %5.3f %6.4f %6.4f  %5.2f\n',...
         iter,byobj,R.b0 * y0,delta,rate,t/maxt,feasratio)
%%% , cumerb(iter));
       if pars.vplot == 1
         vlist = [vlist vfrm.lab/sqrt((R.b0*y0)/n)];
         ratelist = [ratelist rate];
       end
       if rate > 0.99
         STOP = -1;
       end
% ----------------------------------------
% If we get in superlinear region of LP,
% try to guess optimal solution:
% ----------------------------------------
       if abs(1-abs(feasratio)) < 0.1
         if lponly & (rate < 0.05)
           x = scaleK(d,ud, v,K,1);
           [failure,xsol,ysol] = final(d,ud, A,b,c, y0,y,x,z0,dxmdz, ...
              K,L,feasratio, R.b,0,pars.chol,lponly);
           if failure == -1
             STOP = 2;                   % Means that we guessed right !!
             feasratio = sign(feasratio);
           end
         elseif (feasratio < -0.95) & (by > 0) & (R.b0*y0 < 0.5)
           if(min(eigK(-Amul(A,y,1),K)) >= -pars.eps * by)
             STOP = 3;                   % Means Farkas solution found !
           end
         end
       end
     end % !numerr
   end % !failure during initial centering
% --------------------------------------------------
% OPTIMALITY CHECK: stop if y0*resid < eps * (x0+z0).
% For feas. probs, we should divide the residual by x0, otherwise by z0.
% Before stopping, recompute R.norm, since it may have changed due to
% residual updates (the change should be small though).
% --------------------------------------------------
   if STOP==0
     if iter >= pars.maxiter
       STOP = 1;
     end
     if y0 * (1 + R.norm) < pars.eps *(x0+z0)
       R.norm = max([max(abs(R.b)) max(abs(R.c)) abs(R.sd)]);
       if y0 * (1 + R.norm) < pars.eps *(x0 + z0)
         STOP = 1;
       end
     end
   end
 end
% ----------------------------------------
% FINAL TASKS:
% ----------------------------------------
 info.cpusec = cputime-cputime0;
 info.iter = iter;
 info.feasratio = feasratio;
 if STOP ~= -1
   info.numerr = 0;
 end
 if (STOP ~= 2)
   x = scaleK(d,ud, v,K,1);
   [failure,xsol,ysol] = final(d,ud, A,b,c, y0,y,x,z0,dxmdz, ...
          K,L,feasratio, R.b,(STOP == -1),pars.chol,lponly);
 end
 cx = c'*[0;xsol]; by = full(b'*ysol);
 info.pinf = 0; info.dinf = 0;
 info.iter = iter;
% ------------------------------------------------------------
% Interpret the solution as feasible:
% ------------------------------------------------------------
 if(feasratio >= -0.1)
   pinf = norm(Amul(A,[0;xsol],0)-b);
   if cx <= by                % zero or negative duality gap
     sigdig = Inf;
   elseif cx == 0.0           % Dual feasibility problem
     sigdig = -log(-by/(norm(b)*norm(y) +1E-10)) / log(10);
   elseif by == 0.0           % Primal feasibility problem
     sigdig = -log(cx / (norm(c)*norm(x) +1E-10)) / log(10);
   else                       % Optimization problem
     sigdig = -log((cx-by)/(abs(by) +1E-10)) / log(10);
   end
   normxsol = norm(xsol); normysol = norm(ysol);
   fprintf(pars.fid,...
'iter seconds digits       c*x               b*y\n');
   fprintf(pars.fid,'%3d %8.1f %5.1f %- 17.10e %- 17.10e\n',...
     iter,info.cpusec,sigdig,cx,by);
   fprintf(pars.fid,'|Ax-b| = %9.1e, |x|=%9.1e, |y|=%9.1e\n',...
 pinf,normxsol,normysol);
% ------------------------------------------------------------
% Guessed problem to be feasible, but with numerical errors.
% ------------------------------------------------------------
   if STOP == -1
     labz = eigK(vecsym(full(c-Amul(A,ysol,1)),K),K);
     if (10^(-sigdig) < pars.bigeps) & (pinf < pars.bigeps * (1+norm(b))) & ...
        (min(labz) > -pars.bigeps * (norm(labz)+normysol))
       info.numerr = 1;                          % moderate numerical error
     else
       fprintf(pars.fid, 'Failed: no sensible solution/direction found.\n');
       info.numerr = 2;                          % serious numerical error
     end
   end
 else
% --------------------------------------------------
% Infeasible problems:
% --------------------------------------------------
   normxsol = norm(xsol);
   normysol = norm(ysol);
   labz = eigK(vecsym(full(-Amul(A,ysol,1)),K),K);
   minlabz = min(labz);
   if max(-cx,0) + max(by,0) == 0.0
     fprintf(pars.fid, 'Failed: no sensible solution/direction found.\n');
     info.numerr = 2;                          % serious numerical error
   else
     if (cx < -abs(by)) | (cx < -1E-5 * (abs(by) + 1E-5 / (1+normxsol)))
       info.dinf = 1;
       fprintf(pars.fid, ...
	  'Dual infeasible, primal improving direction found.\n');
     end
     if ((by > 0) & (minlabz >= 0)) | ...
         (by >= abs(cx)) | (by > 1E-5 * (abs(cx) + 1E-5 / (1+normysol)))
       info.pinf = 1;
       fprintf(pars.fid, ...
	  'Primal infeasible, dual improving direction found.\n');
     end
   end
   pinf = norm(Amul(A,[0;xsol],0));
   fprintf(pars.fid,'iter seconds  |Ax|      |x|       |y|\n');
   fprintf(pars.fid,'%3d %8.1f %9.1e %9.1e %9.1e\n',...
      iter,info.cpusec,pinf,normxsol,normysol);
% --------------------------------------------------
% Guess infeasible, but stopped due to numerical problems
% --------------------------------------------------
   if STOP == -1
     if (by > 0) & (minlabz >= 0)
       info.numerr = 0;
     else
       if ( (info.dinf == 1) & (pinf < pars.bigeps * abs(cx)) )
         info.numerr = 1;
       elseif (info.pinf == 1) & ( minlabz > -pars.bigeps * abs(by) )
         info.numerr = 1;
       else
         fprintf(pars.fid, 'Failed: no sensible solution/direction found.\n');
         info.numerr = 2;
       end
     end
   end
 end
% ----------------------------------------
% - Bring xsol into the complex format of original (At,c),
% - transform q-variables into r-variables (Lorentz),
% - bring ysol into complex format, indicated by K.ycomplex.
% - at 0's in ysol where rows where removed.
% ----------------------------------------'
 [xsol,ysol] = posttransfo(xsol,ysol,prep,K);
% ----------------------------------------
% Make a fancy v-plot if desired
% ----------------------------------------
 if pars.vplot == 1
   subplot(2,1,1)
   plot(0:iter,vlist,'o',[0 iter],[1 1],'b',...
     [0 iter],[pars.theta pars.theta],'g')
   title('Wide region v-plot')
   xlabel('iterations')
   ylabel('normalized v-values')
   subplot(2,1,2)
   plot(1:iter,ratelist)
   axis([0 iter 0 1])
   title('Reduction rates')
   xlabel('iterations')
   ylabel('reduction rate')
 end
