/*
   y = qdivv(q,vlab,K)
   Computes y = (Q / diag(sqrt(vlab)))'
*/
#include "mex.h"
#include "triuaux.h"

#define Y_OUT plhs[0]

#define NPAROUT 1

#define Q_IN prhs[0]
#define V_IN prhs[1]
#define K_IN prhs[2]

#define NPARIN 3

/* ************************************************************
   PROCEDURE qdivv - Computes Y = (Q / diag(sqrt(vlab)))'
   ************************************************************ */
void qdivv(double *y, const double *q,const double *v,const int n)
{
  int i,j,jcol;
  double svi;

  for(inz = 0, i = 0; i < n; i++){
    svi = sqrt(v[i]);
    for(j = 0, jcol = i; j < n; j++, jcol += n)
      y[jcol] = q[inz++] / svi;              /* y(i,j) = q(j,i) / svi */
  }
}

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  int k, nk, nksqr, lenud, lendiag;
  double *y;
  const double *q,*v;
  coneK cK;
/* ------------------------------------------------------------
   Check for proper number of arguments 
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("qdivv requires at least 3 input arguments.");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("qdivv generates 1 output argument.");
/* ------------------------------------------------------------
   Disassemble cone K structure
   ------------------------------------------------------------ */
  conepars(K_IN, &cK);
/* ------------------------------------------------------------
   Get statistics of cone K structure
   ------------------------------------------------------------ */
  lenud = cK.rDim + cK.hDim;
  lendiag = cK.lpN + 2 * cK.lorN + cK.rLen + cK.hLen;
/* ------------------------------------------------------------
   Get inputs Q, v
   ------------------------------------------------------------ */
  if(mxGetM(Q_IN) * mxGetN(Q_IN) != lenud)
    mexErrMsgTxt("q size mismatch");
  if(mxGetM(V_IN) * mxGetN(V_IN) != lendiag)
    mexErrMsgTxt("v size mismatch");
  q = mxGetPr(Q_IN);
  v = mxGetPr(V_IN) + cK.lpN + 2 * cK.lorN;    /* skip LP and Lorentz */
/* ------------------------------------------------------------
   Allocate output Y
   ------------------------------------------------------------ */
  Y_OUT =  mxCreateDoubleMatrix(lenud, 1, mxREAL);
  y = mxGetPr(Y_OUT);
/* ------------------------------------------------------------
   The actual job is done here: Y = (Q / diag(sqrt(v)))'
   ------------------------------------------------------------ */
  for(k = 0; k < cK.rsdpN; k++){                /* real symmetric */
    nk = cK.sdpNL[k];
    qdivv(y, q,v,nk);
    nksqr = SQR(nk);
    y += nksqr; q += nksqr; v += nk;
  }
  if(k < cK.sdpN){                    /* complex Hermitian */
    mexErrMsgTxt("Complex q not yet supported.");
  }
}
