/*
 [perm, xsuper] = ph2dpr1order(perm, xsuper, ph2nodes)

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <string.h>
#include "mex.h"
#include "blksdp.h"

#define PERM_OUT     myplhs[0]
#define XSUPER_OUT   myplhs[1]
#define NPAROUT 2

#define PERM_IN      prhs[0]
#define XSUPER_IN    prhs[1]
#define PH2NODES_IN  prhs[2]
#define NPARIN 3

/* ************************************************************
   PROCEDURE ph2dpr1order - Insert phase 2 nodes in 1st supernode,
     and remove them from later supernodes.
   INPUT
     perm - Length xsuper[n]. The (sparse) dpr1 ordering, w/o taking care
        of ph2nodes.
     m    - number of rows (nodes) in x
     n    - number of supernodes
     nph2 - number of ph2nodes
   UPDATED
     newperm - Length m. On input, newperm(0:nph2-1) lists the phase 2 nodes.
       On output,
       the remaining nodes are appended in the (sparse) dpr1 ordering.
     xsuper - Length n+1. On input, the supernodal partition. On output,
       the ph2nodes are all in the 1st supernode.
   WORK
     isph2 - length m character working array.
   ************************************************************ */
void ph2dpr1order(int *newperm, int *xsuper, const int *perm,
                  const int m, const int n, const int nph2, char *isph2)
{
  int k,inz,jnz, i;

/* ------------------------------------------------------------
   Initialize: isph2(0:m-1) = 0, isph2(ph2nodes) = 1.
   ------------------------------------------------------------ */
  for(k = 0; k < m; k++)
    isph2[k] = 0;
  for(k = 0; k < nph2; k++)
    isph2[newperm[k]] = 1;
/* ------------------------------------------------------------
   Source: inz. Dest: jnz. Copy all supernodes, skipping ph2-subnodes.
   ------------------------------------------------------------ */
  jnz = nph2;
  inz = 0;
  for(k = 1; k <= n; k++){
    for(; inz < xsuper[k]; inz++){
      i = perm[inz];
      if(!isph2[i])
        newperm[jnz++] = i;
    }
    xsuper[k] = jnz;
  }
/* ------------------------------------------------------------
   Copy remaining node-numbers, so that perm is permutation of 0:m-1.
   ------------------------------------------------------------ */
  for(; inz < m; inz++){
    i = perm[inz];
    if(!isph2[i])
      newperm[jnz++] = i;
  }
  mxAssert(jnz == m,"");
}

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  mxArray *myplhs[NPAROUT];
  int i,j, m,n,nph2, m1;
  int *newperm, *perm, *xsuper;
  char *cwork;
  double *xsuperPr, *permPr;
  const double *ph2nodesPr;
/* ------------------------------------------------------------
   Check for proper number of arguments
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("dpr1order requires more input arguments");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("dpr1order produces less output arguments");
/* ------------------------------------------------------------
   Get inputs perm, xsuper, ph2nodes
   ------------------------------------------------------------ */
  n = mxGetM(XSUPER_IN) * mxGetN(XSUPER_IN) - 1;
  nph2 = mxGetM(PH2NODES_IN) * mxGetN(PH2NODES_IN);
  m = mxGetM(PERM_IN) * mxGetN(PERM_IN);
  permPr = mxGetPr(PERM_IN);
  xsuperPr = mxGetPr(XSUPER_IN);
  ph2nodesPr = mxGetPr(PH2NODES_IN);
/* ------------------------------------------------------------
   Allocate integer working arrays:
   newperm(m), perm(m), xsuper(n+1);
   char work array cwork(m).
   ------------------------------------------------------------ */
  m1 = MAX(m,1);                /* avoid alloc to 0 */
  newperm = (int *) mxCalloc(m1, sizeof(int));
  perm    = (int *) mxCalloc(m1, sizeof(int));
  xsuper  = (int *) mxCalloc(n+1, sizeof(int));
  cwork = (char *) mxCalloc(m1, sizeof(char));
/* ------------------------------------------------------------
   Convert PERM_IN, XSUPER_IN and PH2NODES_IN to C-style.
   Bring ph2nodes to "newperm".
   ------------------------------------------------------------ */
  for(i = 0; i < m; i++){
    j = permPr[i];
    perm[i] = --j;
  }
  for(i = 0; i <= n; i++){
    j = xsuperPr[i];
    xsuper[i] = --j;
  }
  for(i = 0; i < nph2; i++){
    j = ph2nodesPr[i];
    newperm[i] = --j;
  }
/* ------------------------------------------------------------
   The main job is done here:
   ------------------------------------------------------------ */
  ph2dpr1order(newperm, xsuper, perm, m, n, nph2, cwork);
/* ------------------------------------------------------------
   Outputs (perm, xsuper) in Fortran style.
   ------------------------------------------------------------ */
  PERM_OUT = mxCreateDoubleMatrix(m, 1, mxREAL);
  permPr = mxGetPr(PERM_OUT);
  XSUPER_OUT  = mxCreateDoubleMatrix(n+1, 1, mxREAL);
  xsuperPr = mxGetPr(XSUPER_OUT);
  for(i = 0; i < m; i++)
    permPr[i] = newperm[i] + 1.0;
  for(i = 0; i <= n; i++)
    xsuperPr[i] = xsuper[i] + 1.0;
/* ------------------------------------------------------------
   Release working arrays
   ------------------------------------------------------------ */
  mxFree(cwork);
  mxFree(xsuper);
  mxFree(perm);
  mxFree(newperm);
/* ------------------------------------------------------------
   Copy requested output parameters (at least 1), release others.
   ------------------------------------------------------------ */
  i = MAX(nlhs, 1);
  memcpy(plhs,myplhs, i * sizeof(mxArray *));
  for(; i < NPAROUT; i++)
    mxDestroyArray(myplhs[i]);
}
