/*
 [perm,fi,idep] = innerorder(lab,p,dep,maxu,tol)
 Orders [diag(lab), p; q', p0] for stable pivot order "perm".
 The length of perm is the number of stable pivots (before the last row).
 Example:
   [perm,fi,idep] = innerorder(lab,p,dep,maxu,tol);
   if idep(1) > 0
     p(idep(1)) = idep(2);
   end

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <math.h>
#include <string.h>
#include "mex.h"
#include "blksdp.h"  /* for definition "MAX" */

/* [perm,fi,idep] = innerorder(lab,p,dep,maxu,tol) */
#define PERM_OUT myplhs[0]
#define FI_OUT   myplhs[1]
#define IDEP_OUT myplhs[2]
#define NPAROUT 3

#define LAB_IN prhs[0]
#define P_IN prhs[1]
#define DEP_IN prhs[2]
#define MAXU_IN prhs[3]
#define TOL_IN prhs[4]
#define NPARIN 5

/* ************************************************************
   PROCEDURE getInnerOrder
   INPUT:
     absp - abs(p). If there exists i s.t. lab[perm[i]]==0 then absp[perm[i]]
       is sufficiently positive, and contains norm(p(dep)).
     p    - original p.
     lab  - diagonal entries.
     maxu - Pivot such that max(abs(U-factor)) < maxu.
     n    - Length of perm.
   UPDATED
     perm - On input, lists all poassible pivots. Thus, lab(perm) is positive,
        except for possibly one position, at which p(idep) is.
        Output undefined. It's iteratively used to list still undetermined
        nodes.
   OUTPUT
     fi     - sparse (n+1) x 1. Has nonzeros at pivot numbers where a dense
       pivot is made, and at n+1. Lists corresponding multipliers (which have
       decreasing order, <= 1.0)
     permPr - length n pivot order (Fortran index, i.e. from 1:n), in float.
   ************************************************************ */
void getInnerOrder(jcir fi,double *permPr,int *perm,
                   const double *absp, const double *p, const double *lab,
                   const double maxu,int n)
{
  int inz,k,i,j,kmax,finnz;
  double psi, maxp, phi;

/* ------------------------------------------------------------
   INITIALIZE: phi = 1
   inz = length of final permutation (so far);
   finnz = nnz of fi (so far);
   n = number of (still) undetermined nodes.
   ------------------------------------------------------------ */
  phi = 1.0;
  inz = 0;
  fi.jc[0] = 0;
  finnz = fi.jc[0];
  while(n > 0){
/* ------------------------------------------------------------
   RUN THROUGH perm(1:n).
   psi = |phi|/maxu;
   k = # rejected nodes. We reject if |psi*p_j| < lab_j
   (kmax,maxp) = max |p_j| for rejected j = perm[kmax]
   ------------------------------------------------------------ */
    k = 0;
    psi = fabs(phi) / maxu;
    maxp = 0.0;
    for(i = 0; i < n; i++){
      j = perm[i];
      mxAssert(lab[j] > 0,"");
      if(psi * absp[j] <= lab[j])
        permPr[inz++] = j+1;           /* lab[j] is stable pivot */
      else{
        if(absp[j] > maxp){           /* reject this pivot */
          kmax = k;
          maxp = absp[j];
        }
        perm[k++] = j;
      }
    }
    n = k;
/* ------------------------------------------------------------
   If pivots have been rejected (i.e. n > 0) then pivot on the
   dense column, at row j=perm[kmax].
   ------------------------------------------------------------ */
    if(n > 0){
      j = perm[kmax];
      permPr[inz]  = j+1;
      fi.ir[finnz] = inz;
      fi.pr[finnz] = phi;
      perm[kmax]   = perm[--n];
/* ------------------------------------------------------------
   The elimination copies the dense column (phi*p) towards column j,
   with a proportion -lab[j]/(phi*p[j]). So newphi = (-lab_j/p_j).
   If lab[j]==0 then the remainder is purely diagonal, and any
   pivot order is stable.
   ------------------------------------------------------------ */
      mxAssert(lab[j] > 0.0,"");
      if(lab[j] == 0.0){
        phi = 0.0;                  /* p[j] is the only nonzero on this row */
        for(i = 0; i < n; i++)
          permPr[++inz] = perm[i] + 1;
        n = 0;
      }
      else
        phi = -lab[j] / p[j];           /* new multiplier for p-column */
      inz++; finnz++;
    }
  }
/* ------------------------------------------------------------
   Store final phi-multiplier at location inz = length(permPr).
   ------------------------------------------------------------ */
  fi.ir[finnz] = inz;
  fi.pr[finnz] = phi;
  fi.jc[1] = ++finnz;
}

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   [perm,fi,idep] = innerorder(lab,p,dep,maxu,tol)
   NOTE: dep should be in increasing order.
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  mxArray *myplhs[NPAROUT];
  int m,n,i,j,inz,ndep,idep;
  double maxu,pdep,tol;
  const double *lab, *pPr,*depPr;
  double *absp,*permPr, *pidep;
  int *perm, *dep, *iwork;
  jcir fi;
/* ------------------------------------------------------------
   Check for proper number of arguments
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("innerorder requires more input arguments");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("innerorder produces less output arguments");
 /* ------------------------------------------------------------
    Get input vectors lab, pPr, depPr and scalars maxu, tol
    ------------------------------------------------------------ */
  if(mxIsSparse(LAB_IN) || mxIsSparse(P_IN))
    mexErrMsgTxt("lab and p should be full");
  m = mxGetM(LAB_IN) * mxGetN(LAB_IN);
  lab = mxGetPr(LAB_IN);
  if(m != mxGetM(P_IN) * mxGetN(P_IN))
    mexErrMsgTxt("Size mismatch p");
  pPr = mxGetPr(P_IN);
  ndep = mxGetM(DEP_IN) * mxGetN(DEP_IN);
  depPr = mxGetPr(DEP_IN);
  maxu = mxGetScalar(MAXU_IN);
  tol = mxGetScalar(TOL_IN);
  if(tol < 1e-200)
    tol = 1e-200;
/* ------------------------------------------------------------
   Allocate working arrays dep=iwork(ndep + 1) and absp(m)
   ------------------------------------------------------------ */
  iwork = (int *) mxCalloc(ndep + 1, sizeof(int));
  dep = iwork;
  absp = (double *) mxCalloc(MAX(m,1), sizeof(double));
/* ------------------------------------------------------------
   Initialize dep(:) = [depPr(:)-1; m] and absp(:) = abs(pPr(:))
   ------------------------------------------------------------ */
  for(i = 0; i < ndep; i++){
    j = depPr[i];
    dep[i] = --j;
  }
  dep[ndep] = m;          /* tail of dep */
  for(i = 0; i < m; i++)
    absp[i] = fabs(pPr[i]);
/* ------------------------------------------------------------
   Let pdep = norm(p(dep)).  If pdep > tol, then this removes
   one dependency: let idep = dep[0], absp(idep) = pdep, dep = dep(2:ndep+1).
   Set n = m - length(dep).
   ------------------------------------------------------------ */
#ifdef ALLOW_DEP
  for(pdep = 0.0, i = 0; i < ndep; i++)
    pdep += SQR(absp[dep[i]]);
  pdep = sqrt(pdep);
  if(pdep > tol){
    idep = dep[0];
    absp[idep] = pdep;                       /* removes 1 dependency */
    dep++;
    ndep--;
    ++idep;         /* Make Fortran-type index to return to MATLAB */
  }
  else
#endif
  {
    idep = 0;       /* As invalid Fortran index, means |p(dep)| < tol */
    pdep = 0.0;
  }
  n = m - ndep;                                 /* resulting rank */
/* ------------------------------------------------------------
   Allocate working array perm(n), and
   initialize perm = (0:m-1) \ dep. Recall that dep[ndep] == m. 
   ------------------------------------------------------------ */
  perm = (int *) mxCalloc(MAX(1,n), sizeof(int));
  for(i = 0, j = 0, inz = 0; i <= ndep; i++){
    while(j < dep[i])
      perm[inz++] = j++;
    j++;                      /* skip dependent node */
  }
/* ------------------------------------------------------------
   Create output vectors idep(2),
   permPr(n) and fi = sparse(n+1,1)
   ------------------------------------------------------------ */
  IDEP_OUT = mxCreateDoubleMatrix(1, 2, mxREAL);
  pidep = mxGetPr(IDEP_OUT);
  PERM_OUT = mxCreateDoubleMatrix(n, 1, mxREAL);
  permPr = mxGetPr(PERM_OUT);
  FI_OUT = mxCreateSparse(n+1,1, n+1,mxREAL);
  fi.jc = mxGetJc(FI_OUT);
  fi.ir = mxGetIr(FI_OUT);
  fi.pr = mxGetPr(FI_OUT);
/* ------------------------------------------------------------
   Set IDEP-OUT = [idep, pdep].
   ------------------------------------------------------------ */
  pidep[0] = idep;
  pidep[1] = pdep;
/* ------------------------------------------------------------
   The main task: determine the stable pivot order, and store
   in permPr, filling also the sparse multiplier vector fi.
   ------------------------------------------------------------ */
  getInnerOrder(fi,permPr,perm, absp, pPr, lab, maxu,n);
/* ------------------------------------------------------------
   REALLOC fi, which finally has only fi.jc[1] nonzeros.
   ------------------------------------------------------------ */
  mxAssert(fi.jc[1] > 0,"");             /* has at least 1 nonzero */
  mxSetNzmax(FI_OUT,fi.jc[1]);
  if((fi.ir = (int *) mxRealloc(fi.ir, fi.jc[1] * sizeof(int))) == NULL)
    mexErrMsgTxt("Memory allocation error");
  mxSetIr(FI_OUT, fi.ir); 
  if((fi.pr = (double *) mxRealloc(fi.pr, fi.jc[1] * sizeof(double))) == NULL)
    mexErrMsgTxt("Memory allocation error");
  mxSetPr(FI_OUT, fi.pr);
/* ------------------------------------------------------------
   RELEASE WORKING ARRAYS iwork, perm, absp.
   ------------------------------------------------------------ */
  mxFree(perm);
  mxFree(absp);
  mxFree(iwork);
/* ------------------------------------------------------------
   Copy requested output parameters (at least 1), release others.
   ------------------------------------------------------------ */
  i = MAX(nlhs, 1);
  memcpy(plhs,myplhs, i * sizeof(mxArray *));
  for(; i < NPAROUT; i++)
    mxDestroyArray(myplhs[i]);
}    
