/* ************************************************************
function ABLK = indexa(At,K)
  ABLK = indexa(At,K) --  Create ABLK
 ABLK(:,i) lists the block numbers that are involved in constraint i:
   0=LP, 1:|K.q| = LORENTZ, |K.q|+(1:|K.s|) = PSD.

 NB: we only use the sparsity-structures of ABLK;
   the floating point contents is arbitrary.
   
    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   ************************************************************ */
#include "mex.h"
#include "blksdp.h"   /* link with vec2blks.c */

#define ABLK_OUT plhs[0]
#define NPAROUT 1

#define AT_IN prhs[0]
#define K_IN prhs[1]
#define NPARIN 2

/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
                 const int nrhs, const mxArray *prhs[])
{
  const mxArray *K_FIELD;
  coneK cK;
  jcir At, Ablk;
  int i,k,isub,maxsub,nblk,ncols, blknz,lastsub, maxnnz, nk;
  const int *blkstart;
  int *iwork;
  char newblkstart;
/* ------------------------------------------------------------
   Check for proper number of arguments
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("indexa requires more input arguments.");
  if (nlhs > NPAROUT)
    mexErrMsgTxt("indexa produces 1 output argument.");
/* ------------------------------------------------------------
   Disassemble cone K structure
   ------------------------------------------------------------ */
  conepars(K_IN, &cK);
/* --------------------------------------------------
   GET STATISTICS:
   -------------------------------------------------- */
  nblk = 1 + cK.lorN + cK.sdpN;
/* ------------------------------------------------------------
   Get K.blkstart
   ------------------------------------------------------------ */
  if( (K_FIELD = mxGetField(K_IN,0,"blkstart")) == NULL){      /* K.blkstart */
    newblkstart = 1;                                    /* create here */
    iwork = (int *) mxCalloc(nblk + 1, sizeof(int));
    iwork[0] = 0;
    iwork[1] = cK.lpN;
    k = 1; blknz = cK.lpN;
    for(i = 0; i < cK.lorN; i++)
      iwork[++k] = (blknz += cK.lorNL[i]);
    for(i = 0; i < cK.rsdpN; i++){
      nk = cK.sdpNL[i];
      iwork[++k] = (blknz += SQR(nk));
    }
    for(; i < cK.sdpN; i++){
      nk = cK.sdpNL[i];
      iwork[++k] = (blknz += 2*SQR(nk));
    }
    blkstart = iwork;
  }
  else if(!mxIsSparse(K_FIELD))
    mexErrMsgTxt("K.blkstart must be a sparse matrix.");
  else{                                   /* blkstart already available */
    newblkstart = 0;
    blkstart = mxGetIr(K_FIELD);
  }
/* --------------------------------------------------
   GET At: the "A" matrix
   -------------------------------------------------- */
  if(!mxIsSparse(AT_IN))
    mexErrMsgTxt("At must be a sparse matrix.");
  At.jc = mxGetJc(AT_IN);
  At.ir = mxGetIr(AT_IN);
  ncols = mxGetN(AT_IN);
/* ------------------------------------------------------------
   Allocate working array iwork(nblk + 1)
   ------------------------------------------------------------ */
  iwork = (int *) mxCalloc(nblk + 1, sizeof(int));
/* ------------------------------------------------------------
   ALLOCATE:
   Ablk = sparse(nblk,m,maxnnz);
   Initially set maxnnz = nnz(At), which is upper-bound on nnz(Ablk).
   Allocate only Jc and Ir (sparsity structure) part here, the double
   part comes when nnz(Ablk) is known.
   ------------------------------------------------------------ */
  ABLK_OUT = mxCreateSparse(nblk,ncols, 1,mxREAL);
  mxFree(mxGetIr(ABLK_OUT));
  mxFree(mxGetPr(ABLK_OUT));
  Ablk.jc = mxGetJc(ABLK_OUT);
  maxnnz = At.jc[ncols];
  Ablk.ir = (int *) mxCalloc(maxnnz, sizeof(int));
/* --------------------------------------------------
   Partition each constraint At(:,i) into blocks, and let Ablk denote
   the corresponding block number, for the nonempty blocks.
   -------------------------------------------------- */
  blknz = 0;
  for(i = 0; i < ncols; i++){
    Ablk.jc[i] = blknz;
    vec2blks(iwork, blkstart, At.ir, At.jc[i], At.jc[i+1], nblk);
    iwork[nblk] = At.jc[i+1];            /* close block partition */
    for(k = 0; k < nblk; k++)           /* find nonempty blocks */
      if(iwork[k] < iwork[k+1])
        Ablk.ir[blknz++] = k;           /* block j is nonempty */
  }
  Ablk.jc[ncols] = blknz;
  mxAssert(blknz <= maxnnz,"");
/* ------------------------------------------------------------
   REALLOC ABLK, and make "nonzeros" all equal to 1.0
   The all-1 float part must be allocated from scratch, since we
   released it before (to avoid unnecessary memory swapping).
   ------------------------------------------------------------ */
  blknz = MAX(1,blknz);              /* avoid realloc to 0 */
  if((Ablk.ir = (int *) mxRealloc(Ablk.ir, blknz * sizeof(int))) == NULL)
    mexErrMsgTxt("Memory allocation error");
  mxSetIr(ABLK_OUT, Ablk.ir);
  Ablk.pr = (double *) mxCalloc(blknz, sizeof(double));
  for(i = 0; i < blknz; i++)
    Ablk.pr[i] = 1.0;
  mxSetPr(ABLK_OUT, Ablk.pr);
  mxSetNzmax(ABLK_OUT,blknz);
/* ------------------------------------------------------------
   Release working array iwork
   ------------------------------------------------------------ */
  mxFree(iwork);
  if(newblkstart){
    iwork = (int *) blkstart;
    mxFree(iwork);
  }
}
