/*
  lm = factorize(lab,p,p0,q,fi)
  On input, lab,p,q, and fi are already permuted+selected to stable ordering,
  and hence [diag(lab), p;q', p0] is full rank.
  fi.ir indicates when a pivot on the dense column is needed.

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <math.h>
#include "mex.h"

/*  lm = factorize(lab,p,p0,q,fi) */
#define LM_OUT plhs[0]

#define LAB_IN prhs[0]
#define P_IN prhs[1]
#define P0_IN prhs[2]
#define Q_IN prhs[3]
#define FI_IN prhs[4]

/* ------------------------------------------------------------
   Type definitions:
   ------------------------------------------------------------ */
typedef struct{
 double *pr;
 int *jc, *ir;
    } jcir;

/* ************************************************************
   PROCEDURE: innerfactor
   INPUT:
     fi  - sparse vector of order m+1; The nonzeros are located (fi.ir) at
         the dense-column pivot numbers, and these nonzeros (fi.pr) are
         {1, lab(nz1)/p(nz1), lab(nz2)/p(nz2), ..., }
     lab - The m diagonal entries. lab >= 0, and can have at most 1 zero,
       viz. on the one-but-last nz-position of fi.
     p   - the dense column. Length m.
     p0  - the lower-right corner.
     q   - The bottom row. length m.
     m   - order of lab,p,q.
   OUTPUT:
     lm - Length m+1 vector.
   ************************************************************ */
void innerfactor(double *lm, jcir fi,const double *lab,const double *p,
                 double p0, const double *q, const int m)
{
  int i,inz;

/* ------------------------------------------------------------
   Initial block of diagonal pivots:
    [diag(lab), p; q', p0] = [eye(m), 0; lm, 1] * [diag(lab), p; 0', p0NEW],
   with lm = q./lab, p0NEW = p0 - lm'*p.
   ------------------------------------------------------------ */
  inz = fi.jc[0];
  for(i = 0; i < fi.ir[inz]; i++){
    lm[i] = q[i] / lab[i];
    p0 -= lm[i] * p[i];
  }
/* ------------------------------------------------------------
   At the pivots i = fi.ir[inz], we pivot on the dense column fi(i)*p(i). 
   This moves the dense column to the i-th column, with the next nz-fi
   as multiplier for p(i+1:m), provided that lab(i) > 0. Else, if lab(i)==0,
   then we've eliminated the dense column, and only diagonal pivots remain.
   ------------------------------------------------------------ */
  for(++inz; inz < fi.jc[1]; inz++){
    mxAssert(i == fi.ir[inz-1],"");
    lm[i] = p0 / p[i];
    if(lab[i] > 0){
      p0 -= p[i] * q[i] / lab[i];
      for(++i; i < fi.ir[inz]; i++){      /* do intermediate diagonal pivots */
        lm[i] = q[i] / lab[i];
        p0 -= lm[i] * p[i];
      }
    }
    else
      for(++i; i < m; i++){              /* lab[i]==0 => remains pure diag */
        lm[i] = q[i] / lab[i];
      }
  }
/* ------------------------------------------------------------
   The lower right corner of the L-factor is fi(m+1) * p0, if lab > 0.
   But if lab has one zero, then fi(m+1) = 0, and
   lm(m+1) = q(last dense column pivot).
   ------------------------------------------------------------ */
  --inz;                           /* now inz = fi.jc[1] - 1. */
  if(fi.pr[inz] == 0.0)           /* fi(m+1) */
    lm[m] = q[fi.ir[inz-1]];        /* |qdep| */
  else
    lm[m] = fi.pr[inz] * p0;                /* fi(m+1) * p0 */
}


/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   lm = factorize(lab,p,p0,q,fi)
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  int m;
  const double *lab,*p,*q;
  jcir fi;
  double *lm;
  double p0;

/* ------------------------------------------------------------
   Check for proper number of arguments
   ------------------------------------------------------------ */
  if(nrhs < 5)
    mexErrMsgTxt("factorize requires 5 input arguments.");
  if(nlhs > 1)
    mexErrMsgTxt("factorize generates 1 output argument.");
 /* ------------------------------------------------------------
    Get input vectors lab,p,q, fi
    ------------------------------------------------------------ */
  m = mxGetM(LAB_IN) * mxGetN(LAB_IN);
  lab = mxGetPr(LAB_IN);
  if( (mxGetM(P_IN) * mxGetN(P_IN)) != m)
    mexErrMsgTxt("p size mismatch");
  p = mxGetPr(P_IN);
  p0 = mxGetScalar(P0_IN);
  if( (mxGetM(Q_IN) * mxGetN(Q_IN)) != m)
    mexErrMsgTxt("q size mismatch");
  q = mxGetPr(Q_IN);
  if( mxGetM(FI_IN) != m+1 || mxGetN(FI_IN) != 1)
    mexErrMsgTxt("fi size mismatch");
  if( !mxIsSparse(FI_IN) )
    mexErrMsgTxt("fi should be sparse");
  fi.jc = mxGetJc(FI_IN);
  fi.ir = mxGetIr(FI_IN);
  fi.pr = mxGetPr(FI_IN);
/* ------------------------------------------------------------
   Create output vector lm(m+1)
   ------------------------------------------------------------ */
  LM_OUT = mxCreateDoubleMatrix(m+1, 1, mxREAL);
  lm = mxGetPr(LM_OUT);
/* ------------------------------------------------------------
   Compute last row of factor L
   ------------------------------------------------------------ */
  innerfactor(lm, fi,lab,p,p0,q,m);
}
