/* ************************************************************
function [dz, perm] = dzstruct(At,K)
   
    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   ************************************************************ */
#include <string.h>
#include "mex.h"
#include "blksdp.h"

#define DZ_OUT myplhs[0]
#define PERM_OUT myplhs[1]
#define NPAROUT 2

#define AT_IN prhs[0]
#define K_IN prhs[1]
#define NPARIN 2

/* ************************************************************
   PROCEDURE initAtir - Let xir = yir(yir >= ifirst).
   INPUT
     yir - subscripts of input matrix
     yjc - column pointers into yir, length m+1.
     m - number of columns
     ifirst - we select from yir only subscripts i>=ifirst, into xir.
   OUTPUT
     xir  - yir(i>=ifirst,:), alligned at xjc.
     xnnz - length of column xir[i], for i=0:m-1.
   ************************************************************ */
void initAtir(int *xir,int *xnnz, const int *xjc,const int *yir,
              const int ifirst,const int m)
{
  int j,nnzj,inz;
/* ------------------------------------------------------------
   For each column j=0:m-1, find offset for first i>=ifirst,
   and copy subscripts from there to xir+xjc[j].
   ------------------------------------------------------------ */
  for(j = 0; j < m; j++){
    inz = xjc[j];
    intbsearch(&inz, yir, xjc[j+1], ifirst); /* inz to 1st PSD block */
    nnzj = xjc[j+1] - inz;
    xnnz[j] = nnzj;
    memcpy(xir+xjc[j], yir + inz, nnzj * sizeof(int));   /* from y to x */
  }
}

/* ************************************************************
   PROCEDURE colsAtOrder - Greedy ordering of columns in At,
       starting with least number of (PSD)-nonzeros. Good fot getada.
   INPUT
     Atjc - column pointers, length m+1.
     m    - number of columns in At.
   OUTPUT
     perm - length m array, the greedy ordering, starting from sparsest.
     dzjc - length m+1, column pointers into dzir.
     dzir - length dzjc[m] <= dim(PSD). jth column lists additional
       subscripts to dz(:,0:j-1).
   UPDATED
     Atir   - (length Atjc[m]) On input, the PSD subscripts of At
     length - (length m) The length of the PSD part in each column, as in Atir.
     discard - (length m), On input, discard[i]=0 for all PSD subscripts i.
   ************************************************************ */
void colsAtOrder(int *perm, int *dzjc, int *dzir,
                 const int *Atjc,int *Atir,int *length,
                 const int m, char *discard)
{
  int kmin,lenmin,i,j,k, inz, permk;
/* ------------------------------------------------------------
   initialize: dzjc[0]=0, perm[0:m-1] = 0:m-1.
   ------------------------------------------------------------ */
  dzjc[0] = 0;
  for(i = 0; i < m; i++)
    perm[i] = i;
/* ------------------------------------------------------------
   In iterate k=0:m-1, pivot on column with length[k] minimal.
   ------------------------------------------------------------ */
  for(k = 0; k < m; k++){
    if(k > 0)
      dodiscard(Atir,length+k, Atjc,perm+k,discard, m-k);
    kmin = k;
    lenmin = length[k];
    for(j = k+1; j < m; j++)
      if(length[j] < lenmin){
        lenmin = length[j];
        kmin = j;
      }
    permk = perm[kmin];                  /* make pivot in perm */
    perm[kmin] = perm[k];
    perm[k] = permk;
    length[kmin] = length[k];
/* ------------------------------------------------------------
   Write the (additional) subscripts of the pivot column into
   k-th column of dz, i.e. dz(:,k) = At(:,permk).
   ------------------------------------------------------------ */
    memcpy(dzir + dzjc[k], Atir+Atjc[permk], lenmin * sizeof(int));
    dzjc[k+1] = dzjc[k] + lenmin;
/* ------------------------------------------------------------
   Discard dz(:,k)'s subscripts from At(:,perm(k+1:m)).
   (The actual removing is at start of loop, with k++.)
   ------------------------------------------------------------ */
    for(inz = dzjc[k]; inz < dzjc[k+1]; inz++)
      discard[dzir[inz]] = 1;
  }
}


/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
      [dz,perm] = dzstruct(At,K)   
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  mxArray *myplhs[NPAROUT];
  const mxArray *K_FIELD;
  coneK cK;
  int i,lenfull,m, psdfirst, maxnnz;
  const int *blkstart, *Atjc;
  int *iwork, *dzir, *dzjc, *Atir, *perm, *length;
  char *cwork, *discard;
  double *permPr, *dzpr;
/* ------------------------------------------------------------
   Check for proper number of arguments
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("dzstruct requires more input arguments.");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("dzstruct produces less output arguments.");
/* ------------------------------------------------------------
   Disassemble cone K structure
   ------------------------------------------------------------ */
  conepars(K_IN, &cK);
  if( (K_FIELD = mxGetField(K_IN,0,"blkstart")) == NULL)      /* K.blkstart */
    mexErrMsgTxt("Missing field `K.blkstart'.");
  else if(!mxIsSparse(K_FIELD))
    mexErrMsgTxt("K.blkstart must be a sparse matrix.");
  else
    blkstart = mxGetIr(K_FIELD);
/* ------------------------------------------------------------
   Compute some statistics based on cone K structure
   psdfirst = 1st psd-index = blkstart[1+|K.q|].
   NB blkstart[nblk] = lenfull.
   ------------------------------------------------------------ */
  lenfull = cK.lpN +  cK.qDim + cK.rDim + cK.hDim;
  psdfirst = blkstart[1+cK.lorN];         /* blkstart=length(nblk+1) */
/* --------------------------------------------------
   GET At.jc
   -------------------------------------------------- */
  if(!mxIsSparse(AT_IN))
    mexErrMsgTxt("At must be a sparse matrix.");
  Atjc = mxGetJc(AT_IN);
  m = mxGetN(AT_IN);
/* ------------------------------------------------------------
   Alloc Atir(Atjc[m]), perm(m), length(m), cwork(lenfull-psdfirst),
   ------------------------------------------------------------ */
  Atir = (int *) mxCalloc(Atjc[m], sizeof(int));
  iwork = (int *) mxCalloc(2*m, sizeof(int));
  length = iwork;                       /* length(m) */
  perm = iwork + m;                  /* perm(m) */
  cwork = (char *) mxCalloc(MAX(1,lenfull - psdfirst), sizeof(char));
/* ------------------------------------------------------------
   Initialize Atir = AT(psdfirst:lenfull, :)
   length(1:m) the corresponding column length.
   cwork = all-0
   ------------------------------------------------------------ */
  initAtir(Atir,length, Atjc,mxGetIr(AT_IN),psdfirst,m);
  discard = cwork - psdfirst;
  for(i = psdfirst; i < lenfull; i++)
    discard[i] = 0;
/* ------------------------------------------------------------
   Allocate dzjc(m+1) and dzir(lenfull-psdfirst). The actual number
   of nonzeros may be less.
   ------------------------------------------------------------ */
  dzjc = (int *) mxCalloc(m+1, sizeof(int));
  dzir = (int *) mxCalloc(MAX(1,lenfull - psdfirst), sizeof(int));
/* ------------------------------------------------------------
   The main job: greedy order of columns in of At
   ------------------------------------------------------------ */
  colsAtOrder(perm, dzjc,dzir, Atjc,Atir,length, m, discard);
/* ------------------------------------------------------------
   Realloc dzir to length dzjc[m], and create output matrix
   DZ(lenfull,m,_,_,_,dzjc[m]).
   ------------------------------------------------------------ */
  maxnnz = MAX(1,dzjc[m]);                     /* avoid realloc to 0 */
  if((dzir = (int *) mxRealloc(dzir, maxnnz * sizeof(int))) == NULL)
    mexErrMsgTxt("Memory allocation error");
  DZ_OUT = mxCreateSparse(lenfull,m,maxnnz,mxREAL);
  mxFree(mxGetJc(DZ_OUT));
  mxSetJc(DZ_OUT,dzjc);
  mxFree(mxGetIr(DZ_OUT));
  mxSetIr(DZ_OUT,dzir);
  dzpr = mxGetPr(DZ_OUT);
  for(i = 0; i < dzjc[m]; i++)
    dzpr[i] = 1.0;
/* ------------------------------------------------------------
   Create output PERM, and let PERM=perm+1.0.
   ------------------------------------------------------------ */
  PERM_OUT = mxCreateDoubleMatrix(m,1,mxREAL);
  permPr = mxGetPr(PERM_OUT);
  for(i = 0; i < m; i++)
    permPr[i] = perm[i] + 1.0;
/* ------------------------------------------------------------
   Release working arrays
   ------------------------------------------------------------ */
  mxFree(cwork);
  mxFree(iwork);
  mxFree(Atir);
/* ------------------------------------------------------------
   Copy requested output parameters (at least 1), release others.
   ------------------------------------------------------------ */
  i = MAX(nlhs, 1);
  memcpy(plhs,myplhs, i * sizeof(mxArray *));
  for(; i < NPAROUT; i++)
    mxDestroyArray(myplhs[i]);
}
