/*
    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include "blksdp.h"

/* ************************************************************
   PROCEDURE vec2blks - partition sparse vector into blocks,
     using divide and conquer (recursion, needs some stack-levels).
   INPUT
     blkstart  - length nblk array, listing the start-index of each block
     yir,ynnz  - length ynnz row indices of the vector y, to be partitioned.
     ystart    - partition y[ystart:ynnz-1]
     nblk - number of blocks in blkstart (valid for yir[ystart:ynnz-1]).
   OUTPUT
     blklocs - length nblk, start of each block in y.
   ************************************************************ */
void vec2blks(int *blklocs, const int *blkstart, const int *yir,
              const int ystart, const int ynnz, const int nblk)
{
  int i, inz;

  if(ystart == ynnz)          /* y = [] */
    for(i = 0; i < nblk; i++)          /* partition [] into blocks */
      blklocs[i] = ystart;
  else if(nblk > 1){        /* if !isempty(y) & multiple blocks */
    i = nblk / 2;           /* nblk >= 2, i >= 1 */
    inz = ystart;
    intbsearch(&inz, yir, ynnz, blkstart[i]);
    blklocs[i] = inz;
/* ------------------------------------------------------------
   Partition on the LEFT: yir(ystart,inz-1) into blocks 0:i-1.
   ------------------------------------------------------------ */
    vec2blks(blklocs, blkstart, yir, ystart, inz, i);
/* ------------------------------------------------------------
   Parition on the RIGHT: yir(inz:ynnz) into blocks i:nblk-1.
   ------------------------------------------------------------ */
    vec2blks(blklocs+i, blkstart+i, yir, inz, ynnz, nblk-i);
  }
/* ------------------------------------------------------------
   If there's only 1 possible block, then it's start is ystart
   ------------------------------------------------------------ */
  else if(nblk == 1)   /* !isempty(y) & only 1 block */
    *blklocs = ystart;
}

/* ************************************************************
   PROCEDURE vec2selblks - partition sparse vector into selected blocks,
     using divide and conquer (recursion, needs some stack-levels).
     Completely analogous to vec2blks.
   INPUT
     blkstart  - lists the start-index of each block
     yir,ynnz  - length ynnz row indices of the vector y, to be partitioned.
     ystart    - partition y[ystart:ynnz-1]
     blkir     - length blknnz array of block numbers
     blknnz    - number of block numbers in blkir
   OUTPUT
     blklocs - length blknnz, blklocs[i] is start of block blkir[i] in y.
   ************************************************************ */
void vec2selblks(int *blklocs, const int *blkstart, const int *yir,
                 const int ystart, const int ynnz,
                 const int *blkir, const int blknnz)
{
  int i, inz;

  if(ystart == ynnz)        /* if y=[] */
    for(i = 0; i < blknnz; i++)          /* partition [] into sel.blocks */
      blklocs[i] = ystart;
  else if(blknnz > 1){       /* if !isempty(y) & multiple blocks */
    i = blknnz / 2;           /* blknnz >= 2, i >= 1 */
    inz = ystart;
    intbsearch(&inz, yir, ynnz, blkstart[blkir[i]]);
    blklocs[i] = inz;        /* determined start ith selected block */
/* ------------------------------------------------------------
   Partition on the LEFT: yir(ystart,inz-1) into blocks blkir[0:i-1].
   ------------------------------------------------------------ */
    vec2selblks(blklocs, blkstart, yir, ystart, inz, blkir, i);
/* ------------------------------------------------------------
   Parition on the RIGHT: yir[inz:ynnz-1] into blocks blkir[i+1:blknnz-1].
   ------------------------------------------------------------ */
    i++;                  /* Done up to and including block i */
    vec2selblks(blklocs+i, blkstart, yir, inz, ynnz, blkir+i, blknnz-i);
  }
/* ------------------------------------------------------------
   There's only 1 selected block; determine its position.
   ------------------------------------------------------------ */
  else if(blknnz == 1){
    inz = ystart;
    intbsearch(&inz, yir, ynnz, blkstart[blkir[0]]);
    *blklocs = inz;
  }
}
