/*
   y = scaleK(d,ud,x,K,transp)
   Computes y = D(d) x with d in K.
   !transp (default) then Y = Ld' * X * Ld
   transp == 1 then Y = Ud' * X * Ud

   Uses pivot ordering ud.perm if available and nonempty.

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "mex.h"
#include "triuaux.h"
#include "blksdp.h"

/*    y = scaleK(d,ud,x,K,transp) */

#define Y_OUT plhs[0]
#define NPAROUT 1

#define D_IN prhs[0]
#define UD_IN prhs[1]
#define X_IN prhs[2]
#define K_IN prhs[3]
#define NPARINMIN 4
#define TRANSP_IN prhs[4]
#define NPARIN 5

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   y = scaleK(d,ud,x,K,transp)
   Computes y = D(d) x with d in K.
   !transp (default) then Y = Ld' * X * Ld
   transp == 1 then Y = Ud' * X * Ud
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  const mxArray *UD_FIELD;
  int lenfull, lend, lenud, sdplen, transp, fwsiz, i,k;
  double *fwork, *y,*permPr;
  const double *d,*x,*ud, *qdetd;
  int *perm, *iwork;
  coneK cK;
  char use_pivot;

/* ------------------------------------------------------------
   Check for proper number of arguments 
   ------------------------------------------------------------ */
  if(nrhs < NPARIN){
    transp = 0;
    if(nrhs < NPARINMIN)
      mexErrMsgTxt("scaleK requires more input arguments.");
  }
  else
    transp = mxGetScalar(TRANSP_IN);
  if (nlhs > NPAROUT)
    mexErrMsgTxt("scaleK generates 1 output argument.");
/* ------------------------------------------------------------
   Disassemble cone K structure
   ------------------------------------------------------------ */
  conepars(K_IN, &cK);
/* ------------------------------------------------------------
   Get statistics of cone K structure
   ------------------------------------------------------------ */
  lenfull = cK.lpN +  cK.qDim + cK.rDim + cK.hDim;
  lend = cK.lpN +  cK.qDim;
  lenud = cK.rDim + cK.hDim;
  sdplen = cK.rLen + cK.hLen;
/* ------------------------------------------------------------
   Get scale data: (d,ud) and input x.
   ------------------------------------------------------------ */
  d = mxGetPr(D_IN);
  if(!mxIsStruct(UD_IN))
    mexErrMsgTxt("Parameter `ud' should be a structure.");
  if( (UD_FIELD = mxGetField(UD_IN,0,"u")) == NULL)            /* ud.u */
    mexErrMsgTxt("Field ud.u missing.");
  if(mxGetM(UD_FIELD) * mxGetN(UD_FIELD) != lenud)
    mexErrMsgTxt("ud.u size mismatch.");
  ud = mxGetPr(UD_FIELD);
  UD_FIELD = mxGetField(UD_IN,0,"perm");                       /* ud.perm */
  if((use_pivot = (UD_FIELD != NULL))){
    if(mxGetM(UD_FIELD) * mxGetN(UD_FIELD) == sdplen)
      permPr = mxGetPr(UD_FIELD);
    else if(mxGetM(UD_FIELD) * mxGetN(UD_FIELD) == 0)
      use_pivot = 0;
    else
      mexErrMsgTxt("ud.perm size mismatch");
  }
  if( (UD_FIELD = mxGetField(UD_IN,0,"qdet")) == NULL)         /* ud.qdet */
    mexErrMsgTxt("Field ud.qdet missing.");
  if(mxGetM(UD_FIELD) * mxGetN(UD_FIELD) != cK.lorN)
    mexErrMsgTxt("ud.qdet size mismatch.");
  qdetd = mxGetPr(UD_FIELD);
/* ------------------------------------------------------------
   Get input x
   ------------------------------------------------------------ */
  if(mxIsSparse(X_IN))
    mexErrMsgTxt("Sparse x not supported by this version of scaleK.");
  x = mxGetPr(X_IN);
/* ------------------------------------------------------------
   Validate input:
   length(d) == lend ?  length(x) == lenfull ?
   NB: the idea is that X is Hermitian, but this we don't check.
   ------------------------------------------------------------ */
  if(mxGetM(D_IN) * mxGetN(D_IN) < lend)
    mexErrMsgTxt("size d mismatch.");
  if(mxGetM(X_IN) * mxGetN(X_IN) != lenfull)
    mexErrMsgTxt("size x mismatch.");
/* ------------------------------------------------------------
   Allocate output Y
   ------------------------------------------------------------ */
  Y_OUT =  mxCreateDoubleMatrix(lenfull, 1, mxREAL);
  y = mxGetPr(Y_OUT);
/* ------------------------------------------------------------
   Allocate fwork 2 * [ max(cK.rMaxn^2, 2*cK.hMaxn^2) ]
   MAKE SURE THAT A POSITIVE SIZE IS ALLOCATED (FOR MATLAB 5.0)
   iwork = int(rLen+hLen)
   ------------------------------------------------------------ */
  fwsiz = MAX(SQR(cK.rMaxn),2*SQR(cK.hMaxn));
  fwork = (double *) mxCalloc( MAX(1,2 * fwsiz), sizeof(double));
  iwork = (int *) mxCalloc( MAX(1, cK.rLen + cK.hLen), sizeof(int) );
/* ------------------------------------------------------------
   Convert Fortran to C-style in perm:
   ------------------------------------------------------------ */
  if(use_pivot){
    perm = iwork;
    for(k = 0; k < cK.rLen + cK.hLen; k++){
      i = permPr[k];
      perm[k] = --i;
    }
  }
  else
    perm = (int *) NULL;
/* ------------------------------------------------------------
   The actual job is done here:.
   ------------------------------------------------------------ */
  scaleK(y, d,ud,qdetd,perm,x,cK,transp, fwork);
/* ------------------------------------------------------------
   Release working arrays.
   ------------------------------------------------------------ */
  mxFree(fwork);
  mxFree(iwork);
}
