/*
 y = qbxqb(qr,beta,x,K)
*/

#include "mex.h"
#include "blksdp.h"

#define Y_OUT plhs[0]

#define NPAROUT 1

#define QR_IN prhs[0]
#define BETA_IN prhs[1]
#define X_IN prhs[2]
#define K_IN prhs[3]
#define NPARIN 4

/* ************************************************************
   PROCEDURE elqxq - Compute (I-beta * c*c') * X * (I-beta * c*c')
   INPUT
     beta - Householder scalar coefficient
     c    - lentgth n elementary Householder vector
     m    - length of columns in X. m >= n.
     n    - order of Householder reflection.
   UPDATED
     x -  m x (n-1)+n. On output, Xnew = (I-beta * c*c') * X * (I-beta * c*c')
       This means: start with order m reflection, up to order 1 reflection.
   WORK
     y - lentgth n working vector.
   ************************************************************ */
void elqxq(double *x, const double beta, const double *c,
           const int n, const int m, double *y)
{
  int j;
  double *xj;
  double alpha;
/* ------------------------------------------------------------
   Compute y = c'*X
   ------------------------------------------------------------ */
  for(j = 0, xj = x; j < n; j++, xj += m)
    y[j] = realdot(c, xj, n);
/* ------------------------------------------------------------
   Let X += beta*c*(beta*(y'*c) * c' - y') - beta * y*c'
   ------------------------------------------------------------ */
  alpha = beta * realdot(y, c, n);
  for(j = 0, xj = x; j < n; j++, xj += m){
    addscalarmul(xj, beta*(alpha * c[j] - y[j]), c, n);
    addscalarmul(xj, beta*c[j], y, n);
  }
}

/* ************************************************************
   PROCEDURE qxqt - computes Qb * X * Qb'
   INPUT
     beta - length m vector
     c    - m x m matrix, lower triangular gives Householder reflections
     m    - order
   UPDATED
     x -  m x m. On output, Xnew = Qb * X * Qb'
       This means: start with order m reflection, up to order 1 reflection.
   WORK
     fwork - lentgth m working vector.
   ************************************************************ */
void qxqt(double *x, const double beta, const double *c,
           const int m, double *fwork)
{
  int k, n, inz;

  n = m;
  inz = 0;
  for(k = 0; k < m; k++, inz += m+1, n--)
    elqxq(x + inz, beta[k], c + inz, n, m, fwork);
}

/* ************************************************************
   PROCEDURE qtxq - computes Qb' * X * Qb
   INPUT
     beta - length m vector
     c    - m x m matrix, lower triangular gives Householder reflections
     m    - order
   UPDATED
     x -  m x m. On output, Xnew = Qb' * X * Qb
       This means: start with order 1 reflection, up to order m reflection.
   WORK
     fwork - lentgth m working vector.
   ************************************************************ */
void qxqt(double *x, const double beta, const double *c,
           const int m, double *fwork)
{
  int k, inz;

  inz = SQR(m) - 1;
  for(k = 1; k <= m; k++, inz -= m+1)
    elqxq(x + inz, beta[m-k], c + inz, k, m, fwork);
}
