/*
   [d,qdetd,vfrmq] = pdtranslq(x,qdetx,vinvq,v,K);

   Computes d=D(x)* vinv, with vinv = (D(x)z)^{-1/2}, v = sqrt(eigK(D(x)z)),
   LP and LORENTZ part only. Also computes Lorentz frame of fullv = D(d^{-1})x.

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <math.h>
#include <string.h>
#include "mex.h"
#include "blksdp.h"

#define D_OUT myplhs[0]
#define QDETD_OUT myplhs[1]
#define VFRMQ_OUT myplhs[2]
#define NPAROUT 3

#define X_IN prhs[0]
#define QDETX_IN prhs[1]
#define VINVQ_IN prhs[2]
#define V_IN prhs[3]
#define K_IN prhs[4]
#define NPARIN 5

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  mxArray *myplhs[NPAROUT];
  int i,k, nk, lendiag;
  double *d,*qdetd,*vfrmq,*fwork;
  const double *x,*qdetx,*v,*vinvq;
  coneK cK;
/* ------------------------------------------------------------
   Check for proper number of arguments 
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("pdtranslq requires more input arguments.");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("pdtranslq generates less output arguments.");
/* ------------------------------------------------------------
   Disassemble cone K structure
   ------------------------------------------------------------ */
  conepars(K_IN, &cK);
/* ------------------------------------------------------------
   Get statistics of cone K structure
   ------------------------------------------------------------ */
  lendiag = cK.lpN + 2 * cK.lorN + cK.rLen + cK.hLen;
/* ------------------------------------------------------------
   Get inputs x, qdetx, vinvq, v
   ------------------------------------------------------------ */
  x = mxGetPr(X_IN);
  if(mxGetM(QDETX_IN) * mxGetN(QDETX_IN) != cK.lorN)
    mexErrMsgTxt("qdetx size mismatch");
  qdetx = mxGetPr(QDETX_IN);
  if(mxGetM(V_IN) * mxGetN(V_IN) != lendiag)
    mexErrMsgTxt("v size mismatch");
  v = mxGetPr(V_IN);
  if(mxGetM(VINVQ_IN) * mxGetN(VINVQ_IN) != cK.qDim)
    mexErrMsgTxt("vinvq size mismatch");
  vinvq = mxGetPr(VINVQ_IN);
/* ------------------------------------------------------------
   Allocate outputs d,qdetd,vfrmq
   ------------------------------------------------------------ */
  D_OUT =  mxCreateDoubleMatrix(cK.lpN + cK.qDim, 1, mxREAL);
  d = mxGetPr(D_OUT);
  QDETD_OUT =  mxCreateDoubleMatrix(cK.lorN, 1, mxREAL);
  qdetd = mxGetPr(QDETD_OUT);
  VFRMQ_OUT =  mxCreateDoubleMatrix(cK.qDim - cK.lorN, 1, mxREAL);
  vfrmq = mxGetPr(VFRMQ_OUT);
/* ------------------------------------------------------------
   Allocate working array fwork(qMaxn)
   ------------------------------------------------------------ */
 fwork = (double *) mxCalloc(MAX(1,cK.qMaxn), sizeof(double));
/* ------------------------------------------------------------
   The actual job is done here:
   ------------------------------------------------------------ */
/* ------------------------------------------------------------
   LP: d = x./v
   ------------------------------------------------------------ */
  realHadadiv(d, x, v, cK.lpN);
  d += cK.lpN;
  x += cK.lpN;
  v += cK.lpN;
/* ------------------------------------------------------------
   LORENTZ: d = D(x)*vinv,  detd = detx/detv
   ------------------------------------------------------------ */
  for(k = 0; k < cK.lorN; k++){                 /* LORENTZ */
    nk = cK.lorNL[k];
    qdetd[k] = qdetx[k] / sqrt( v[0] * v[1] );
    qlmul(d,x,vinvq,qdetx[k],nk);               /* d = D(x)*vinvq */
/* ------------------------------------------------------------
   vfull = D(d^{-1})x. To get its frame f, we note that
   vfull(1:n-1) = f * (v2-v1).  If v2-v1 = 0, then we take f = e1/sqrt(2).
   ------------------------------------------------------------ */
    if(v[1]-v[0] > 1e-12 * v[1]){
      qldiv(fwork, d,x, qdetd[k],nk);             /* v = D(d)\x */
      scalarmul(vfrmq, 1.0 / (v[1] - v[0]), fwork+1, nk-1);
    }
    else
      vfrmq[0] = M_SQRT1_2;              /* identity: we can pick any frame */
    x += nk; d += nk; vinvq += nk; vfrmq += nk-1; v+=2;
  }
/* ------------------------------------------------------------
   Release working array
   ------------------------------------------------------------ */
  mxFree(fwork);
/* ------------------------------------------------------------
   Copy requested output parameters (at least 1), release others.
   ------------------------------------------------------------ */
  i = MAX(nlhs, 1);
  memcpy(plhs,myplhs, i * sizeof(mxArray *));
  for(; i < NPAROUT; i++)
    mxDestroyArray(myplhs[i]);
}
