/*
 y = givensrot(gjc,g,x,K)
*/
#include <string.h>
#include "mex.h"
#include "blksdp.h"
#include "givens.h"

#define Y_OUT plhs[0]
#define NPAROUT 1

#define GJC_IN prhs[0]
#define G_IN prhs[1]
#define X_IN prhs[2]
#define K_IN prhs[3]
#define NPARIN 4

/* ************************************************************
   PROCEDURE matgivens
   INPUT
     gjc, g - sequence of givens rotations
   UPDATED
     y - let Y := Q_g * Y.
   ************************************************************ */
void matgivens(double *y, const twodouble *g, const int *gjc, const int n)
{
  int j,k;

  for(j = 0; j < n; j++, y += n)       /* For all n columns of y */
/* ------------------------------------------------------------
   At step k, we apply m rotations involving rows k:k+m, m=gjc[k+1]-gjc[k].
   ------------------------------------------------------------ */
    for(k = 0; k < n-1; k++)
      givensrot(y+k, g+gjc[k], gjc[k+1]-gjc[k]);
}

/* complex case */
void prpimatgivens(double *y,double *ypi, const tridouble *g,
                   const int *gjc, const int n)
{
  int j,k;

  for(j = 0; j < n; j++, y += n, ypi += n)       /* For all n columns of y */
/* ------------------------------------------------------------
   At step k, we apply m rotations involving rows k:k+m, m=gjc[k+1]-gjc[k].
   ------------------------------------------------------------ */
    for(k = 0; k < n-1; k++)
      prpigivensrot(y+k,ypi+k, g+gjc[k], gjc[k+1]-gjc[k]);
}


/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  int inz, i, k, nk, nksqr, lenud, sdplen, gnnz;
  int *gjc, *iwork;
  const double *gjcPr;
  const double *g, *gk;
  double *y;
  coneK cK;
/* ------------------------------------------------------------
   Check for proper number of arguments 
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("givensrot requires more input arguments.");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("givensrot generates less output arguments.");
/* ------------------------------------------------------------
   Disassemble cone K structure
   ------------------------------------------------------------ */
  conepars(K_IN, &cK);
/* ------------------------------------------------------------
   Get statistics of cone K structure
   ------------------------------------------------------------ */
  lenud = cK.rDim + cK.hDim;
  sdplen = cK.rLen + cK.hLen;
/* ------------------------------------------------------------
   Get inputs gjc,g,x
   ------------------------------------------------------------ */
  if(mxGetM(GJC_IN) * mxGetN(GJC_IN) != sdplen)
    mexErrMsgTxt("gjc size mismatch");
  gjcPr = mxGetPr(GJC_IN);
  g = (double *) mxGetPr(G_IN);
  gnnz = mxGetM(G_IN) * mxGetN(G_IN);
  if(mxGetM(X_IN) != lenud || mxGetN(X_IN) != 1)
    mexErrMsgTxt("x size mismatch");
/* ------------------------------------------------------------
   Allocate output y(lenud), and let y = x.
   ------------------------------------------------------------ */
  Y_OUT = mxCreateDoubleMatrix(lenud, 1, mxREAL);
  y = mxGetPr(Y_OUT);
  memcpy(y, mxGetPr(X_IN), lenud * sizeof(double));
/* ------------------------------------------------------------
   Allocate working array iwork(sum(K.s))
   ------------------------------------------------------------ */
  iwork = (int *) mxCalloc(MAX(1,sdplen), sizeof(int));
/* ------------------------------------------------------------
   Convert gjcPr from float to int, and store in gjc:=iwork.
   ------------------------------------------------------------ */
  gjc = iwork;
  for(i = 0; i < sdplen; i++)
    gjc[i] = gjcPr[i];                 /* don't subtract 1: already C-style */
/* ------------------------------------------------------------
   The actual job is done here: U_NEW = Q(g) * U_OLD
   ------------------------------------------------------------ */
  inz = 0;
  for(k = 0; k < cK.rsdpN; k++){                /* real symmetric */
    nk = cK.sdpNL[k];
    nksqr = SQR(nk);
    if(inz + 2 * gjc[nk-1] > gnnz)
      mexErrMsgTxt("g size mismatch");
    gk = g+inz;
    matgivens(y, (twodouble *) gk, gjc, nk);
    y += nksqr;
    inz += 2 * gjc[nk-1];        /* each rotation consists of 2 doubles */
    gjc += nk;
  }
  for(; k < cK.sdpN; k++){                       /* complex Hermitian */
    nk = cK.sdpNL[k];
    nksqr = SQR(nk);
    if(inz + 3 * gjc[nk-1] > gnnz)
      mexErrMsgTxt("g size mismatch");
    gk = g+inz;
    prpimatgivens(y,y+nksqr, (tridouble *) gk, gjc, nk);
    nksqr += nksqr;
    y += nksqr;
    inz += 3 * gjc[nk-1];            /* each rotation consists of 3 doubles */
    gjc += nk;
  }
/* ------------------------------------------------------------
   Release working arrays
   ------------------------------------------------------------ */
  mxFree(iwork);
}
