/*
 fwdpr1 - 
    y = fwdpr1(Lden, b)  Solves
    "PROD_k L(pk,betak) * y = b", where
    where L(p,beta) = eye(n) + tril(p*beta',-1).

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <math.h>
#include <string.h>
#include "mex.h"
#include "blksdp.h"

/* y = fwdpr1(Lden, b) */
#define Y_OUT plhs[0]
#define NPAROUT 1

#define LDEN_IN prhs[0]
#define B_IN prhs[1]
#define NPARIN 2

/* ************************************************************
   PROCEDURE fwprodform - Solves PROD_j L(pj,betaj) * yNEW = yOLD.
   INPUT
     p - nonzeros of sparse m x n matrix P. Has xsuper(j+1) nonzeros in
      column j.
     xsuper - xsuper(j+1) is number of nonzeros in p(:,j).
     perm - lists pivot order for columns where ordered(j)==1.
     beta, betajc - Column beta(betajc[j]:betajc[j+1]-1) provides the
       beta vector in the j-th L-factor, L(p,beta)=eye(m) + tril(p*beta',-1).
     ordered - ordered[j]==1 iff p(:,j) and beta(L:,j) have been reordered;
       the original row numbers are in perm(:,j).
     n - number of columns in p, beta.
   UPDATED
     y - length m vector. On input, the rhs. On output the solution to
       PROD_j L(pj,betaj) * yNEW = yOLD.
   ************************************************************ */
void fwprodform(double *y, const int *xsuper, const int *perm,
                const double *p, const double *beta, const int *betajc,
                const char *ordered, const int n)
{
  int k,nk, mk;
/* ------------------------------------------------------------
   Forward solve L(pk,betak) * yNEXT = yPREV   for k=0,1,..,n-1, resp.
   ------------------------------------------------------------ */
  for(k = 1; k <= n; k++){
    mk = xsuper[k];
    nk = betajc[k] - betajc[k-1];
    if(ordered[k-1]){
      fwipr1o(y, perm, p, beta, mk, nk);
      perm += mk;
    }
    else
      fwipr1(y, p, beta, mk, nk);
    beta += nk;
    p += mk;
  }
}

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
       y = fwdpr1(Lden, b)
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
 const mxArray *L_FIELD;
 char *ordered;
 int m,n,nden, i,j, permnnz, nadd;
 const double *beta, *betajcPr, *permPr, *b, *orderedPr, *xsuperPr,
   *pivpermPr, *p;
 int *betajc, *perm, *pivperm, *xsuper;
 double *y;
/* ------------------------------------------------------------
   Check for proper number of arguments 
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("fwdpr1 requires more input arguments.");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("fwdpr1 generates less output arguments.");
/* ------------------------------------------------------------
   Get input b
   ------------------------------------------------------------ */
  if(mxIsSparse(B_IN))
    mexErrMsgTxt("b should be full");
  m = mxGetM(B_IN);
  n = mxGetN(B_IN);
  b = mxGetPr(B_IN);
/* ------------------------------------------------------------
   Disassemble dense-update structure Lden (p,xsuper,beta,betajc,rowperm)
   ------------------------------------------------------------ */
  if(!mxIsStruct(LDEN_IN))
    mexErrMsgTxt("Parameter `Lden' should be a structure.");
  if( (L_FIELD = mxGetField(LDEN_IN,0,"betajc")) == NULL)      /* betajc */
    mexErrMsgTxt("Missing field Lden.betajc.");
  nden = mxGetM(L_FIELD) * mxGetN(L_FIELD) - 1;
  if(nden > 0){
    betajcPr = mxGetPr(L_FIELD);
    if( (L_FIELD = mxGetField(LDEN_IN,0,"p")) == NULL)            /* p */
      mexErrMsgTxt("Missing field Lden.p.");
    p = mxGetPr(L_FIELD);
    if( (L_FIELD = mxGetField(LDEN_IN,0,"pivperm")) == NULL)      /* pivperm */
      mexErrMsgTxt("Missing field Lden.pivperm.");
    pivpermPr = mxGetPr(L_FIELD);
    permnnz = mxGetM(L_FIELD) * mxGetN(L_FIELD);
    if( (L_FIELD = mxGetField(LDEN_IN,0,"xsuper")) == NULL)       /* xsuper */
      mexErrMsgTxt("Missing field Lden.xsuper.");
    nadd = mxGetM(L_FIELD) * mxGetN(L_FIELD) - 1;
    if(nadd > nden)
      mexErrMsgTxt("Size mismatch xsuper.");
    xsuperPr = mxGetPr(L_FIELD);
    if( (L_FIELD = mxGetField(LDEN_IN,0,"beta")) == NULL)          /* beta */
      mexErrMsgTxt("Missing field Lden.beta.");
    beta = mxGetPr(L_FIELD);
    if( (L_FIELD = mxGetField(LDEN_IN,0,"rowperm")) == NULL)      /* rowperm */
      mexErrMsgTxt("Missing field Lden.rowperm.");
    if(m != mxGetM(L_FIELD) * mxGetN(L_FIELD))
      mexErrMsgTxt("Size mismatch Lden.rowperm.");
    permPr = mxGetPr(L_FIELD);
    if( (L_FIELD = mxGetField(LDEN_IN,0,"dopiv")) == NULL)         /* dopiv */
      mexErrMsgTxt("Missing field Lden.dopiv.");
    if(mxGetM(L_FIELD) * mxGetN(L_FIELD) != nden)
      mexErrMsgTxt("Size mismatch Lden.dopiv.");
    orderedPr = mxGetPr(L_FIELD);
  }
/* ------------------------------------------------------------
   Create output y
   ------------------------------------------------------------ */
  Y_OUT = mxCreateDoubleMatrix(m, n, mxREAL);
  y = mxGetPr(Y_OUT);
  if(nden == 0)
    memcpy(y, b, m*n * sizeof(double));             /* if no dense cols */
  else{
/* ------------------------------------------------------------
   Allocate working arrays betajc(nden+1), perm(m), pivperm(permnnz),
   ordered(nden), xsuper(nden+1)
   ------------------------------------------------------------ */
    betajc = (int *) mxCalloc(nden + 1,sizeof(int));
    ordered = (char *) mxCalloc(nden,sizeof(char));
    perm = (int *) mxCalloc(MAX(m,1),sizeof(int));
    pivperm = (int *) mxCalloc(MAX(permnnz,1),sizeof(int));
    xsuper = (int *) mxCalloc(nden+1,sizeof(int));
/* ------------------------------------------------------------
   Convert xsuper to integer. Append xsuper[nadd] up to entry n.
   ------------------------------------------------------------ */
  for(i = 0; i <= nadd; i++){
    j = xsuperPr[i];
    xsuper[i] = --j;
  }
  while(i <= nden)
    xsuper[i++] = j;   /*  The phase-2 cols are all length xsuper[nadd]. */
/* ------------------------------------------------------------
   Convert betajcPr, ordered, permPr, pivperm to C-style
   ------------------------------------------------------------ */
    for(i = 0; i <= nden; i++){
      j = betajcPr[i];
      betajc[i] = --j;
    }
    for(i = 0; i < nden; i++){
      ordered[i] = orderedPr[i];
    }
    for(i = 0; i < permnnz; i++){
      pivperm[i] = pivpermPr[i];
    }
    L_FIELD = mxGetField(LDEN_IN,0,"beta");
    if(mxGetM(L_FIELD) * mxGetN(L_FIELD) != betajc[nden])
      mexErrMsgTxt("Size mismatch Lden.beta.");
    for(i = 0; i < m; i++){
      j = permPr[i];
      perm[i] = --j;
    }
/* ------------------------------------------------------------
   The actual job is done here: y = PROD_L\b(perm).
   ------------------------------------------------------------ */
    for(j = 0; j < n; j++, y += m, b+=m){
      for(i = 0; i < m; i++)            /* y = b(perm) */
        y[i] = b[perm[i]];
      fwprodform(y, xsuper, pivperm, p, beta, betajc, ordered, nden);
    }
/* ------------------------------------------------------------
   Release working arrays
   ------------------------------------------------------------ */
    mxFree(xsuper);
    mxFree(pivperm);
    mxFree(ordered);
    mxFree(perm);
    mxFree(betajc);
  }
}
