/*
   x = frameit(lab,frmq,frms,K);

   Computes x = FRM*lab. For Lorentz, we have nk-1 length frame "frmq";
   for PSD, the frame "frms" is a product-form Householder reflection. 

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <math.h>
#include <string.h>
#include "mex.h"
#include "blksdp.h"
#include "reflect.h"

#define X_OUT plhs[0]
#define NPAROUT 1

#define LAB_IN prhs[0]
#define FRMQ_IN prhs[1]
#define FRMS_IN prhs[2]
#define K_IN prhs[3]
#define NPARIN 4

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  int i,k, nk,nksqr, inz, lendiag, lenfull, sdpdim,qsize;
  double *x, *fwork;
  const double *lab,*frmq,*frms, *beta;
  coneK cK;
/* ------------------------------------------------------------
   Check for proper number of arguments 
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("frameit requires more input arguments.");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("frameit generates less output arguments.");
/* ------------------------------------------------------------
   Disassemble cone K structure
   ------------------------------------------------------------ */
  conepars(K_IN, &cK);
/* ------------------------------------------------------------
   Get statistics of cone K structure
   ------------------------------------------------------------ */
  sdpdim = cK.rDim + cK.hDim;
  qsize = sdpdim + cK.hLen;
  lenfull = cK.lpN +  cK.qDim + sdpdim;
  lendiag = cK.lpN + 2 * cK.lorN + cK.rLen + cK.hLen;
/* ------------------------------------------------------------
   Get inputs lab,frmq,frms
   ------------------------------------------------------------ */
  if(mxGetM(LAB_IN) * mxGetN(LAB_IN) != lendiag)
    mexErrMsgTxt("lab size mismatch");
  lab = mxGetPr(LAB_IN);
  if(mxGetM(FRMQ_IN) * mxGetN(FRMQ_IN) != cK.qDim-cK.lorN)
    mexErrMsgTxt("frmq size mismatch");
  frmq = mxGetPr(FRMQ_IN);
  if(mxGetM(FRMS_IN) * mxGetN(FRMS_IN) != qsize)
    mexErrMsgTxt("frms size mismatch");
  frms = mxGetPr(FRMS_IN);
/* ------------------------------------------------------------
   Allocate output x
   ------------------------------------------------------------ */
  X_OUT =  mxCreateDoubleMatrix(lenfull, 1, mxREAL);
  x = mxGetPr(X_OUT);
/* ------------------------------------------------------------
   Allocate working array fwork(max(rmaxn,2*hmaxn))
   ------------------------------------------------------------ */
  fwork = (double *) mxCalloc(MAX(1,MAX(cK.rMaxn,2*cK.hMaxn)),sizeof(double));
/* ------------------------------------------------------------
   The actual job is done here:
   ------------------------------------------------------------ */
/* ------------------------------------------------------------
   LP: x = lab
   ------------------------------------------------------------ */
  memcpy(x, lab, cK.lpN * sizeof(double));
  x += cK.lpN;
  lab += cK.lpN;
/* ------------------------------------------------------------
   LORENTZ:  y = F*lab,
    with F*lab = [(lab1+lab2)/sqrt(2);  (lab2-lab1)*frame]
   ------------------------------------------------------------ */
  for(k = 0; k < cK.lorN; k++){                 /* LORENTZ */
    nk = cK.lorNL[k];
    x[0] = (lab[0] + lab[1]) / M_SQRT2;
    scalarmul(x+1, lab[1] - lab[0], frmq, nk-1);
    x += nk; frmq += nk-1; lab += 2;
  }
/* ------------------------------------------------------------
   PSD: X = Qb' * diag(lab) * Qb,          Qb = Q_1*Q_2*..*Q_{n-1}
   where Q_k = I-ck*ck'/betak is an elementary Householder reflection.
   Format: frms = [c1, .., c_{n-1}, beta].
   ------------------------------------------------------------ */
  for(k = 0; k < cK.rsdpN; k++){                /* real symmetric */
    nk = cK.sdpNL[k];
    nksqr = SQR(nk);
    for(i = 0, inz = 0; i < nk; i++, inz += nk+1)  /* Let X = diag(lab) */
      x[inz] = lab[i];
    qtxq(x, frms + nksqr - nk, frms, nk, fwork);
    tril2sym(x,nk);
    x += nksqr; frms += nksqr;
    lab += nk;
  }
/* ------------------------------------------------------------
   For complex Hermitian, we have X = Qb' * diag(lab) * Qb,
   where Qb = Q_1 * ... * Q_{n-1} * diag(q(:,n)),
   with q(:,n) a complex sign-vector.
   Format: [RE q(:,1:n-1), RE q(:,n), IM q(:,1:n-1), IM q(:,n), beta].
   ------------------------------------------------------------ */
  for(; k < cK.sdpN; k++){                    /* complex Hermitian */
    nk = cK.sdpNL[k];
    nksqr = SQR(nk);
    for(i = 0, inz = 0; i < nk; i++, inz += nk+1)  /* Let X = diag(lab) */
      x[inz] = lab[i];
    beta = frms + 2*nksqr;                   /* beta = frms(:,2*n+1) */
    prpiqtxq(x,x+nksqr, beta, frms,frms+nksqr, nk, fwork);
    tril2herm(x,x+nksqr, nk);
    nksqr += nksqr;
    x += nksqr; frms += nksqr + nk;          /* skip also beta. */
    lab += nk;
  }
/* ------------------------------------------------------------
   Release working array
   ------------------------------------------------------------ */
  mxFree(fwork);
}
