/*
 y = bwslvp(U,perm,b)

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <string.h>
#include "mex.h"
#include "blksdp.h"

/*
 y = bwslvp(U,perm,b)
*/

#define Y_OUT plhs[0]
#define NPAROUT 1

#define U_IN prhs[0]
#define PERM_IN prhs[1]
#define B_IN prhs[2]
#define NPARIN 3

/* ************************************************************
   PROCEDURE bwslvp - Solves U*y = b, with U(:,perm) upper triangular,
      and (implicit) unit diagonal.
   INPUT
     u    - Unit diag. upper triag. Chol. factor, n x n.
     perm - length n permutation.
     n    - order of system.
   UPDATED
     b    - Length n vector. On Input, contains the rhs.
          On output, b = y(perm).
   ************************************************************ */
void bwslvp(const double *u, const int *perm, double *b, const int n)
{
  int k;

  for(k = n-1; k > 0; k--)
    subscalarmul(b,b[k],u + perm[k] * n,k);
}

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   y = bwslvp(U,perm,b)
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  int i,j,m,n;
  double *y, *fwork;
  int *iwork;
  const double *u, *permPr, *b;
/* ------------------------------------------------------------
   Check for proper number of arguments
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("bwslvp requires more input arguments");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("bwslvp produces less output arguments");
/* ------------------------------------------------------------
   Get inputs U,perm,b.
   ------------------------------------------------------------ */
  m = mxGetM(PERM_IN) * mxGetN(PERM_IN);
  if(mxGetM(U_IN) != m || mxGetN(U_IN) != m)
    mexErrMsgTxt("U size mismatch");
  u = mxGetPr(U_IN);
  permPr = mxGetPr(PERM_IN);
  if(mxIsSparse(B_IN))                              /* b */
    mexErrMsgTxt("b should be full");
  if(mxGetM(B_IN) != m)
    mexErrMsgTxt("b size mismatch");
  n = mxGetN(B_IN);
  b = mxGetPr(B_IN);
/* ------------------------------------------------------------
   Create output y(m x n)
   ------------------------------------------------------------ */
  Y_OUT = mxCreateDoubleMatrix(m,n,mxREAL);
  y = mxGetPr(Y_OUT);
/* ------------------------------------------------------------
   Allocate working arrays iwork(m), fwork(m).
   Let iwork = perm - 1.
   ------------------------------------------------------------ */
  iwork = (int *) mxCalloc(MAX(m,1), sizeof(int));
  for(i = 0; i < m; i++){
    j = permPr[i];
    iwork[i] = --j;
  }
  fwork = (double *) mxCalloc(MAX(m,1), sizeof(double));
/* ------------------------------------------------------------
   For each column j = 1:n, solve U*y(:,j) = b(:,j)
   ------------------------------------------------------------ */
  for(j = 0; j < n; j++, b+=m, y+=m){
    memcpy(fwork, b, m * sizeof(double));
    bwslvp(u, iwork,fwork, m);
    for(i = 0; i < m; i++)
      y[iwork[i]] = fwork[i];
  }
/* ------------------------------------------------------------
   Release working arrays
   ------------------------------------------------------------ */
  mxFree(fwork);
  mxFree(iwork);
}
