/* ************************************************************
   MODULE blkaux.c  -- Several low-level subroutines for the
   mex-files in the Self-Dual-Minimization package.

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


   ************************************************************ */

#include "blksdp.h"

/* Integer compare: (for ibsearch) */
int icmp(const int *a, const int *b)
{
   return( (*a > *b) - (*a < *b)  );
}

/* ************************************************************
   PROCEDURE intbsearch - search through (in strictly ascending order)
      sorted integer array.  DIFFERENT FROM ansi BSEARCH, since it returns
      always the offset k=*pi, such that x(0:k) <= key, x(k+1:n-1) > key.
      Thus, x(k) CAN be smaller than key (viz. if key not in array).
   INPUT
     x - length n integer array
     key - integer to search for
     n - length of x
   UPDATED
     pi - On input, 0<= *pi <n; we'll search only in x[*pi,n-1].
       On output, x(0:*pi-1) < key, x(*pi:n) >= key.
       NB1: *piIN <= *piOUT <= n, so possibly *piOUT=n.
       NB2: if *piIN==n then *piOUT=n.
   RETURNS
     1 if found, 0 otherwise. If found, then x[*pi]=key.
   ************************************************************ */
int intbsearch(int *pi, const int *x, const int n, const int key)
{
 int i,j,r;

 i = *pi;
 mxAssert(i >= 0,"");
 if(i < n){
   if(x[i] < key){
     r = n;
/* ------------------------------------------------------------
   During the loop, x[i] < key and r<n => key < x[r], i.e. key has
   to be strictly
   within (i,r). Therefore, we also take j strictly in (i,r).
   ------------------------------------------------------------ */
     for(j = (i+n) / 2; j > i; j = (i+r) / 2){
       if(x[j] > key)
         r = j;                    /* new right limit */
       else if(x[j] < key)
         i = j;                    /* new left limit */
       else{
         *pi = j;                  /* Found : x[j] == key */
         return 1;
       }
     }
     *pi = r;
     return 0;              /* not found: i==j==(r-1), x[r-1] < key < x[r] */
   }
/* ------------------------------------------------------------
   If, for the initial i, x[i] >= key, then keep this i, and
   return found = (x[i] == key).
   ------------------------------------------------------------ */
   else /* x[(i = *pi)] >= key */
     return (x[i] == key);
 }
 return 0;     /* if i>=n, i.e. the list is empty */
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- r=realssqr(x,n)
   Computes r=sum(x_i^2) using loop-unrolling.
   ************************************************************ */
double realssqr(const double *x, const int n)
{
 int i;
 double r;

 r=0.0;
 for(r=0.0, i=0; i< n-3; i++){          /* LEVEL 4 */
   r+= SQR(x[i]); i++;
   r+= SQR(x[i]); i++;
   r+= SQR(x[i]); i++;
   r+= SQR(x[i]);
 }
/* ------------------------------------------------------------
   Now, i in {n-3, n-2, n-1, n}. Do the last n-i elements.
   ------------------------------------------------------------ */
 if(i < n-1){                           /* LEVEL 2 */
   r+= SQR(x[i]); i++;
   r+= SQR(x[i]); i++;
 }
 if(i < n)                              /* LEVEL 1 */
   r+= SQR(x[i]);
 return r;
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- r=realdot(x,y,n)
   Computes r=sum(x_i * y_i) using loop-unrolling.
   ************************************************************ */
double realdot(const double *x, const double *y, const int n)
{
 int i;
 double r;

 r = 0.0;
 for(i = 0; i < n-7; i++){          /* LEVEL 8 */
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i];
 }
/* ------------------------------------------------------------
   Now, i in {n-7, n-6, ..., n}. Do the last n-i elements.
   ------------------------------------------------------------ */
 if(i < n-3){                            /* LEVEL 4 */
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
 }
 if(i < n-1){                           /* LEVEL 2 */
   r+= x[i] * y[i]; i++;
   r+= x[i] * y[i]; i++;
 }
 if(i < n)                              /* LEVEL 1 */
   r+= x[i] * y[i];
 return r;
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- r=realdotrow(x,y,n)
   Computes r= x' * y(1:n:n^2)
   ************************************************************ */
double realdotrow(const double *x, const double *y, const int n)
{
 int i;
 double r;

 r=0.0;
 for(r=0.0, i=0; i< n-3; y+=n, i++){          /* LEVEL 4 */
   r+= x[i] * *y; y+=n; i++;
   r+= x[i] * *y; y+=n; i++;
   r+= x[i] * *y; y+=n; i++;
   r+= x[i] * *y;
 }
/* ------------------------------------------------------------
   Now, i in {n-3, n-2, n-1, n}. Do the last n-i elements.
   ------------------------------------------------------------ */
 if(i < n-1){                           /* LEVEL 2 */
   r+= x[i] * *y; y+=n; i++;
   r+= x[i] * *y; y+=n; i++;
 }
 if(i < n)                              /* LEVEL 1 */
   r+= x[i] * *y;
 return r;
}

/* ************************************************************
   FUNCTION selrealdot  --  r = x(sel)'*y(sel).
   INPUT: x,y -- point to two vectors
	  sel, nnz -- indices and length of selection.
   OUTPUT: dot product of x and y.
   ************************************************************ */
double selrealdot(const double *x, const double *y,
		  const int *sel, const int nnz)
{
 double r;
 int i,inz;

 for(r=0.0, inz=0;  inz < nnz-3; inz++){     /* LEVEL 4 */
   i = sel[inz++];  r += x[i] * y[i];
   i = sel[inz++];  r += x[i] * y[i];
   i = sel[inz++];  r += x[i] * y[i];
   i = sel[inz];  r += x[i] * y[i];
 }
 if(inz < nnz-1){                              /* LEVEL 2 */
   i = sel[inz++];  r += x[i] * y[i];
   i = sel[inz++];  r += x[i] * y[i];
 }
 if(inz < nnz){                                 /* LEVEL 1 */
   i = sel[inz]; r += x[i] * y[i];
 }
 return r;
}

/* ************************************************************
   PROCEDURE tril2sym -- assign R(i,j) = R(j,i) for all j>i.
   INPUT n - order of n x n matrix R.
   UPDATED r - on output, the strict lower triangular is copied
       to the strict upper triangular.
   ************************************************************ */
void tril2sym(double *r, const int n)
{
 int colp,i,j;

 /* ------------------------------------------------------------
    r points to R(:,i);     r+colp = R(:,j).
    ------------------------------------------------------------ */
 for(i=0; i<n; r += n, i++)
   for(colp = n + i, j=i+1; j<n; j++, colp += n)
     r[colp] = r[j];                          /* R(i,j) = R(j,i) */
}

/* ************************************************************
   PROCEDURE tril2herm -- Given R = [RE R, IM R],
     assign RE R(i,j) = RE R(j,i) and IM R(i,j) = - IM R(j,i) for all j>i.
   INPUT n - order of 2*(n x n) matrix R.
   UPDATED r,rpi - on output, is made Hermitian (sym, skew-sym resp).
   ************************************************************ */
void tril2herm(double *r, double *rpi, const int n)
{
  int colp,i,j;
/* ------------------------------------------------------------
   First, make the real block symmetric. Then, make the imaginary
   part skew-symmetric.
    ------------------------------------------------------------ */
  tril2sym(r,n);
/* ------------------------------------------------------------
   r points to R(:,i);     r+colp = R(:,j).
   ------------------------------------------------------------ */
  for(i = 0; i < n; rpi += n, i++){
    rpi[i] = 0.0;                                 /* zero-diagonal */
    for(colp = n + i, j = i + 1; j < n; j++, colp += n)
      rpi[colp] = -rpi[j];                        /* R(i,j) = -R(j,i) */
  }
}

/* ************************************************************
   PROCEDURE triu2sym -- assign R(j,i) = R(i,j) for all j>i.
   INPUT n - order of n x n matrix R.
   UPDATED r - on output, the strict lower triangular is copied
       to the strict upper triangular.
   ************************************************************ */
void triu2sym(double *r, const int n)
{
 int colp,i,j;

 /* ------------------------------------------------------------
    r points to R(:,i);     r+colp = R(:,j).
    ------------------------------------------------------------ */
 for(i = 0; i < n; r += n, i++)
   for(colp = n + i, j=i+1; j<n; j++, colp += n)
     r[j] = r[colp];                          /* R(j,i) = R(i,j) */
}

/* ************************************************************
   PROCEDURE triu2herm -- Given R = [RE R, IM R],
     assign RE R(i,j) = RE R(j,i) and IM R(i,j) = - IM R(j,i) for all j<i.
   INPUT n - order of 2*(n x n) matrix R.
   UPDATED r, rpi - on output, is made Hermitian (sym and skewsym resp).
   ************************************************************ */
void triu2herm(double *r, double *rpi, const int n)
{
 int colp,i,j;
 /* ------------------------------------------------------------
    First, make the real block symmetric. Then, let r point to
    the imaginary part and make that skew-symmetric.
    ------------------------------------------------------------ */
 triu2sym(r,n);
/* ------------------------------------------------------------
   rpi points to R(:,i);     rpi+colp = R(:,j).
   ------------------------------------------------------------ */
 for(i = 0; i < n; rpi += n, i++){
   rpi[i] = 0.0;                                 /* zero-diagonal */
   for(colp = n + i, j=i+1; j < n; j++, colp += n)
     rpi[j] = -rpi[colp];                          /* R(j,i) = -R(i,j) */
 }
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- scalarmul
   Computes  r = alpha * x  using loop-unrolling.
   ************************************************************ */
void scalarmul(double *r, const double alpha,const double *x,const int n)
{
 int k;

 for(k = 0; k < n-3; k++){                 /* LEVEL 4 */
   r[k] = alpha * x[k]; k++;
   r[k] = alpha * x[k]; k++;
   r[k] = alpha * x[k]; k++;
   r[k] = alpha * x[k];
 }
 if(k < n-1){                              /* LEVEL 2 */
   r[k] = alpha * x[k]; k++;
   r[k] = alpha * x[k]; k++;
 }
 if(k < n)                                 /* LEVEL 1 */
   r[k] = alpha * x[k];
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- addscalarmul
   Computes  r += alpha * x  using loop-unrolling.
   ************************************************************ */
void addscalarmul(double *r, const double alpha,const double *x,const int n)
{
 int k;

 for(k = 0; k < n-3; k++){                 /* LEVEL 4 */
   r[k] += alpha * x[k]; k++;
   r[k] += alpha * x[k]; k++;
   r[k] += alpha * x[k]; k++;
   r[k] += alpha * x[k];
 }
 if(k < n-1){                              /* LEVEL 2 */
   r[k] += alpha * x[k]; k++;
   r[k] += alpha * x[k]; k++;
 }
 if(k < n)                                 /* LEVEL 1 */
   r[k] += alpha * x[k];
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- subscalarmul(x,alpha,y,n)
   Computes x -= alpha * y using LEVEL 8 loop-unrolling.
   ************************************************************ */
void subscalarmul(double *x, const double alpha, const double *y, const int n)
{
  int i;
  
  for(i=0; i< n-7; i++){          /* LEVEL 8 */
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i];
  }
/* ------------------------------------------------------------
   Now, i in {n-7, n-6, ..., n}. Do the last n-i elements.
   ------------------------------------------------------------ */
  if(i < n-3){                           /* LEVEL 4 */
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
  }
  if(i < n-1){                           /* LEVEL 2 */
    x[i] -= alpha * y[i]; i++;
    x[i] -= alpha * y[i]; i++;
  }
  if(i < n)                              /* LEVEL 1 */
    x[i] -= alpha * y[i];
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- realHadamard
   Computes  r = x .* y  using loop-unrolling.
   ************************************************************ */
void realHadamard(double * r, const double *x, const double *y, const int n)
{
 int i;

 for(i=0; i< n-3; i++){              /* LEVEL 4 */
   r[i] = x[i] * y[i]; i++;
   r[i] = x[i] * y[i]; i++;
   r[i] = x[i] * y[i]; i++;
   r[i] = x[i] * y[i];
 }
/* ------------------------------------------------------------
   Now, i in {n-3, n-2, n-1, n}. Do the last n-i elements.
   ------------------------------------------------------------ */
 if(i < n-1){                        /* LEVEL 2 */
   r[i] = x[i] * y[i]; i++;
   r[i] = x[i] * y[i]; i++;
 }
 if(i< n)                            /* LEVEL 1 */
   r[i] = x[i] * y[i];
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- minusHadamard
   Computes  r = -x .* y  using loop-unrolling.
   ************************************************************ */
void minusHadamard(double * r, const double *x, const double *y, const int n)
{
 int i;

 for(i=0; i< n-3; i++){              /* LEVEL 4 */
   r[i] = -x[i] * y[i]; i++;
   r[i] = -x[i] * y[i]; i++;
   r[i] = -x[i] * y[i]; i++;
   r[i] = -x[i] * y[i];
 }
/* ------------------------------------------------------------
   Now, i in {n-3, n-2, n-1, n}. Do the last n-i elements.
   ------------------------------------------------------------ */
 if(i < n-1){                        /* LEVEL 2 */
   r[i] = -x[i] * y[i]; i++;
   r[i] = -x[i] * y[i]; i++;
 }
 if(i< n)                            /* LEVEL 1 */
   r[i] = -x[i] * y[i];
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- realHadarow
   Computes  r = x .* y(1:n:n^2)  using loop-unrolling.
   ************************************************************ */
void realHadarow(double * r, const double *x, const double *y, const int n)
{
 int i;

 for(i=0; i< n-3; y+=n, i++){              /* LEVEL 4 */
   r[i] = x[i] * *y; y+=n; i++;
   r[i] = x[i] * *y; y+=n; i++;
   r[i] = x[i] * *y; y+=n; i++;
   r[i] = x[i] * *y;
 }
/* ------------------------------------------------------------
   Now, i in {n-3, n-2, n-1, n}. Do the last n-i elements.
   ------------------------------------------------------------ */
 if(i < n-1){                        /* LEVEL 2 */
   r[i] = x[i] * *y; y+=n; i++;
   r[i] = x[i] * *y; y+=n; i++;
 }
 if(i< n)                            /* LEVEL 1 */
   r[i] = x[i] * *y;
}

/* ************************************************************
   TIME-CRITICAL PROCEDURE -- realHadadiv
   Computes  r = x ./ y  using loop-unrolling.
   ************************************************************ */
void realHadadiv(double * r, const double *x, const double *y, const int n)
{
 int i;

 for(i=0; i< n-3; i++){              /* LEVEL 4 */
   r[i] = x[i] / y[i]; i++;
   r[i] = x[i] / y[i]; i++;
   r[i] = x[i] / y[i]; i++;
   r[i] = x[i] / y[i];
 }
/* ------------------------------------------------------------
   Now, i in {n-3, n-2, n-1, n}. Do the last n-i elements.
   ------------------------------------------------------------ */
 if(i < n-1){                        /* LEVEL 2 */
   r[i] = x[i] / y[i]; i++;
   r[i] = x[i] / y[i]; i++;
 }
 if(i< n)                            /* LEVEL 1 */
   r[i] = x[i] / y[i];
}

/* ************************************************************
   PROCEDURE fzeros -- z = zeros(n,1)
   INPUT  n = length(z)
   OUTPUT z = zeros(n,1)
   ************************************************************ */
void fzeros(double *z,const int n)
{
  int k;
  
  for(k=0; k < n - 3; k++){           /* LEVEL 4 */
    z[k] = 0.0; k++;
    z[k] = 0.0; k++;
    z[k] = 0.0; k++;
    z[k] = 0.0;
  }
  if(k < n - 1){                      /* LEVEL 2 */
    z[k] = 0.0; k++;
    z[k] = 0.0; k++;
  }
  if(k < n)                           /* LEVEL 1 */
    z[k] = 0.0;
}

/* ************************************************************
   PROCEDURE conepars - Read cone K parameters from K-structure
   INPUT
     mxK  -  the Matlab structure "K", as passes as input argument "K_IN".
   OUTPUT
     *pK - struct where cone K parameters get stored.
   ************************************************************ */
void conepars(const mxArray *mxK, coneK *pK)
{
 const mxArray *K_FIELD;
 const int *blkstart;
 int idummy;
 char gotthem;

 if(!mxIsStruct(mxK))
   mexErrMsgTxt("Parameter `K' should be a structure.");
 if( (K_FIELD = mxGetField(mxK,0,"l")) == NULL)      /* K.l */
   pK->lpN = 0;
 else
   pK->lpN = mxGetScalar(K_FIELD);
 if( (K_FIELD = mxGetField(mxK,0,"q")) == NULL)      /* K.q */
   pK->lorN = 0;
 else{
   pK->lorN = mxGetM(K_FIELD) * mxGetN(K_FIELD);
   pK->lorNL = mxGetPr(K_FIELD);
   if(pK->lorN == 1)                                /* K.q=0 -> lorN = 0 */
     if(pK->lorNL[0] == 0.0)
       pK->lorN = 0;
 }
 if( (K_FIELD = mxGetField(mxK,0,"r")) == NULL)      /* K.r */
   pK->rconeN = 0;
 else{
   pK->rconeN = mxGetM(K_FIELD) * mxGetN(K_FIELD);
   pK->rconeNL = mxGetPr(K_FIELD);
   if(pK->rconeN == 1)                                /* K.r=0 -> rconeN = 0 */
     if(pK->rconeNL[0] == 0.0)
       pK->rconeN = 0;
 }
 if( (K_FIELD = mxGetField(mxK,0,"s")) == NULL){     /* K.s */
   pK->sdpN = 0;
 }
 else{
   pK->sdpN = mxGetM(K_FIELD) * mxGetN(K_FIELD);
   pK->sdpNL = mxGetPr(K_FIELD);
   if(pK->sdpN == 1)                                /* K.s=0 -> sdpN = 0 */
     if(pK->sdpNL[0] == 0.0)
       pK->sdpN = 0;
 }
 if( (K_FIELD = mxGetField(mxK,0,"rsdpN")) == NULL)      /* K.rsdpN */
   pK->rsdpN = pK->sdpN;                           /* default to all real */
 else
   if((pK->rsdpN = mxGetScalar(K_FIELD)) > pK->sdpN)
     mexErrMsgTxt("K.rsdpN mismatches K.s");
 /* --------------------------------------------------
    GET STATISTICS: try to read from K, otherwise compute them.
    -------------------------------------------------- */
 gotthem = 0;
 if( (K_FIELD = mxGetField(mxK,0,"rLen")) != NULL){      /* K.rLen */
   pK->rLen = mxGetScalar(K_FIELD);
   if( (K_FIELD = mxGetField(mxK,0,"hLen")) != NULL){      /* K.hLen */
     pK->hLen = mxGetScalar(K_FIELD);
     if( (K_FIELD = mxGetField(mxK,0,"qMaxn")) != NULL){      /* K.qMaxn */
       pK->qMaxn = mxGetScalar(K_FIELD);
       if( (K_FIELD = mxGetField(mxK,0,"rMaxn")) != NULL){      /* K.rMaxn */
	 pK->rMaxn = mxGetScalar(K_FIELD);
	 if( (K_FIELD = mxGetField(mxK,0,"hMaxn")) != NULL){    /* K.hMaxn */
	   pK->hMaxn = mxGetScalar(K_FIELD);
	   if( (K_FIELD = mxGetField(mxK,0,"blkstart"))!=NULL){ /*K.blkstart*/
	     if(!mxIsSparse(K_FIELD))
	       mexErrMsgTxt("K.blkstart must be a sparse matrix.");
	     blkstart = mxGetIr(K_FIELD);
	     ++ blkstart;
	     pK->qDim = blkstart[pK->lorN] - blkstart[0];
	     blkstart += pK->lorN;
	     pK->rDim = blkstart[pK->rsdpN] - blkstart[0];
	     pK->hDim = blkstart[pK->sdpN] - blkstart[pK->rsdpN];
	     gotthem = 1;
	   }
	 }
       }
     }
   }
 }
 if(!gotthem){
   someStats(&(pK->qMaxn), &(pK->qDim), &idummy, pK->lorNL, pK->lorN);
   someStats(&(pK->rMaxn), &(pK->rLen), &(pK->rDim), pK->sdpNL, pK->rsdpN);
   someStats(&(pK->hMaxn), &(pK->hLen), &(pK->hDim), pK->sdpNL+pK->rsdpN,
	     (pK->sdpN) - (pK->rsdpN));
   pK->hDim *= 2;
 }
}

/* ************************************************************
   PROCEDURE someStats  --  Computes maximum, sum and sum of squares
   INPUT
   x, n - length n vector
   OUTPUT
   xmax, xsum, xssqr - Maximum, sum total and sum of squares
   IMPORTANT: this routine is especially designed for use with the
    blk.s structure, which contains nonneg integers stored as doubles.
   ************************************************************ */
void someStats(int *pxmax, int *pxsum, int *pxssqr,
	       const double *x, const int n)
{
 int xi, xmax, xsum, xssqr;
 int i;

 xmax = 0;             /* assume that all integers are nonnegative */
 xsum = 0; xssqr = 0;
 for(i = 0; i < n; i++){
   xi = x[i];
   xmax = MAX(xmax, xi);
   xsum += xi;
   xssqr += SQR(xi);
 }
 *pxmax = xmax;
 *pxsum = xsum;
 *pxssqr = xssqr;
}

/* ************************************************************
   PROCEDURE qlmul : LORENTZ SCALE z = D(x)y (full version)
     z=D(x)y = [x'*y / sqrt(2);  alpha * x(2:n) + rdetx * y(2:n)],
     where alpha = (z1+rdetx*y1) / (x(1)+ sqrt(2) * rdetx)
   INPUT
     x,y - full n x 1
     rdetx - sqrt(det(x))
     n - order of x,y,z.
   OUTPUT
     z - full n x 1. Let z := D(x)y.
   ************************************************************ */
void qlmul(double *z,const double *x,const double *y,
	   const double rdetx,const int n)
{
 double alpha,z1;
  /* ------------------------------------------------------------
     z1 = x'*y / sqrt(2),
     alpha = (z1+rdetx*y1) / (x(1)+ sqrt(2) * rdetx)
     ------------------------------------------------------------ */
 z1 = realdot(x,y,n) / M_SQRT2;
 alpha = (z1 + rdetx * y[0]) / (x[0] + M_SQRT2 * rdetx);
 /* ------------------------------------------------------------
    z(1) = z1, z(2:n) = alpha * x(2:n) + rdetx * y(2:n).
    ------------------------------------------------------------ */
 z[0] = z1;
 scalarmul(z+1,alpha,x+1,n-1);
 addscalarmul(z+1,rdetx,y+1,n-1);
}

/* ************************************************************
   PROCEDURE qldiv : LORENTZ SCALE z = D(x)\y (full version)
    D(x)\y = (1/det x) * [x'Jy/sqrt(2); rdetx * y2-alpha*x2],
    where alpha = (x'Jy/sqrt(2) + rdetx*y1) / (x(1)+ sqrt(2) * rdetx)
   INPUT
     x,y - full n x 1
     rdetx - sqrt(det(x))
     n - order of x,y,z.
   OUTPUT
     z - full n x 1. Let z := D(x)^{-1}y.
   ************************************************************ */
void qldiv(double *z,const double *x,const double *y,
	   const double rdetx,const int n)
{
 double alpha,x1,y1,z1;
/* ------------------------------------------------------------
   z1 = x'*J*y / (sqrt(2) * det x),
   alpha = (z1+y1/rdetx) / (x(1)+ sqrt(2) * rdetx)
   ------------------------------------------------------------ */
 x1 = x[0]; y1 = y[0];
 z1 = (x1*y1 - realdot(x+1,y+1,n-1)) / (M_SQRT2 * SQR(rdetx));
 alpha = (z1 + y1 / rdetx) / (x1 + M_SQRT2 * rdetx);
/* ------------------------------------------------------------
   z(1) = z1, z(2:n) = y(2:n)/rdetx - alpha * x(2:n).
   ------------------------------------------------------------ */
 z[0] = z1;
 scalarmul(z+1,-alpha,x+1,n-1);
 addscalarmul(z+1,1/rdetx,y+1,n-1);
}

/* ************************************************************
   PROCEDURE uperm - Let triu(Y) = triu(U(:,perm)),
                     leaving tril(Y,-1) undefined.
   INPUT
     u - nxn input matrix
     perm - length n pivot ordering
     n -   order
   OUTPUT
     y - triu(y) = triu(u(:,perm).
   ************************************************************ */
void uperm(double *y, const double *u, const int *perm, const int n)
{
  int j;
  const double *uj;

  for(j = 0; j < n; y += n){
    uj = u + perm[j] * n;
    memcpy(y, uj, (++j) * sizeof(double));
  }
}

/* ************************************************************
   PROCEDURE dodiscard - discard selected subscripts from selected
      columns.
   INPUT
     xjc - column starts (can be more than m)
     perm - Length m, selection of columns in At.
     discard - discard[xir[inz]] is 1 iff subscript xir[inz] should be
       discarded.
     m - length of perm and xnnz (number of remaining columns).
   UPDATED
     xir - Input: subscript columns, starting at Atjc[perm[j]], of length
       xnnz[j] <= Atjc[perm[j]+1]-Atjc[perm[j]].
       Output: xir(!discard(xir)) adjusted from start of columns
     xnnz - The length of column j, updated after discarding. Length m.
   ************************************************************ */
void dodiscard(int *xir,int *xnnz, const int *xjc,
              const int *perm,const char *discard, const int m)
{
  int i,j,nnzj, inz,jnz;
  int *xj;

  for(j = 0; j < m; j++){
    xj = xir + xjc[perm[j]];
    nnzj = xnnz[j];
    for(inz = 0, jnz = 0; inz < nnzj; inz++)
      if(!discard[(i = xj[inz])])
        xj[jnz++] = i;
    xnnz[j] = jnz;
  }
}
