/*
 [y,y0] = backward(lab,p,fi,b)

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <math.h>
#include <string.h>
#include "mex.h"
#include "blksdp.h"

/*  [y,y0] = backward(lab,p,fi,b) */
#define Y_OUT plhs[0]
#define Y0_OUT plhs[1]

#define LAB_IN prhs[0]
#define P_IN prhs[1]
#define FI_IN prhs[2]
#define B_IN prhs[3]

/* ************************************************************
   PROCEDURE innerbw
   INPUT
     lab  - length m vector: diagonal entries.
     p    - length m vector: dense column
     fi   - order m+1 sparse vector, with nonzeros positions where
        we pivot on the dense column.
     b    - order m + 1 vector: the right hand side. b[m] is for the
        bottom, special row. The upper-factor is e_{m+1}' in the last
        row, so the function returns y0 = b[m].
     m    - order of lab.
   OUTPUT
     y    - length m vector.
   RETURNS y0 = b[m].
   ************************************************************ */
double innerbw(double *y, const double *lab,const double *p,const jcir fi,
               const double *b,const int m)
{
  int inz,i,j;
  double y0,phi,fiy0;

/* ------------------------------------------------------------
   BACKWARD solve:
   [ diag(lab), fi(prev)*p;  0', 1] * [y; y0] = [b; b(m+1)]
   ------------------------------------------------------------ */
  y0 = b[m];
  inz = fi.jc[1];
  phi = fi.pr[--inz];              /* fi[m] */
  for(i = m-1; inz > 0; i--){
/* ------------------------------------------------------------
   Let j be the column on which the previous dense-column pivot was done;
   thus "y0" is actually y[j].
   ------------------------------------------------------------ */
    j = fi.ir[--inz];
    fiy0 = phi * y0;
    for( ; i > j; i--)
      y[i] = (b[i] - fiy0 * p[i]) / lab[i];
    y[i] = y0;
/* ------------------------------------------------------------
   Let phi be the preceding fi (which will finally be 1),
   and solve for the associated "y0", from
   [lab(j),  fi*p(j)] * [yj, y0] = b(j).
   ------------------------------------------------------------ */
    phi = fi.pr[inz];
    y0 = (b[i] - y0 * lab[i]) / (phi * p[i]);
  }
/* ------------------------------------------------------------
   Solve the top yi's from  [diag(lab), p]*[y;y0] = b.
   ------------------------------------------------------------ */
  for( ; i >= 0; i--)
    y[i] = (b[i] - y0 * p[i]) / lab[i];
  return y0;
}


/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   [y,y0] = backward(lab,p,fi,b)
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  int m,j,N;
  const double *lab,*p,*b;
  jcir fi;
  double *y, *y0;

/* ------------------------------------------------------------
   Check for proper number of arguments
   ------------------------------------------------------------ */
  if(nrhs < 4)
    mexErrMsgTxt("backward requires 4 input arguments.");
  if(nlhs > 2)
    mexErrMsgTxt("backward generates 2 output arguments.");
 /* ------------------------------------------------------------
    Get input vectors lab, p, fi, b
    ------------------------------------------------------------ */
  m = mxGetM(LAB_IN) * mxGetN(LAB_IN);
  lab = mxGetPr(LAB_IN);
  if(m != mxGetM(P_IN) * mxGetN(P_IN))
    mexErrMsgTxt("p size mismatch.");
  p = mxGetPr(P_IN);
  b = mxGetPr(B_IN);
  if(mxGetM(B_IN) != m+1)
    mexErrMsgTxt("b size mismatch.");
  N = mxGetN(B_IN);
  if(!mxIsSparse(FI_IN))
    mexErrMsgTxt("fi should be sparse.");
  if(mxGetM(FI_IN) != m+1 || mxGetN(FI_IN) != 1)
    mexErrMsgTxt("fi size mismatch.");
  fi.jc = mxGetJc(FI_IN);
  fi.ir = mxGetIr(FI_IN);
  fi.pr = mxGetPr(FI_IN);
/* ------------------------------------------------------------
   Create output matrix y(m,N), y0(1,N)
   ------------------------------------------------------------ */
  Y_OUT = mxCreateDoubleMatrix(m, N, mxREAL);
  y = mxGetPr(Y_OUT);
  Y0_OUT = mxCreateDoubleMatrix(1, N, mxREAL);
  y0 = mxGetPr(Y0_OUT);
/* ------------------------------------------------------------
   Solve U(lab,p) * y = b
   ------------------------------------------------------------ */
  for(j = 0; j < N; j++, b+=(m+1), y+=m)
    y0[j] = innerbw(y, lab,p,fi,b, m);
}

