/*
 Ad = Adenscale(Aden, Ablkq, qloc, dencols, d, rdetd)

    This file is part of SeDuMi 1.03BETA
    Copyright (C) 1999 Jos F. Sturm
    Dept. Quantitative Economics, Maastricht University, the Netherlands.
    Affiliations up to SeDuMi 1.02 (AUG1998):
      CRL, McMaster University, Canada.
      Supported by the Netherlands Organization for Scientific Research (NWO).

  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <string.h>
#include "mex.h"
#include "blksdp.h"

#define AD_OUT plhs[0]
#define NPAROUT 1

#define ADEN_IN prhs[0]
#define ABLKQ_IN prhs[1]
#define QLOC_IN prhs[2]
#define DENCOLS_IN prhs[3]
#define D_IN prhs[4]
#define RDETD_IN prhs[5]
#define ADOTD_IN prhs[6]
#define NPARIN 7

/* ============================================================
   MAIN: MEXFUNCTION
   ============================================================ */
/* ************************************************************
   PROCEDURE mexFunction - Entry for Matlab
   ************************************************************ */
void mexFunction(const int nlhs, mxArray *plhs[],
  const int nrhs, const mxArray *prhs[])
{
  int i,inz, nj, j, k, m,nden, nqden;
  const int *adenjc, *adenir, *Ablkqjc, *Ablkqir, *adotdjc, *adotdir;
  int *qloc, *dencols, *adjc, *adir;
  double dj;
  const double *d, *adenpr, *qlocPr, *dencolsPr, *rdetd, *adotdpr;
  double *ad, *adq, *fwork;
/* ------------------------------------------------------------
   Check for proper number of arguments
   ------------------------------------------------------------ */
  if(nrhs < NPARIN)
    mexErrMsgTxt("adenscale requires more input arguments");
  if(nlhs > NPAROUT)
    mexErrMsgTxt("adenscale produces less output arguments");
/* ------------------------------------------------------------
   Get inputs Aden, Ablkq
   ------------------------------------------------------------ */
  if(!mxIsSparse(ADEN_IN))                            /* Aden */
    mexErrMsgTxt("Aden must be sparse");
  m = mxGetM(ADEN_IN);
  nden = mxGetN(ADEN_IN);
  adenjc = mxGetJc(ADEN_IN);
  adenir = mxGetIr(ADEN_IN);
  adenpr = mxGetPr(ADEN_IN);
  if(!mxIsSparse(ABLKQ_IN))                            /* Ablkq */
    mexErrMsgTxt("Ablkq must be sparse");
  nqden = mxGetN(ABLKQ_IN);
  if((nqden > 0) && (m != mxGetM(ABLKQ_IN)))
    mexErrMsgTxt("Size mismatch Ablkq");
  Ablkqjc = mxGetJc(ABLKQ_IN);
  Ablkqir = mxGetIr(ABLKQ_IN);
/* ------------------------------------------------------------
   Get inputs qloc, dencols, d, rdetd, adotd
   ------------------------------------------------------------ */
  if(nqden != mxGetM(QLOC_IN) * mxGetN(QLOC_IN))
    mexErrMsgTxt("Size mismatch qloc");
  qlocPr = mxGetPr(QLOC_IN);
  if(nden != mxGetM(DENCOLS_IN) * mxGetN(DENCOLS_IN))
    mexErrMsgTxt("Size mismatch dencols");
  dencolsPr = mxGetPr(DENCOLS_IN);
  d = mxGetPr(D_IN);
  if(nqden != mxGetM(RDETD_IN) * mxGetN(RDETD_IN))
    mexErrMsgTxt("Size mismatch rdetd");
  rdetd = mxGetPr(RDETD_IN);
  if(!mxIsSparse(ADOTD_IN))                            /* adotd */
    mexErrMsgTxt("adotd must be sparse");
  if((m != mxGetM(ADOTD_IN) && nqden > 0) || nqden != mxGetN(ADOTD_IN))
    mexErrMsgTxt("Size mismatch adotd");
  adotdjc = mxGetJc(ADOTD_IN);
  adotdir = mxGetIr(ADOTD_IN);
  adotdpr = mxGetPr(ADOTD_IN);
/* ------------------------------------------------------------
   Create working arrays qloc(nqden+1), dencols(nden),
   fwork(m)
   ------------------------------------------------------------ */
  qloc    = (int *) mxCalloc(nqden + 1, sizeof(int));
  dencols = (int *) mxCalloc(MAX(1,nden), sizeof(int));
  fwork = (double *) mxCalloc(m, sizeof(double));
/* ------------------------------------------------------------
   Convert to integer C-style; let qloc = [qlocPr, nden].
   ------------------------------------------------------------ */
  for(i = 0; i < nqden; i++){
    j = qlocPr[i];
    qloc[i] = --j;
  }
  qloc[nqden] = nden;
  for(i = 0; i < nden; i++){
    j = dencolsPr[i];
    dencols[i] = --j;
  }
/* ------------------------------------------------------------
   Create output: AD = sparse(m,nden+nqden, nnz(ADEN)+nnz(Ablkq)).
   ------------------------------------------------------------ */
  AD_OUT = mxCreateSparse(m,nden + nqden,
                          adenjc[nden] + Ablkqjc[nqden], mxREAL);
  adjc = mxGetJc(AD_OUT);
  adir = mxGetIr(AD_OUT);
  ad = mxGetPr(AD_OUT);                  /* points to ADEN part */
/* ------------------------------------------------------------
   Fill sparsity structure of AD by ADEN
   ------------------------------------------------------------ */
  memcpy(adir, adenir, adenjc[nden] * sizeof(int));
  memcpy(adjc, adenjc, (nden + 1) * sizeof(int));
/* ------------------------------------------------------------
   Structure of the Ablkq part
   ------------------------------------------------------------ */
  adjc += nden;
  inz = adjc[0];
  memcpy(adir + inz, Ablkqir, Ablkqjc[nqden] * sizeof(int));
  for(i = 1; i <= nqden; i++)
    adjc[i] = Ablkqjc[i] + inz;
/* ------------------------------------------------------------
   LP: AD(:,j) = d(dencols(j)) * Aden(:,j)
   ------------------------------------------------------------ */
  inz = 0;
  for(j = 0; j < qloc[0]; j++){
    nj = adenjc[j+1] - inz;
    scalarmul(ad, d[dencols[j]], adenpr + inz, nj);
    inz += nj, ad += nj;
  }
/* ------------------------------------------------------------
   LORENTZ: AD(:,qloc(k):qloc(k+1)-1) = rdetd(k) * Aden(:,qloc(k):qloc(k+1)-1)
   ------------------------------------------------------------ */
  mxAssert(inz == adenjc[qloc[0]],"");
  for(k = 0; k < nqden; k++){
    j = qloc[k+1];
    nj = adenjc[j] - inz;
    scalarmul(ad, rdetd[k], adenpr + inz, nj);
    inz += nj; ad += nj;
  }
/* ------------------------------------------------------------
   "ad" now points to "adq" part - so Ablkq.{jc,ir} points into ad.
   LORENTZ: ADq(:,k) = Aden(:,jL) * d( dencols(jL) ),
   where jL = qloc(k):qloc(k+1)-1.
   We let (j,inz) walk thru Aden, from the first Lorentz column.
   fwork is used for ADq(:,k) in full.
   ------------------------------------------------------------ */
  j = qloc[0];
  inz = adenjc[j];
  for(k = 0; k < nqden; k++){
    for(i = Ablkqjc[k]; i < Ablkqjc[k+1]; i++)        /* fwork = all-0 */
      fwork[Ablkqir[i]] = 0.0;
/* ------------------------------------------------------------
   Init with contribution of sparse columns in dense Lorentz block k, i.e.
   let Adq(:,k) = adotd(:,k) = A(:,sparse cols in q-block k) * d(q-block k)
   ------------------------------------------------------------ */
    for(i = adotdjc[k]; i < adotdjc[k+1]; i++)
      fwork[adotdir[i]] = adotdpr[i];
/* ------------------------------------------------------------
   Add contribution of dense columns, i.e.
   let ADq(:,k) += Aden(:,jL) * d( dencols(jL) )
   ------------------------------------------------------------ */
    for(; j < qloc[k+1]; j++){                    /* fwork += dj * Aden(:,j) */
      dj = d[dencols[j]];
      for(; inz < adenjc[j+1]; inz++)
        fwork[adenir[inz]] += dj * adenpr[inz];
    }
    for(i = Ablkqjc[k]; i < Ablkqjc[k+1]; i++)     /* ADq(:,k) = fwork */
      ad[i] = fwork[Ablkqir[i]];
  }
/* ------------------------------------------------------------
   Release working arrays
   ------------------------------------------------------------ */
  mxFree(fwork);
  mxFree(dencols);
  mxFree(qloc);
}
