/**
 * 3 Band EQ VST Plugin
 *
 * Copyright (C) 2010 Simo Srkk (STS)
 *
 * $Id: sts3bandeq_vst.cpp,v 1.4 2010/01/09 20:33:47 ssarkka Exp $
 */

#include <cstdio>
#include <cmath>
#include "sts3bandeq_vst.h"

AudioEffect* createEffectInstance (audioMasterCallback audioMaster)
{
	return new Sts3BandEQ_VST (audioMaster);
}

Sts3BandEQ_VST::Sts3BandEQ_VST (audioMasterCallback audioMaster)
    : AudioEffectX (audioMaster, 1, 15), // 1 program, 9 parameters
      m_eq1(44100), m_eq2(44100) 
{
	setNumInputs (2);		// mono in
	setNumOutputs (2);		// mono out
	setUniqueID ('S3EQ');	// identify
	isSynth(false);
	canProcessReplacing ();	// supports replacing output
	canDoubleReplacing ();	// supports double precision processing

	vst_strncpy (programName, "Default", kVstMaxProgNameLen);	// default program name
}

Sts3BandEQ_VST::~Sts3BandEQ_VST ()
{
	// nothing to do here
}

void Sts3BandEQ_VST::setProgramName (char* name)
{
	vst_strncpy (programName, name, kVstMaxProgNameLen);
}

void Sts3BandEQ_VST::getProgramName (char* name)
{
	vst_strncpy (name, programName, kVstMaxProgNameLen);
}

#define LF_F1  20.0
#define LF_F2  500.0
#define LMF_F1 40.0
#define LMF_F2 1000.0
#define MF_F1  125.0
#define MF_F2  8000.0
#define HMF_F1 200.0
#define HMF_F2 18000.0
#define HF_F1  1800.0
#define HF_F2  20000.0

void Sts3BandEQ_VST::setParameter (VstInt32 index, float value)
{
    switch (index) {
    case EQP_LF_Q:
	m_eq1.getLF().setQuality(0.1 + 9.9*value);
	m_eq2.getLF().setQuality(0.1 + 9.9*value);
	break;

    case EQP_LF_F:
	m_eq1.getLF().setFrequency(LF_F1 + (LF_F2 - LF_F1) * value);
	m_eq2.getLF().setFrequency(LF_F1 + (LF_F2 - LF_F1) * value);
	break;

    case EQP_LF_DB:
	m_eq1.getLF().setLevel(40*value - 20);
	m_eq2.getLF().setLevel(40*value - 20);
	break;


    case EQP_LMF_Q:
	m_eq1.getLMF().setQuality(0.1 + 9.9*value);
	m_eq2.getLMF().setQuality(0.1 + 9.9*value);
	break;

    case EQP_LMF_F:
	m_eq1.getLMF().setFrequency(LMF_F1 + (LMF_F2 - LMF_F1) * value);
	m_eq2.getLMF().setFrequency(LMF_F1 + (LMF_F2 - LMF_F1) * value);
	break;

    case EQP_LMF_DB:
	m_eq1.getLMF().setLevel(40*value - 20);
	m_eq2.getLMF().setLevel(40*value - 20);
	break;


    case EQP_MF_Q:
	m_eq1.getMF().setQuality(0.1 + 9.9*value);
	m_eq2.getMF().setQuality(0.1 + 9.9*value);
	break;

    case EQP_MF_F:
	m_eq1.getMF().setFrequency(MF_F1 + (MF_F2 - MF_F1) * value);
	m_eq2.getMF().setFrequency(MF_F1 + (MF_F2 - MF_F1) * value);
	break;

    case EQP_MF_DB:
	m_eq1.getMF().setLevel(40*value - 20);
	m_eq2.getMF().setLevel(40*value - 20);
	break;


    case EQP_HMF_Q:
	m_eq1.getHMF().setQuality(0.1 + 9.9*value);
	m_eq2.getHMF().setQuality(0.1 + 9.9*value);
	break;

    case EQP_HMF_F:
	m_eq1.getHMF().setFrequency(HMF_F1 + (HMF_F2 - HMF_F1) * value);
	m_eq2.getHMF().setFrequency(HMF_F1 + (HMF_F2 - HMF_F1) * value);
	break;

    case EQP_HMF_DB:
	m_eq1.getHMF().setLevel(40*value - 20);
	m_eq2.getHMF().setLevel(40*value - 20);
	break;


    case EQP_HF_Q:
	m_eq1.getHF().setQuality(0.1 + 9.9*value);
	m_eq2.getHF().setQuality(0.1 + 9.9*value);
	break;

    case EQP_HF_F:
	m_eq1.getHF().setFrequency(HF_F1 + (HF_F2 - HF_F1) * value);
	m_eq2.getHF().setFrequency(HF_F1 + (HF_F2 - HF_F1) * value);
	break;

    case EQP_HF_DB:
	m_eq1.getHF().setLevel(40*value - 20);
	m_eq2.getHF().setLevel(40*value - 20);
	break;

    default:
	break;
    }
}

float Sts3BandEQ_VST::getParameter (VstInt32 index)
{
    switch (index) {
    case EQP_LF_Q:
	return (float)((m_eq1.getLF().getQuality() - 0.1) / 9.9);
	break;

    case EQP_LF_F:
	return (float)((m_eq1.getLF().getFrequency() - LF_F1)
		       / (LF_F2 - LF_F1));
	break;

    case EQP_LF_DB:
	return (float)((m_eq1.getLF().getLevel() + 20) / 40);
	break;


    case EQP_LMF_Q:
	return (float)((m_eq1.getLMF().getQuality() - 0.1) / 9.9);
	break;

    case EQP_LMF_F:
	return (float)((m_eq1.getLMF().getFrequency() - LMF_F1)
		       / (LMF_F2 - LMF_F1));
	break;

    case EQP_LMF_DB:
	return (float)((m_eq1.getLMF().getLevel() + 20) / 40);
	break;


    case EQP_MF_Q:
	return (float)((m_eq1.getMF().getQuality() - 0.1) / 9.9);
	break;

    case EQP_MF_F:
	return (float)((m_eq1.getMF().getFrequency() - MF_F1)
		       / (MF_F2 - MF_F1));
	break;

    case EQP_MF_DB:
	return (float)((m_eq1.getMF().getLevel() + 20) / 40);
	break;


    case EQP_HMF_Q:
	return (float)((m_eq1.getHMF().getQuality() - 0.1) / 9.9);
	break;

    case EQP_HMF_F:
	return (float)((m_eq1.getHMF().getFrequency() - HMF_F1)
		       / (HMF_F2 - HMF_F1));
	break;

    case EQP_HMF_DB:
	return (float)((m_eq1.getHMF().getLevel() + 20) / 40);
	break;


    case EQP_HF_Q:
	return (float)((m_eq1.getHF().getQuality() - 0.1) / 9.9);
	break;

    case EQP_HF_F:
	return (float)((m_eq1.getHF().getFrequency() - HF_F1)
		       / (HF_F2 - HF_F1));
	break;

    case EQP_HF_DB:
	return (float)((m_eq1.getHF().getLevel() + 20) / 40);
	break;


    default:
	return 0;
	break;
    }
}

void Sts3BandEQ_VST::getParameterName (VstInt32 index, char* label)
{
    switch (index) {
    case EQP_LF_Q:
	vst_strncpy (label, "LF Quality", kVstMaxParamStrLen);
	break;

    case EQP_LF_F:
	vst_strncpy (label, "LF Frequency", kVstMaxParamStrLen);
	break;

    case EQP_LF_DB:
	vst_strncpy (label, "LF Level", kVstMaxParamStrLen);
	break;


    case EQP_LMF_Q:
	vst_strncpy (label, "LMF Quality", kVstMaxParamStrLen);
	break;

    case EQP_LMF_F:
	vst_strncpy (label, "LMF Frequency", kVstMaxParamStrLen);
	break;

    case EQP_LMF_DB:
	vst_strncpy (label, "LMF Level", kVstMaxParamStrLen);
	break;


    case EQP_MF_Q:
	vst_strncpy (label, "MF Quality", kVstMaxParamStrLen);
	break;

    case EQP_MF_F:
	vst_strncpy (label, "MF Frequency", kVstMaxParamStrLen);
	break;

    case EQP_MF_DB:
	vst_strncpy (label, "MF Level", kVstMaxParamStrLen);
	break;


    case EQP_HMF_Q:
	vst_strncpy (label, "HMF Quality", kVstMaxParamStrLen);
	break;

    case EQP_HMF_F:
	vst_strncpy (label, "HMF Frequency", kVstMaxParamStrLen);
	break;

    case EQP_HMF_DB:
	vst_strncpy (label, "HMF Level", kVstMaxParamStrLen);
	break;


    case EQP_HF_Q:
	vst_strncpy (label, "HF Quality", kVstMaxParamStrLen);
	break;

    case EQP_HF_F:
	vst_strncpy (label, "HF Frequency", kVstMaxParamStrLen);
	break;

    case EQP_HF_DB:
	vst_strncpy (label, "HF Level", kVstMaxParamStrLen);
	break;


    default:
	vst_strncpy (label, "", kVstMaxParamStrLen);
	break;
    }
}

void Sts3BandEQ_VST::getParameterDisplay (VstInt32 index, char* text)
{
    switch (index) {
    case EQP_LF_Q:
	float2string ((float)m_eq1.getLF().getQuality(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_LF_F:
	float2string ((float)m_eq1.getLF().getFrequency(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_LF_DB:
	float2string ((float)m_eq1.getLF().getLevel(),
		      text, kVstMaxParamStrLen);
	break;


    case EQP_LMF_Q:
	float2string ((float)m_eq1.getLMF().getQuality(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_LMF_F:
	float2string ((float)m_eq1.getLMF().getFrequency(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_LMF_DB:
	float2string ((float)m_eq1.getLMF().getLevel(),
		      text, kVstMaxParamStrLen);
	break;


    case EQP_MF_Q:
	float2string ((float)m_eq1.getMF().getQuality(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_MF_F:
	float2string ((float)m_eq1.getMF().getFrequency(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_MF_DB:
	float2string ((float)m_eq1.getMF().getLevel(),
		      text, kVstMaxParamStrLen);
	break;


    case EQP_HMF_Q:
	float2string ((float)m_eq1.getHMF().getQuality(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_HMF_F:
	float2string ((float)m_eq1.getHMF().getFrequency(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_HMF_DB:
	float2string ((float)m_eq1.getHMF().getLevel(),
		      text, kVstMaxParamStrLen);
	break;


    case EQP_HF_Q:
	float2string ((float)m_eq1.getHF().getQuality(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_HF_F:
	float2string ((float)m_eq1.getHF().getFrequency(),
		      text, kVstMaxParamStrLen);
	break;

    case EQP_HF_DB:
	float2string ((float)m_eq1.getHF().getLevel(),
		      text, kVstMaxParamStrLen);
	break;


    default:
	break;
    }
}

void Sts3BandEQ_VST::getParameterLabel (VstInt32 index, char* label)
{
    switch (index) {
    case EQP_LF_Q:
    case EQP_LMF_Q:
    case EQP_MF_Q:
    case EQP_HMF_Q:
    case EQP_HF_Q:
	vst_strncpy (label, "", kVstMaxParamStrLen);
	break;

    case EQP_LF_F:
    case EQP_LMF_F:
    case EQP_MF_F:
    case EQP_HMF_F:
    case EQP_HF_F:
	vst_strncpy (label, "Hz", kVstMaxParamStrLen);
	break;

    case EQP_LF_DB:
    case EQP_LMF_DB:
    case EQP_MF_DB:
    case EQP_HMF_DB:
    case EQP_HF_DB:
	vst_strncpy (label, "dB", kVstMaxParamStrLen);
	break;

    default:
	vst_strncpy (label, "", kVstMaxParamStrLen);
	break;
    }
}

bool Sts3BandEQ_VST::getEffectName (char* name)
{
	vst_strncpy (name, "Sts3BandEQ", kVstMaxEffectNameLen);
	return true;
}

bool Sts3BandEQ_VST::getProductString (char* text)
{
	vst_strncpy (text, "Sts3BandEQ", kVstMaxProductStrLen);
	return true;
}

bool Sts3BandEQ_VST::getVendorString (char* text)
{
	vst_strncpy (text, "SS", kVstMaxVendorStrLen);
	return true;
}

VstInt32 Sts3BandEQ_VST::getVendorVersion ()
{ 
	return 1100; 
}

void Sts3BandEQ_VST::processReplacing (float** inputs,
				      float** outputs,
				      VstInt32 sampleFrames)
{
    float* in1  =  inputs[0];
    float* in2  =  inputs[1];
    float* out1 = outputs[0];
    float* out2 = outputs[1];

    while (--sampleFrames >= 0)
    {
        (*out1++) =  m_eq1.process((*in1++));
        (*out2++) =  m_eq2.process((*in2++));
    }
}

void Sts3BandEQ_VST::processDoubleReplacing (double** inputs,
					    double** outputs,
					    VstInt32 sampleFrames)
{
    double* in1  =  inputs[0];
    double* in2  =  inputs[1];
    double* out1 = outputs[0];
    double* out2 = outputs[1];

    while (--sampleFrames >= 0)
    {
        (*out1++) =  m_eq1.process((*in1++));
        (*out2++) =  m_eq2.process((*in2++));
    }
}

