/**
 * 3 Band EQ based on Shannon interpolation and
 * continuous-time filtering concept.
 *
 * Copyright (C) 2010 Simo Srkk (STS)
 *
 * $Id: sts3bandeq_core.cc,v 1.4 2010/01/09 12:24:44 ssarkka Exp $
 */

#include "sts3bandeq_core.h"

/**
 * Print all the filter parameters to stdout.
 * Note that this is included into executable only
 * if symbol STS3BANDEQ_PRINT is defined.
 */
#ifdef STS3BANDEQ_PRINT
void Sts2ndSection::print(int mask) const {

    printf("BEGIN ");
    switch (m_type) {
    case SECTYPE_LOWSHELF:  printf("LOWSHELF Filter\n"); break;
    case SECTYPE_PEAK:      printf("PEAK Filter\n"); break;
    case SECTYPE_HIGHSHELF: printf("HIGHSHELF Filter\n"); break;
    default:                printf("UNKNOWN Filter\n"); break;
    }

    if (mask & PRINTMASK_PARAM) {
	printf("  fs = %g\n", 1/m_T);
	printf("  db = %g\n", m_db);
	printf("   q = %g\n", m_q);
	printf("   f = %g\n", m_f);
	printf("   A = [%.15f  %.15f;\n        %.15f  %.15f]\n",
	       m_ta11, m_ta12, m_ta21, m_ta22);

	for (int i = 0; i < 2*INPUT_COUNT+1; i++) {
	    printf("   B%d = [%.15f;  %.15f]\n", i+1, m_bl1[i], m_bl2[i]);
	}
    }    

    if (mask & PRINTMASK_STATE) {
	printf("  x1 = %.15f\n", m_x1);
	printf("  x2 = %.15f\n", m_x2);
    }

    if (mask & PRINTMASK_INPUT) {
	m_delayline.print(false);
    }

    printf("END\n");
}
#endif

/**
 * Evaluate matrix exponential expm(t*w0*[-a1 1; -a2 0])
 * and store the results to m_ta11, m_ta12, m_ta21, m_ta22.
 * The other values than t are taken from instance variables.
 * 
 * @param t The time parameter
 */
void Sts2ndSection::eval_expm(double t) {
    double d, w0, ep, si, co;

    w0 = 2.0*M_PI*m_f;
    d = m_a1*m_a1 - 4*m_a2;

    if (d < 0) {
	/* We have Q > 1/2 */
	d = sqrt(-d);
	si = sin(d*w0*t/2);
	co = cos(d*w0*t/2);
    }
    else {
	/* We have Q <= 1/2 */
	d = sqrt(d);
	si = sinh(d*w0*t/2);
	co = cosh(d*w0*t/2);
    }

    ep = exp(-m_a1*w0*t/2);

    m_ta11 = ep*co - m_a1/d*ep*si;
    m_ta12 = 2/d*ep*si;
    m_ta21 = -2*m_a2/d*ep*si;
    m_ta22 = ep*co + m_a1/d*ep*si;
}

/**
 * Evaluate normalized sinc function.
 *
 * @param x Input value
 * @return Function value at x
 */
static double sinc_func(double x) {
    double y;

    if (fabs(x) == fabs(0.0)) {
	y = 1.0;
    }
    else {
	y = sin(M_PI*x) / (M_PI*x);
    }

    return y;
}

/**
 * Recalculate all parameters of the filter section.
 * This should be called always when parameters change.
 */
void Sts2ndSection::recalculate() {

    /*
     * Compute analog filter parameters
     * and feedforward gain
     */
    double K;
    K = pow(10.0,m_db/40);

    switch (m_type) {
    case SECTYPE_LOWSHELF:
	m_c = 1;
	m_b1 = (sqrt(K)-1/sqrt(K))/m_q;
	m_b2 = K-1/K;
	m_a1 = 1/sqrt(K)/m_q;
	m_a2 = 1/K;
	break;
    case SECTYPE_PEAK:
	m_c = 1;
	// m_b1 = (K*K-1)/m_q; // EE definition
	m_b1 = (K-1/K)/m_q; // Symmetric definition
	m_b2 = 0;
	// m_a1 = 1/m_q; // EE definition
	m_a1 = 1/m_q/K; // Symmetric definition definition
	m_a2 = 1;
	break;
    case SECTYPE_HIGHSHELF:
	m_c = K*K;
	m_b1 = (K*sqrt(K) - K*K*sqrt(K))/m_q;
	m_b2 = K-K*K*K;
	m_a1 = sqrt(K)/m_q;
	m_a2 = K;
	break;
    default:
	// Make silence
	m_c  = 0;
	m_b1 = 0;
	m_b2 = 0;
	m_a1 = 0;
	m_a2 = 1;
	break;
    }

    /*
     * Compute the input coefficients
     * with Simpson's rule
     */
    double h;

    h = m_T / SIMPSON_STEPS;

    for (int i = 0, j = -INPUT_COUNT; j <= INPUT_COUNT; j++, i++) {

	double bl1 = 0, bl2 = 0;

	for (int k = 0; k <= SIMPSON_STEPS; k++) {
	    /* Evaluate the integrand */
	    double f1, f2, tmp, t;
	    t = k*h;

	    tmp = sinc_func((t+j*m_T)/m_T)
		* (0.54+0.46*cos(M_PI*(t+j*m_T) / (INPUT_COUNT*m_T)));
	    eval_expm(m_T-t);
	    f1 = (m_ta11 * m_b1 + m_ta12 * m_b2) * 2 * M_PI * m_f * tmp;
	    f2 = (m_ta21 * m_b1 + m_ta22 * m_b2) * 2 * M_PI * m_f * tmp;
	    
	    /* Do the Simpson step */
	    if (k == 0 || k == SIMPSON_STEPS) {
		bl1 += f1;
		bl2 += f2;
	    }
	    else if ((k % 2) == 1) {
		bl1 += 4*f1;
		bl2 += 4*f2;
	    }
	    else {
		bl1 += 2*f1;
		bl2 += 2*f2;
	    }
	}
	bl1 *= h/3;
	bl2 *= h/3;

	m_bl1[i] = bl1;
	m_bl2[i] = bl2;
    }

    /*
     * Evaluate the transition matrix
     */
    eval_expm(m_T);
}

