%EQ_DESIGN_NEW  Design discrete-time EQ w/ the new method
%
% Syntax:
%   [TA,BL,B,A,c] = eq_design_new(type,fs,f,q,g_db,n,int_n,use_ee_q)
%
% In:
%    type - Type of the filter as string 'lowpass', 'highpass',
%           'bandpass', 'notch', 'allpass', 'peak', 'lowshelf',
%           'highself'
%      fs - Sampling frequency
%       f - Working frequency
%       q - The quality factor
%    g_db - Gain in decibels (for 'bandpass', 'peak', 'lowshelf'
%           and 'highshelf')
%       n - Order of the interpolation            (default 10)
%   int_n - Number of numerical integration steps (default 10)
% use_ee_q - Use EE definition for q (1) or not (0). Default 0.
%
% Out:
%     TA - Transition matrix of the state space model
%     BL - List of input coefficients in the model
%      B - 1xN numerator filter polynomial coefficients
%      A - 1xM denumerator filter polynomial coefficients
%      c - Feedforward signal gain
%   
% Description:
%   Design a discrete-time biquad filter of the given type
%   with the new method.

% Copyright (C) Simo Srkk, 2009

function [TA,BL,B,A,c,AA,BB,CC] = eq_design_new(type,fs,f,Q,g_db,n,int_n,use_ee_q)

    if nargin < 6
        n = [];
    end
    if nargin < 7
        int_n = [];
    end
    if nargin < 8
        use_ee_q = [];
    end
    
    if isempty(n)
        n = 10;
    end
    if isempty(int_n)
        int_n = 10;
    end
    if isempty(use_ee_q)
        use_ee_q = 0;
    end
    
    K = 10^(g_db/40);
      
    if ~use_ee_q && strcmp(type, 'peak')
        Q = K*Q;
    end
    
    switch lower(type)
    case 'lowpass'      
        % H(s) = 1 / (s^2 + s/Q + 1)
        c = 0;
        b1 = 0;
        b2 = 1;
        a1 = 1/Q;
        a2 = 1;        
    case 'highpass'
        % H(s) = s^2 / (s^2 + s/Q + 1)
        %      = 1 - (s/Q+1) / (s^2 + s/Q + 1)
        c = 1;
        b1 = -1/Q;
        b2 = -1;
        a1 = 1/Q;
        a2 = 1;        
    case 'bandpass'
        % H(s) = (Ks/Q) / (s^2 + s/Q + 1)
        c = 0;
        b1 = K/Q;
        b2 = 0;
        a1 = 1/Q;
        a2 = 1;        
    case 'notch'
        % H(s) = (s^2 + 1) / (s^2 + s/Q + 1)
        %      = 1 - (s/Q) / (s^2 + s/Q + 1)
        c = 1;
        b1 = -1/Q;
        b2 = 0;
        a1 = 1/Q;
        a2 = 1;        
    case 'allpass'
        % H(s) = (s^2 - s/Q + 1) / (s^2 + s/Q + 1)
        %      = 1 - (2s/Q) / (s^2 + s/Q + 1)
        c = 1;
        b1 = -2/Q;
        b2 = 0;
        a1 = 1/Q;
        a2 = 1;
    case 'peak'
        % H(s) = (s^2 + (K^2/Q)*s + 1) / (s^2 + s/Q + 1)
        %      = 1 + (K^2-1)/Q*s / (s^2 + s/Q + 1)        
        c = 1;
        b1 = (K^2-1)/Q;
        b2 = 0;
        a1 = 1/Q;
        a2 = 1;
    case 'lowshelf'
        % H(s) = (s^2 + (sqrt(K)/Q)*s + K) / (s^2 + (1/sqrt(K)/Q)*s + 1/K)
        %      = 1 + [(sqrt(K)-1/sqrt(K))/Q*s + (K-1/K)]
        %             / (s^2 + (1/sqrt(K)/Q)*s + 1/K)
        c = 1;
        b1 = (sqrt(K)-1/sqrt(K))/Q;
        b2 = K-1/K;
        a1 = 1/sqrt(K)/Q;
        a2 = 1/K;
    case 'highshelf'
        % H(s) = (K^2*s^2 + (K*sqrt(K)/Q)*s + K) / (s^2 + (sqrt(K)/Q)*s + K)
        %      = K^2 + [(K sqrt(K)-K^2 sqrt(K))/Q*s + (K-K^3)]
        %              / (s^2 + (sqrt(K)/Q)*s + K)        
        c = K^2;
        b1 = (K*sqrt(K) - K^2*sqrt(K))/Q;
        b2 = K-K^3;
        a1 = sqrt(K)/Q;
        a2 = K;
    otherwise
        error(sprintf('Unknown filter type "%s"',type));
    end

    w0 = 2*pi*f;
    T  = 1/fs;
    
    if nargout > 4
      [AA,BB,CC,TA,BL] = eqf_ss(a1,a2,b1,b2,c,w0,T,n,int_n);
    else
      [TA,BL] = eqf_disc(a1,a2,b1,b2,w0,T,n,int_n);
    end

    if nargout > 2
        G = 0;
        iz = 1/tf('z',T);
        izn = iz;
        for i=1:size(BL,2)
            G = G + BL(:,i)*izn;
            izn = izn*iz;
        end
        G = [1 0]*inv(eye(2) - TA/tf('z',T))*G + c*tf('z',T)^-n;
        G = minreal(G);
        A = G.den{1};
        B = G.num{1};
    end
    
    