(* 2017 (C) Jussi Rintanen *)

datatype opttype = NOOPT | ARGOPT | LONEOPT;

datatype commandlineoption = OUTFILE of string | TOCNF | TOACYC | DEBUGOUTPUT;

fun optiontype "-o" = ARGOPT
  | optiontype "-d" = LONEOPT
  | optiontype "-A" = LONEOPT
  | optiontype "-C" = LONEOPT
  | optiontype s = if String.sub(s,0) = #"-"
		   then ERROR ("Unknown option "^s)
		   else NOOPT;

fun loneoption "-A" = TOACYC
  | loneoption "-C" = TOCNF
  | loneoption "-d" = DEBUGOUTPUT;

fun argoption ("-o",filename) = OUTFILE filename
  | argoption ("-r",points) = (case Int.fromString points of
				   SOME t => OUTFILE "SHIT"
				 | NONE => (print "ERROR: -t requires an integer parameter\n";raise Match));

fun splitoptions0 ([],(fs,os)) = (rev fs,os)
  | splitoptions0 (a1::a2::ass,(fs,os)) =
    (case optiontype a1 of
	  NOOPT => splitoptions0 (a2::ass,(a1::fs,os))
	| LONEOPT => splitoptions0 (a2::ass,(fs,(loneoption a1)::os))
	| ARGOPT => splitoptions0 (ass,(fs,(argoption (a1,a2))::os)))
  | splitoptions0 ([a],(fs,os)) =
    (case optiontype a of
	  NOOPT => (rev(a::fs),os)
	| LONEOPT => (rev fs,(loneoption a)::os)
	| ARGOPT => (print (a^" requires an argument\n");raise Match));

fun splitoptions ops = splitoptions0 (ops,([],[]));

fun chooseoutfile [] = TextIO.stdOut
  | chooseoutfile ((OUTFILE s)::_) = TextIO.openOut s
  | chooseoutfile (_::os) = chooseoutfile os;

fun main () =
    if(CommandLine.arguments()) = []
    then app print
	     ["graph2dimacs v0.0 2017 (C) Jussi Rintanen\nusage: graph2dimacs [options] [inputfile]\n",
	      "-A        Reduce to DIMACS+Acyclicity\n",
	      "-C        Reduce to DIMACS (CNF only)\n",
	      "-o [file] Write output to [file]\n",
	      "-d        Show clauses with symbolic variable names (for debugging)\n"]
    else
        let
	    val (filenames,options) = splitoptions(CommandLine.arguments())
	    val outfile = chooseoutfile options

	    val entries =  xDIMACSparse (hd filenames)
	    val instance = xDIMACS.extract entries
	    val newinstance = if member(TOCNF,options)
			      then reach2cnf (instance,member(DEBUGOUTPUT,options))
			      else if member(TOACYC,options)
			      then reach2acyc (instance,member(DEBUGOUTPUT,options))
			      else (TextIO.closeOut outfile;ERROR "No reduction specified")

	in
	    xDIMACS.output(outfile,newinstance,member(DEBUGOUTPUT,options));
	    TextIO.closeOut outfile
	end handle Match => (print "Exited...\n");

main();
