#include "deferred.h"
#include "glutil.h"
#include "configuration.h"
#include <assert.h>

static const char* deferredVert =
"\
varying vec3 position;\
varying vec3 normal;\
varying vec4 color;\
\
uniform mat4 cameraTransform;\
\
void main()\
{\
	normal = gl_NormalMatrix * gl_Normal;\
	color = gl_Color;\
	position = (gl_ModelViewMatrix * gl_Vertex).xyz;\
	gl_TexCoord[0] = gl_MultiTexCoord0;\
	gl_Position = gl_ProjectionMatrix * cameraTransform * gl_ModelViewMatrix * gl_Vertex;\
}\
";

static const char* deferredFrag =
"\
#extension GL_ARB_draw_buffers : enable\n\
\
uniform sampler2D texture;\
\
varying vec3 position;\
varying vec3 normal;\
varying vec4 color;\
\
void main()\
{\
	gl_FragData[0] = vec4(position, 0.0);\
	gl_FragData[1] = vec4(normalize(normal), 0.0);\
	gl_FragData[2] = vec4(color.xyz * texture2D(texture, gl_TexCoord[0].xy).rgb + color.w * vec3(1e5, 1e5, 1e5), 1.0);\
}\
";

static const char* mosaicFrag =
"\
#extension GL_ARB_draw_buffers : enable\n\
\
const float invw = 1.0 / $f:screen_width;\
const float invh = 1.0 / $f:screen_height;\
const float invw_n = invw * $f:split_n;\
const float invh_m = invh * $f:split_m;\
\
uniform sampler2D positionMap;\
uniform sampler2D normalMap;\
\
void main()\
{\
	vec2 fc = (gl_FragCoord.xy - vec2(0.5, 0.5)) * vec2(invw_n, invh_m);\
	float fcfx = fract(fc.x);\
	float fcfy = fract(fc.y);\
	vec2 uv;\
	uv.x = (fc.x - fcfx + 0.5) * invw + fcfx;\
	uv.y = (fc.y - fcfy + 0.5) * invh + fcfy;\
	gl_FragData[0] = texture2D(positionMap, uv);\
	gl_FragData[1] = texture2D(normalMap, uv);\
}\
";

static const char* demosaicFrag =
"\
#extension GL_ARB_draw_buffers : enable\n\
#extension GL_EXT_gpu_shader4 : enable\n\
\
const int n = $i:split_n;\
const int m = $i:split_m;\
const int screen_width = $i:screen_width;\
const int screen_height = $i:screen_height;\
\
uniform sampler2D colorMap;\
\
void main()\
{\
	vec2 fc = gl_FragCoord.xy - vec2(0.5, 0.5);\
	fc.x = (mod(fc.x, float(n)) * float(screen_width/n) + truncate(fc.x / float(n)) + 0.5) / float(screen_width);\
	fc.y = (mod(fc.y, float(m)) * float(screen_height/m) + truncate(fc.y / float(m)) + 0.5) / float(screen_height);\
	gl_FragData[0] = texture2D(colorMap, fc);\
}\
";

static int gbufferProgram;
static int mosaicProgram;
static int demosaicProgram;

void Deferred::GBuffer::init(int width, int height)
{
	glGenFramebuffersEXT(1, &fb);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fb);
	glDrawBuffer(GL_NONE);
	glReadBuffer(GL_NONE);

	/* Depth buffer. */

	glGenRenderbuffersEXT(1, &rb);
	glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, rb);
	glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT24_ARB, width, height);
	glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, rb);

	/* Make textures for position, normal and color. */
	/* TODO: GL_BYTE would be enough for normal and color? */

	glGenTextures(3, &tex[0]);

	glBindTexture(GL_TEXTURE_2D, tex[0]);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB16F_ARB, width, height, 0, GL_RGB, GL_HALF_FLOAT_ARB, NULL);
	glBindTexture(GL_TEXTURE_2D, tex[1]);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB16F_ARB, width, height, 0, GL_RGB, GL_HALF_FLOAT_ARB, NULL);
	glBindTexture(GL_TEXTURE_2D, tex[2]);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB16F_ARB, width, height, 0, GL_RGB, GL_HALF_FLOAT_ARB, NULL);

	for (int i = 0; i < 3; i++)
	{
		glBindTexture(GL_TEXTURE_2D, tex[i]);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT+i, GL_TEXTURE_2D, tex[i], 0);
	}

	static const GLenum drawbuffers[3] =
	{
		GL_COLOR_ATTACHMENT0_EXT,
		GL_COLOR_ATTACHMENT1_EXT,
		GL_COLOR_ATTACHMENT2_EXT,
	};

	glDrawBuffersARB(3, drawbuffers);
}

void Deferred::GBuffer::bindToProgram(GLuint program)
{
	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, tex[0]);

	glActiveTexture(GL_TEXTURE1);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, tex[1]);

	glUseProgramObjectARB(program);
	setUniform(program, "positionMap", 0U);
	setUniform(program, "normalMap", 1U);

	assert(glGetError() == GL_NO_ERROR);
}

void Deferred::CBuffer::init(int width, int height)
{
	glGenFramebuffersEXT(1, &fb);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fb);
	//glDrawBuffer(GL_NONE);
	//glReadBuffer(GL_NONE);

	glGenTextures(1, &tex);

	glBindTexture(GL_TEXTURE_2D, tex);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB16F_ARB, width, height, 0, GL_RGB, GL_HALF_FLOAT_ARB, NULL);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, tex, 0);

	static const GLenum drawbuffers[1] =
	{
		GL_COLOR_ATTACHMENT0_EXT,
	};

	glDrawBuffersARB(1, drawbuffers);
}

void Deferred::renderQuad()
{
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(0, 1, 0, 1, -1, 1);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	glDisable(GL_CULL_FACE);
	glDisable(GL_DEPTH_TEST);

	glBegin(GL_QUADS);
	glVertex2f(0, 0);
	glVertex2f(1, 0);
	glVertex2f(1, 1);
	glVertex2f(0, 1);
	glEnd();

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}

void Deferred::mosaic(GBuffer& g1, GBuffer& g2)
{
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, g2.fb);
	g1.bindToProgram(mosaicProgram);
	renderQuad();
	glUseProgramObjectARB(0);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
}

void Deferred::demosaic(GBuffer& g1, GBuffer& g2)
{
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, g2.fb);
	g1.bindToProgram(demosaicProgram);
	renderQuad();
	glUseProgramObjectARB(0);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
}

Deferred::Deferred()
{
	g1.init(CONFI(screen_width), CONFI(screen_height));
	g2.init(CONFI(screen_width), CONFI(screen_height));
	c1.init(CONFI(screen_width), CONFI(screen_height));
	c2.init(CONFI(screen_width), CONFI(screen_height));
	c3.init(CONFI(screen_width), CONFI(screen_height));

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	/* Make programs. */

	if (gbufferProgram == 0)
		gbufferProgram = MAKE_GLSL_PROGRAM(deferredFrag, deferredVert);

	if (mosaicProgram == 0)
		mosaicProgram = MAKE_GLSL_PROGRAM(mosaicFrag, NULL);

	if (demosaicProgram == 0)
		demosaicProgram = MAKE_GLSL_PROGRAM(demosaicFrag, NULL);
}

Deferred::~Deferred()
{
	/* TODO: */
}

void Deferred::renderScene(void (*render)(void*), void* user)
{
	/* Render scene to G1. */

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, g1.fb);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	assert(glGetError() == GL_NO_ERROR);

	glPushMatrix();

	float m[16];
	glGetFloatv(GL_MODELVIEW_MATRIX, m);
	glLoadIdentity();

	glUseProgramObjectARB(gbufferProgram);
	setUniform(gbufferProgram, "texture", 0U);
	setUniform4x4Mat(gbufferProgram, "cameraTransform", m);
	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_CULL_FACE);
	render(user);
	glUseProgramObjectARB(0);

	glPopMatrix();

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
}

void Deferred::do_mosaic()
{
	/* Mosaic G1 => G2. */
	mosaic(g1, g2);
}

void Deferred::prepare(GLuint program)
{
	assert(glGetError() == GL_NO_ERROR);

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, c1.fb);

	g2.bindToProgram(program);
	assert(glGetError() == GL_NO_ERROR);

	glActiveTexture(GL_TEXTURE0);
	assert(glGetError() == GL_NO_ERROR);
}

void Deferred::done()
{
	/* Demosaic. */

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, c2.fb);

	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, c1.tex);
	glUseProgramObjectARB(demosaicProgram);
	setUniform(demosaicProgram, "colorMap", 0U);

	renderQuad();

	glUseProgramObjectARB(0);

	for (int i = 0; i < 3; i++)
	{
		glActiveTexture(GL_TEXTURE0+i);
		glDisable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, 0);
	}

	glActiveTexture(GL_TEXTURE0);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
}
