#include "trimesh.h"
#include "raytracer.h"
#include "textures.h"
#include "utils.h"
#include "configuration.h"
#include "glutil.h"

template<class T>
T* vector_to_array(const std::vector<T>& v)
{
	T* ary = new T [v.size()];
	for (unsigned int i = 0; i < v.size(); i++)
		ary[i] = v[i];
	return ary;
}

Trimesh::Trimesh(const ObjLoader& obj, bool selfAO)
	: position(0.0f, 0.0f, 0.0f),
	  rotation(0.0f, 0.0f, 0.0f),
	  list(0),
	  list2(0)
{
	/* TODO: if vertices don't have normals but they do have material, vertices will be
	 *       always duplicated? */

	for (unsigned int i = 0; i < obj.vertices.size(); i++)
	{
		Vertex v;

		v.p = obj.vertices[i];
		v.n = m::Vector3(0.0f, 0.0f, 0.0f);
		v.c = m::Vector4(1.0f, 1.0f, 1.0f, 0.0f);
		v.t = m::Vector2(0.0f, 0.0f);

		vertices.push_back(v);
	}

	for (unsigned int i = 0; i < obj.polygons.size(); i++)
	{
		const ObjLoader::Polygon& poly = obj.polygons[i];

		for (unsigned int j = 2; j < poly.vertices.size(); j++)
		{
			Triangle t;

			t.v[0] = poly.vertices[0].v;
			t.v[1] = poly.vertices[j-1].v;
			t.v[2] = poly.vertices[j].v;

			m::Vector3 normal = cross(vertices[t.v[1]].p - vertices[t.v[0]].p, vertices[t.v[2]].p - vertices[t.v[0]].p);
			normal.normalize0();

			m::Vector4 color = m::Vector4(1.0f, 1.0f, 1.0f, 0.f);
			if (poly.material >= 0)
			{
				color.x = obj.materials[poly.material].diffuse[0];
				color.y = obj.materials[poly.material].diffuse[1];
				color.z = obj.materials[poly.material].diffuse[2];
				color.w = 0.0f;

				if (obj.materials[poly.material].emission[0] >= 0.1f ||
					obj.materials[poly.material].emission[1] >= 0.1f ||
					obj.materials[poly.material].emission[2] >= 0.1f)
				{
					color.w = 1.f;
				}
			}

			int n[3] =
			{
				poly.vertices[0].n,
				poly.vertices[j-1].n,
				poly.vertices[j].n
			};

			int tex[3] =
			{
				poly.vertices[0].t,
				poly.vertices[j-1].t,
				poly.vertices[j].t
			};

			/* Set normals and duplicate vertices if there are several normals
			 * for the same vertex. */

			for (unsigned int k = 0; k < 3; k++)
			{
				if (vertices[t.v[k]].n.length() < 0.0001f)
				{
					vertices[t.v[k]].n = (n[k] < 0) ? normal : obj.normals[n[k]];
					vertices[t.v[k]].c = color;
					if (tex[k] >= 0)
						vertices[t.v[k]].t = m::Vector2(obj.textureCoords[tex[k]].x, obj.textureCoords[tex[k]].y);
				}
				else
				{
					if ((vertices[t.v[k]].n - ((n[k] < 0) ? normal : obj.normals[n[k]])).length() > 0.0001f ||
					    (vertices[t.v[k]].c - color).length() > 0.001f)
					{
						Vertex v = vertices[t.v[k]];
						v.n = (n[k] < 0) ? normal : obj.normals[n[k]];
						v.c = color;
						if (tex[k] >= 0)
							v.t = m::Vector2(obj.textureCoords[tex[k]].x, obj.textureCoords[tex[k]].y);

						vertices.push_back(v);
						t.v[k] = (int)vertices.size() - 1;
					}
				}
			}

			Textures::Texture& texture = (poly.material >= 0) ? textures.load(obj.materials[poly.material].diffuseMap) : textures.load("");
			t.tex = texture.id;

			triangles.push_back(t);
		}
	}

	for (unsigned int i = 0; i < vertices.size(); i++)
		vertices[i].ao = 1.0f;

	if (selfAO)
	{
		RayTracer rt;
		rt.feedObject(obj);
		rt.update();

		for (unsigned int i = 0; i < vertices.size(); i++)
		{
			const int samples = CONFI(ao_samples);
			m::Vector3 pos = vertices[i].p + vertices[i].n * 0.01f;
			int n = 0;

			for (int j = 0; j < samples; j++)
			{
				float a = (j + 0.5f) / (float)samples;
				float b = halton2(j+1);
				m::Vector3 dir = pointOnHemisphereCos(vertices[i].n, a, b);

				m::Vector3 tmp1, tmp2, tmp3;
				if (!rt.intersect(pos, dir, tmp1, tmp2, tmp3))
					n++;
			}

			vertices[i].ao = n / (float)samples;
			//vertices[i].ao *= vertices[i].ao;
			// TODO: this is wrong, ambient occlusion should only affect to indirect lighting
			vertices[i].c *= vertices[i].ao;
		}
	}

	list = 0;
	list2 = 0;

	glGenBuffersARB(1, &vbuf);
	glGenBuffersARB(1, &vbuf2);
	glGenBuffersARB(1, &ebuf);

	updateBuffers();

	assert(glGetError() == GL_NO_ERROR);
}

Trimesh::~Trimesh()
{
}

template<class T>
std::pair<T, T> sorted_pair(T &a, T &b)
{
	if (a < b)
		return std::pair<T, T>(a, b);
	else
		return std::pair<T, T>(b, a);
}

void Trimesh::updateBuffers()
{
	assert(glGetError() == GL_NO_ERROR);

	/* Vertex buffer. */

	Vertex* vtab = vector_to_array(vertices);

	glBindBufferARB(GL_ARRAY_BUFFER_ARB, vbuf);
	glBufferDataARB(GL_ARRAY_BUFFER_ARB, vertices.size() * sizeof(Vertex), vtab, GL_STATIC_DRAW_ARB);
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);

	assert(glGetError() == GL_NO_ERROR);

	delete [] vtab;

	/* Vertex buffer 2 (only positions for vertices). */

	m::Vector3* vtab2 = new m::Vector3 [vertices.size()];

	for (unsigned int i = 0; i < vertices.size(); i++)
		vtab2[i] = vertices[i].p;

	glBindBufferARB(GL_ARRAY_BUFFER_ARB, vbuf2);
	glBufferDataARB(GL_ARRAY_BUFFER_ARB, vertices.size() * sizeof(m::Vector3), vtab2, GL_STATIC_DRAW_ARB);
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);

	assert(glGetError() == GL_NO_ERROR);

	delete [] vtab2;

	/* Triangle chunks. */

	chunks.clear();

	unsigned int* t = new unsigned int [triangles.size() * 3];
	unsigned int count = 0;

	for (std::map<std::string, Textures::Texture>::iterator iter = textures.textures.begin();
	     iter != textures.textures.end();
		 iter++)
	{
		unsigned int n = 0;

		for (unsigned int i = 0; i < triangles.size(); i++)
		{
			if (triangles[i].tex == iter->second.id)
			{
				t[count++] = triangles[i].v[0];
				t[count++] = triangles[i].v[1];
				t[count++] = triangles[i].v[2];
				n += 3;
			}
		}

		if (n)
			chunks.push_back(std::pair<int, int>(iter->second.id, n));
	}

	assert(count == triangles.size() * 3);

	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, ebuf);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER_ARB, count * sizeof(unsigned int), t, GL_STATIC_DRAW_ARB);
	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, 0);

	assert(glGetError() == GL_NO_ERROR);

	delete [] t;
}

void Trimesh::subdivide(float maxLength)
{
	std::vector<Triangle> tris;

	bool done;

	do
	{
		done = true;
		tris.clear();

		for (unsigned int i = 0; i < triangles.size(); i++)
		{
			Triangle& t = triangles[i];
			float l1 = (vertices[t.v[0]].p - vertices[t.v[1]].p).length();
			float l2 = (vertices[t.v[1]].p - vertices[t.v[2]].p).length();
			float l3 = (vertices[t.v[2]].p - vertices[t.v[0]].p).length();

			if (l1 >= maxLength || l2 >= maxLength || l3 >= maxLength)
			{
				vertices.push_back(vertices[t.v[0]].average(vertices[t.v[1]]));
				vertices.push_back(vertices[t.v[1]].average(vertices[t.v[2]]));
				vertices.push_back(vertices[t.v[2]].average(vertices[t.v[0]]));

				int a = t.v[0];
				int b = t.v[1];
				int c = t.v[2];
				int ab = (int)vertices.size() - 3;
				int bc = (int)vertices.size() - 2;
				int ca = (int)vertices.size() - 1;

#define NEW_TRI(v1, v2, v3) { Triangle t2; t2.tex = triangles[i].tex; t2.v[0] = v1; t2.v[1] = v2; t2.v[2] = v3; tris.push_back(t2); }
				NEW_TRI(a, ab, ca);
				NEW_TRI(ca, ab, bc);
				NEW_TRI(ca, bc, c);
				NEW_TRI(ab, b, bc);

				done = false;
			}
			else
				tris.push_back(t);
		}

		triangles = tris;
	} while(!done);

	if (list != 0)
	{
		glDeleteLists(list, 1);
		list = 0;
	}

	if (list2 != 0)
	{
		glDeleteLists(list2, 1);
		list2 = 0;
	}

	printf("Triangles after subdivide: %zu\n", triangles.size());

	updateBuffers();
}

void Trimesh::render()
{
	assert(glGetError() == GL_NO_ERROR);

	glPushMatrix();
	glTranslatef(position.x, position.y, position.z);
	glRotatef(rotation.y, 0.0f, 1.0f, 0.0f);
	glRotatef(rotation.x, 1.0f, 0.0f, 0.0f);
	glRotatef(rotation.z, 0.0f, 0.0f, 1.0f);

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_NORMAL_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	glBindBufferARB(GL_ARRAY_BUFFER_ARB, vbuf);
	glVertexPointer(3, GL_FLOAT, sizeof(Vertex), &((Vertex*)0)->p.x);

	glBindBufferARB(GL_ARRAY_BUFFER_ARB, vbuf);
	glNormalPointer(GL_FLOAT, sizeof(Vertex), &((Vertex*)0)->n.x);

	glBindBufferARB(GL_ARRAY_BUFFER_ARB, vbuf);
	glColorPointer(4, GL_FLOAT, sizeof(Vertex), &((Vertex*)0)->c.x);

	glBindBufferARB(GL_ARRAY_BUFFER_ARB, vbuf);
	glTexCoordPointer(2, GL_FLOAT, sizeof(Vertex), &((Vertex*)0)->t.x);

	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, ebuf);
	assert(glGetError() == GL_NO_ERROR);

	unsigned int i = 0;
	for (unsigned int j = 0; j < chunks.size(); j++)
	{
		glBindTexture(GL_TEXTURE_2D, chunks[j].first);
		glDrawElements(GL_TRIANGLES, chunks[j].second, GL_UNSIGNED_INT, &((int*)0)[i]);
		i += chunks[j].second;
	}
	assert(glGetError() == GL_NO_ERROR);

	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, 0);
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
	assert(glGetError() == GL_NO_ERROR);

	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	glPopMatrix();

	assert(glGetError() == GL_NO_ERROR);
}

void Trimesh::renderDepth()
{
	glPushMatrix();
	glTranslatef(position.x, position.y, position.z);
	glRotatef(rotation.x, 1.0f, 0.0f, 0.0f);
	glRotatef(rotation.y, 0.0f, 1.0f, 0.0f);
	glRotatef(rotation.z, 0.0f, 0.0f, 1.0f);

	glEnableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	glBindBufferARB(GL_ARRAY_BUFFER_ARB, vbuf2);
	glVertexPointer(3, GL_FLOAT, sizeof(float)*3, NULL);

	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, ebuf);
	glDrawElements(GL_TRIANGLES, (int)triangles.size() * 3, GL_UNSIGNED_INT, NULL);

	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, 0);
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);

	glDisableClientState(GL_VERTEX_ARRAY);

	glPopMatrix();
}
