#include "objloader.h"
#include "configuration.h"
#include <ctype.h>

ObjLoader::ObjLoader(const char* filename, float s)
{
	load(filename);
	scale(m::Vector3(s, s, s));
}

ObjLoader::ObjLoader(const char* filename, const m::Vector3& s)
{
	load(filename);
	scale(s);
}

ObjLoader::~ObjLoader()
{
}

static char* strip_line(char* p)
{
	if (strchr(p, '#'))
		*strchr(p, '#') = '\0';
	if (strchr(p, '\r'))
		*strchr(p, '\r') = '\0';
	if (strchr(p, '\n'))
		*strchr(p, '\n') = '\0';

	while (isspace(*p))
		p++;

	return p;
}

void ObjLoader::load(const char* filename)
{
	FILE* fp = fopen(filename, "rt");
	if (!fp)
		return;

	int lineno = 0;
	int material = -1;
	int unknowns = 0;

	for (;;)
	{
		char buf[2048];
		if (fgets(buf, sizeof(buf), fp) == NULL)
			break;

		lineno++;

		char* p = strip_line(buf);

		if (*p == '\0')
			continue;

		if (strncmp(p, "v ", 2) == 0)
		{
			m::Vector3 v;
			sscanf(p+2, "%f %f %f", &v.x, &v.y, &v.z);
			vertices.push_back(v);
		}
		else if (strncmp(p, "vn ", 3) == 0)
		{
			m::Vector3 n;
			sscanf(p+3, "%f %f %f", &n.x, &n.y, &n.z);
			normals.push_back(normalize(n));
		}
		else if (strncmp(p, "vt ", 3) == 0)
		{
			m::Vector3 t;
			sscanf(p+3, "%f %f %f", &t.x, &t.y, &t.z);
			textureCoords.push_back(t);
		}
		else if (strncmp(p, "f ", 2) == 0)
		{
			Polygon poly;
			char buf2[256] = "";
			int n;

			p += 2;

			int cnt = 0;
			while (sscanf(p, "%s%n", buf2, &n) != EOF)
			{
				VertexIndices vi;

				vi.v = -1;
				vi.t = -1;
				vi.n = -1;

				sscanf(buf2, "%d//", &vi.v);
				sscanf(buf2, "%d//%d", &vi.v, &vi.n);
				sscanf(buf2, "%d/%d/%d", &vi.v, &vi.t, &vi.n);

				vi.v = (vi.v <= 0) ? -1 : (vi.v - 1);
				//vi.t = (vi.t <= 0) ? -1 : (vi.t - 1);
				vi.t = (vi.t <= 0) ? (cnt % 4): (vi.t - 1);	// [samuli] hack for maze
				vi.n = (vi.n <= 0) ? -1 : (vi.n - 1);

				if (CONFI(flat_normals))
					vi.n = -1;

				poly.vertices.push_back(vi);

				p += n;
				cnt++;
			}

			poly.material = material;
			polygons.push_back(poly);
		}
		else if (strncmp(p, "mtllib ", 7) == 0)
		{
			loadMtl(("data/" + std::string(p+7)).c_str());
		}
		else if (strncmp(p, "usemtl ", 7) == 0)
		{
			std::string s = p+7;

			unsigned int i;
			for (i = 0; i < materials.size(); i++)
			{
				if (materials[i].name == s)
					break;
			}

			material = (i == materials.size()) ? -1 : (int)i;
		}
		else
		{
			if (unknowns < 10)
				printf("%s: Unknown line %d: '%s'\n", filename, lineno, p);
			unknowns++;
		}
	}

	if (textureCoords.empty())
	{
		textureCoords.push_back(m::Vector3(0, 0, 0));
		textureCoords.push_back(m::Vector3(1, 0, 0));
		textureCoords.push_back(m::Vector3(1, 1, 0));
		textureCoords.push_back(m::Vector3(0, 1, 0));
	}

	fclose(fp);

	/* Check. */

	for (unsigned int i = 0; i < polygons.size(); i++)
	{
		const Polygon& p = polygons[i];

		for (unsigned int j = 0; j < p.vertices.size(); j++)
		{
			assert(p.vertices[j].v >= 0 && p.vertices[j].v < (int)vertices.size());
			assert(p.vertices[j].n < (int)normals.size());
			assert(p.vertices[j].t < (int)textureCoords.size());
		}
	}

#if 0
	vd::Client vdc;
	vdc.connect();
	vdc.setActiveGroup(33);
	vdc.clearGroup();

	for (unsigned int i = 0; i < vertices.size(); i++)
		vdc.point(vertices[i]);

	for (unsigned int i = 0; i < polygons.size(); i++)
	{
		Polygon &p = polygons[i];

		for (unsigned int j =0 ; j < p.vertices.size(); j++)
		{
			vdc.line(vertices[p.vertices[j].v], vertices[p.vertices[(j+1)%p.vertices.size()].v]);
		}
	}
#endif
}

void ObjLoader::loadMtl(const char* filename)
{
	FILE* fp = fopen(filename, "r");
	if (!fp)
		return;

	int lineno = 0;
	int unknowns = 0;
	Material* mtl = NULL;

	for (;;)
	{
		char buf[2048];
		if (fgets(buf, sizeof(buf), fp) == NULL)
			break;

		lineno++;

		char* p = strip_line(buf);

		if (*p == '\0')
			continue;

		if (strncmp(p, "newmtl ", 7) == 0)
		{
			materials.push_back(Material());
			mtl = &materials[materials.size()-1];
			mtl->name = p+7;
		}
		else if (strncmp(p, "Ns ", 3) == 0)
		{
			if (mtl)
				sscanf(p+3, "%f", &mtl->shininess);
		}
		else if (strncmp(p, "d ", 2) == 0)
		{
			if (mtl)
				sscanf(p+2, "%f", &mtl->d);
		}
		else if (strncmp(p, "Kd ", 3) == 0)
		{
			if (mtl)
				sscanf(p+3, "%f %f %f", &mtl->diffuse[0], &mtl->diffuse[1], &mtl->diffuse[2]);
		}
		else if (strncmp(p, "Ka ", 3) == 0)
		{
			if (mtl)
				sscanf(p+3, "%f %f %f", &mtl->ambient[0], &mtl->ambient[1], &mtl->ambient[2]);
		}
		else if (strncmp(p, "Ks ", 3) == 0)
		{
			if (mtl)
				sscanf(p+3, "%f %f %f", &mtl->specular[0], &mtl->specular[1], &mtl->specular[2]);
		}
		else if (strncmp(p, "Ke ", 3) == 0)
		{
			if (mtl)
				sscanf(p+3, "%f %f %f", &mtl->emission[0], &mtl->emission[1], &mtl->emission[2]);
		}
		else if (strncmp(p, "map_Kd ", 7) == 0)
		{
			if (mtl)
				mtl->diffuseMap = p + 7;
		}
		else
		{
			if (unknowns < 10)
				printf("%s: Unknown line %d: '%s'\n", filename, lineno, p);
			unknowns++;
		}
	}

	fclose(fp);

	/* TODO: check duplicates */
}

void ObjLoader::scale(const m::Vector3& s)
{
	for (unsigned int i = 0; i < vertices.size(); i++)
		vertices[i] = m::scale(vertices[i], s);
}
