#include "light.h"
#include "utils.h"
#include "raytracer.h"
#include "configuration.h"
#include "timing.h"

Light::Light(int n) : shadowMap(CONFI(primary_shadowmap_size)), type(SPOT)
{
	position = m::Vector3(0.0f, 0.0f, 0.0f);
	direction = m::Vector3(1.0f, 0.0f, 0.0f);
	intensity = m::Vector3(10.0f, 10.0f, 10.0f);
	shadowMap.setPosition(position);
	shadowMap.setDirection(direction);

	subLightCount = n;
	subLights = new SubLight [n];

	for (int i = 0; i < n; i++)
		subLights[i].valid = false;
}

Light::~Light()
{
	delete [] subLights;
}

void Light::setPosition(const m::Vector3& p)
{
	position = p;
	shadowMap.setPosition(p);
}

void Light::setDirection(const m::Vector3& d)
{
	direction = d;
	shadowMap.setDirection(d);

	right = m::normalize(m::cross(m::Vector3(0.0f, 1.0f, 0.0f), d));
	up = m::cross(d, right);
}

void Light::setIntensity(const m::Vector3& i)
{
	intensity = i;
}

void Light::updateDelaunay()
{
	tri2.clear();
	tri3.clear();
	indices2.clear();
	indices3.clear();

	tri3.insert(Triangulation3::Point(0, 0, 0));

	for (int i = 0; i < subLightCount; i++)
	{
		if (subLights[i].valid == false)
			continue;

		m::Vector3 p = subLights[i].position - position;
		p.normalize();

		if (type == SPOT)
		{
			/* z may be ignored because we want cosine weighted hemisphere. */
			float x = m::dot(p, right);
			float y = m::dot(p, up);

			Triangulation2::Vertex_handle h = tri2.insert(Triangulation2::Point(x, y));
			indices2[h] = i;

			subLights[i].p = m::Vector3(x, y, 0);
		}
		else
		{
			Triangulation3::Vertex_handle h = tri3.insert(Triangulation3::Point(p.x, p.y, p.z));
			indices3[h] = i;
			subLights[i].p = p;
		}
	}
}

template<class T>
void smart_clear(T& t)
{
	unsigned int n = t.size();
	t.clear();
	t.reserve(n);
}

void Light::updateVoronoi()
{
	for (int i = 0; i < subLightCount; i++)
		subLights[i].area = 0.0f;

	disp = 0.0f;

	/* Omni light. */

	if (type == OMNI)
	{
		smart_clear(debugVoronoiVertices3);
		smart_clear(debugVoronoiEdges3);

		if (tri3.number_of_cells() < 1)
			return;

		/* Faces. */

		Triangulation3::Vertex_handle infiniteVertex = tri3.infinite_vertex();

		std::set<Triangulation3::Facet> faces;
		for (Triangulation3::Facet_iterator iter = tri3.facets_begin();
		     iter != tri3.facets_end();
			 iter++)
		{
			Triangulation3::Facet mirror = tri3.mirror_facet(*iter);
			if (tri3.mirror_vertex(mirror.first, mirror.second) == infiniteVertex)
				faces.insert(*iter);
			else if (tri3.mirror_vertex(iter->first, iter->second) == infiniteVertex)
				faces.insert(mirror);
		}

		/* Voronoi vertices. */

		std::map<Triangulation3::Facet, m::Vector3> voronoiVertices;

		for (std::set<Triangulation3::Facet>::iterator iter = faces.begin();
			 iter != faces.end();
			 iter++)
		{
			Triangulation3::Triangle tri = tri3.triangle(*iter);

			m::Vector3 v0 = m::Vector3(tri.vertex(0).x(), tri.vertex(0).y(), tri.vertex(0).z());
			m::Vector3 v1 = m::Vector3(tri.vertex(1).x(), tri.vertex(1).y(), tri.vertex(1).z());
			m::Vector3 v2 = m::Vector3(tri.vertex(2).x(), tri.vertex(2).y(), tri.vertex(2).z());

			m::Vector3 normal = normalize(cross(v1 - v0, v2 - v0));

			voronoiVertices[*iter] = normal;

			//debugVoronoiVertices3.push_back(normal);

			float d = 1.0f - dot(normal, v0);
			if (d > disp)
			{
				disp = d;
				dispP = normal;
			}
		}

		/* Edges. */

		std::map<std::pair<Triangulation3::Vertex_handle, Triangulation3::Vertex_handle>, Triangulation3::Facet> edges;

		for (std::set<Triangulation3::Facet>::iterator iter = faces.begin();
			 iter != faces.end();
			 iter++)
		{
			Triangulation3::Vertex_handle v[3];

			int i = iter->second;
			if ((i & 1) == 0)
			{
				v[0] = iter->first->vertex((i+2) & 3);
				v[1] = iter->first->vertex((i+1) & 3);
				v[2] = iter->first->vertex((i+3) & 3);
			}
			else
			{
				v[0] = iter->first->vertex((i+1) & 3);
				v[1] = iter->first->vertex((i+2) & 3);
				v[2] = iter->first->vertex((i+3) & 3);
			}

			for (unsigned int j = 0; j < 3; j++)
			{
				Triangulation3::Vertex_handle v0 = v[j];
				Triangulation3::Vertex_handle v1 = v[(j+1)%3];
				std::map<std::pair<Triangulation3::Vertex_handle, Triangulation3::Vertex_handle>, Triangulation3::Facet>::iterator iter2;
				std::pair<Triangulation3::Vertex_handle, Triangulation3::Vertex_handle> p = std::pair<Triangulation3::Vertex_handle, Triangulation3::Vertex_handle>(v1, v0);

				iter2 = edges.find(p);
				if (iter2 == edges.end())
				{
					edges[std::pair<Triangulation3::Vertex_handle, Triangulation3::Vertex_handle>(v0, v1)] = *iter;
				}
				else
				{
					m::Vector3 a = voronoiVertices[*iter];
					m::Vector3 b = voronoiVertices[iter2->second];

					subLights[indices3[v0]].area += triangle_area_on_sphere(
						m::Vector3(v0->point().x(), v0->point().y(), v0->point().z()), a, b);

					subLights[indices3[v1]].area += triangle_area_on_sphere(
						m::Vector3(v1->point().x(), v1->point().y(), v1->point().z()), a, b);

#ifndef NDEBUG
					debugVoronoiEdges3.push_back(std::pair<m::Vector3, m::Vector3>(a, b));
#endif

					//edges.erase(iter2);
				}
			}
		}

		float A = 0.0f;
		for (int i = 0; i < subLightCount; i++)
		{
			subLights[i].area /= 4.0f * 3.14159265f;
			if (subLights[i].valid)
				A += subLights[i].area;
		}
		//printf("%f\n", A);

		return;
	}

	/* Spot light. */

	assert(type == SPOT);

	std::map<int, m::Vector2> boundary;

	edges2.clear();
	smart_clear(debugTris2);

	for (Triangulation2::Edge_iterator iter = tri2.finite_edges_begin();
	     iter != tri2.finite_edges_end();
		 iter++)
	{
		CGAL::Object o = tri2.dual(iter);
		K::Segment_2 seg;
		K::Ray_2 ray;

		Triangulation2::Face_handle f1 = iter->first;
		Triangulation2::Face_handle f2 = f1->neighbor(iter->second);

		int l0 = indices2[f1->vertex(tri2.cw(iter->second))];
		int l1 = indices2[f1->vertex(tri2.ccw(iter->second))];

		VoronoiEdge2 edge;

		bool boundaryVertex = false;

		if (CGAL::assign(seg, o))
		{
			m::Vector2 p0 = m::Vector2(seg.point(0).x(), seg.point(0).y());
			m::Vector2 p1 = m::Vector2(seg.point(1).x(), seg.point(1).y());

			if (p0.length() >= 1.0f)
			{
				if (p1.length() >= 1.0f)
				{
					/* I guess there's no need to handle this case. */
					continue;
				}

				edge.p[0] = p1;
				edge.p[1] = intersect_circle(p1, p0);
				boundaryVertex = true;
			}
			else
			{
				if (p1.length() > 1.0f)
				{
					edge.p[0] = p0;
					edge.p[1] = intersect_circle(p0, p1);
					boundaryVertex = true;
				}
				else
				{
					edge.p[0] = p0;
					edge.p[1] = p1;
				}
			}
		}

		if (CGAL::assign(ray, o))
		{
			m::Vector2 p = m::Vector2(ray.source().x(), ray.source().y());
			m::Vector2 d = m::Vector2(ray.direction().dx(), ray.direction().dy());

			if (p.length() >= 1.0f)
				continue;

			edge.p[0] = p;
			edge.p[1] = intersect_circle(p, p + normalize(d) * 2.1f);
			boundaryVertex = true;
		}

		edges2.push_back(edge);

		subLights[l0].area += triangle_area(edge.p[0], edge.p[1], make_vec2(subLights[l0].p));
		subLights[l1].area += triangle_area(edge.p[0], edge.p[1], make_vec2(subLights[l1].p));

		debugTris2.push_back(DebugTri2(edge.p[0], edge.p[1], make_vec2(subLights[l0].p), 1,0,0));
		debugTris2.push_back(DebugTri2(edge.p[0], edge.p[1], make_vec2(subLights[l1].p), 1,0,0));

		if (boundaryVertex)
		{
			float d = (edge.p[1] - m::Vector2(make_vec2(subLights[l0].p))).length();
			if (d > disp)
			{
				dispP = m::Vector3(edge.p[1].x, edge.p[1].y, 0.0f);
				disp = d;
			}

			/* TODO: normalize lightpoint and check dispersion */

			if (boundary.find(l0) == boundary.end())
				boundary[l0] = edge.p[1];
			else
			{
				/* TODO: not tested if this works. */
				if ((1.0f - subLights[l0].p.length()) > disp)
				{
					if (cross(edge.p[1], make_vec2(subLights[l0].p)) * cross(boundary[l0], make_vec2(subLights[l0].p)) < 0.0f)
					{
						disp = 1.0f - subLights[l0].p.length();
						m::Vector2 p = normalize(make_vec2(subLights[l0].p));
						dispP = m::Vector3(p.x, p.y, 0);
					}
				}

				subLights[l0].area += sector_area(edge.p[1], boundary[l0], make_vec2(subLights[l0].p));
				debugTris2.push_back(DebugTri2(edge.p[1], boundary[l0], make_vec2(subLights[l0].p), 0,0,1));
				boundary.erase(boundary.find(l0));
			}

			if (boundary.find(l1) == boundary.end())
				boundary[l1] = edge.p[1];
			else
			{
				/* TODO: not tested if this works. */
				if ((1.0f - subLights[l1].p.length()) > disp)
				{
					if (cross(edge.p[1], make_vec2(subLights[l1].p)) * cross(boundary[l1], make_vec2(subLights[l1].p)) < 0.0f)
					{
						disp = 1.0f - make_vec2(subLights[l1].p).length();
						m::Vector2 p = normalize(make_vec2(subLights[l1].p));
						dispP = m::Vector3(p.x, p.y, 0);
					}
				}

				subLights[l1].area += sector_area(edge.p[1], boundary[l1], make_vec2(subLights[l1].p));
				debugTris2.push_back(DebugTri2(edge.p[1], boundary[l1], make_vec2(subLights[l1].p), 0,0,1));
				boundary.erase(boundary.find(l1));
			}
		}
	}

	//assert(boundary.empty() == false);

	/* Search dispersion maximum from finite faces. */

	for (Triangulation2::Face_iterator iter = tri2.finite_faces_begin();
		 iter != tri2.finite_faces_end();
		 iter++)
	{
		float x = tri2.circumcenter(iter).x();
		float y = tri2.circumcenter(iter).y();

		if (x*x + y*y > 1.0f)
			continue;

		float d = sqrtf(CGAL::squared_distance(tri2.circumcenter(iter), iter->vertex(0)->point()));

		if (d > disp)
		{
			disp = d;
			dispP = m::Vector3(tri2.circumcenter(iter).x(), tri2.circumcenter(iter).y(), 0);
		}
	}

	/* Check area. */

	float A = 0.0f;
	for (int i = 0; i < subLightCount; i++)
	{
		subLights[i].area /= 3.14159265f;
		if (subLights[i].valid)
			A += subLights[i].area;
	}
#if 0
	printf("Total area of the circle: %f\n", A);
#endif
}

void Light::updateDispersion()
{
	if (type == SPOT)
		updateVoronoi();
	else
	{
		assert(type == OMNI);

		disp = 0.0f;

		if (tri3.number_of_cells() < 1)
			return;

		/* Faces. */

		Triangulation3::Vertex_handle infiniteVertex = tri3.infinite_vertex();

		std::set<Triangulation3::Facet> faces; // TODO: this set is not needed
		for (Triangulation3::Facet_iterator iter = tri3.facets_begin();
		     iter != tri3.facets_end();
			 iter++)
		{
			if (tri3.is_infinite(*iter))
				continue;

			Triangulation3::Facet mirror = tri3.mirror_facet(*iter);
			if (tri3.mirror_vertex(mirror.first, mirror.second) == infiniteVertex)
				faces.insert(*iter);
			else if (tri3.mirror_vertex(iter->first, iter->second) == infiniteVertex)
				faces.insert(mirror);
		}

		/* Voronoi vertices. */

		std::map<Triangulation3::Facet, m::Vector3> voronoiVertices;

		for (std::set<Triangulation3::Facet>::iterator iter = faces.begin();
			 iter != faces.end();
			 iter++)
		{
			Triangulation3::Triangle tri = tri3.triangle(*iter);

			m::Vector3 v0 = m::Vector3(tri.vertex(0).x(), tri.vertex(0).y(), tri.vertex(0).z());
			m::Vector3 v1 = m::Vector3(tri.vertex(1).x(), tri.vertex(1).y(), tri.vertex(1).z());
			m::Vector3 v2 = m::Vector3(tri.vertex(2).x(), tri.vertex(2).y(), tri.vertex(2).z());

			m::Vector3 normal = normalize(cross(v1 - v0, v2 - v0));

			float d = 1.0f - dot(normal, v0);
			if (d > disp)
			{
				disp = d;
				dispP = normal;
			}
		}
	}
}

void Light::renderPrimaryShadowMap(void (*render)(void*), void* user)
{
	shadowMap.render(render, user);
}

void Light::cullUnseenLights(class RayTracer* rt)
{
	int culled = 0;

	/* Cull unseen secondary lights. */

	for (int i = 0; i < subLightCount; i++)
	{
		m::Vector3 pos;
		m::Vector3 normal;
		m::Vector3 dir;
		m::Vector3 color;

		if (subLights[i].valid == false)
			continue;

		/* Test if this one is still valid. */

		dir = subLights[i].position - position;
		dir.normalize();
		if (type == SPOT && m::dot(dir, direction) < 0.0f)
		{
			subLights[i].valid = false;
			culled++;
			continue;
		}

		rt->intersect(position, dir, pos, normal, color);

		/* TODO: check also dot product of normals? */
		if ((subLights[i].position - pos).length() > 0.05f)
		{
			subLights[i].valid = false;
			culled++;
			continue;
		}
	}

	printf("Culled: %d   \n", culled);
}

//#undef REPORT_TIME
//#define REPORT_TIME(x)

void Light::updateSecondaryLights(class RayTracer* rt, void (*srender)(void*), void* suser, bool initial)
{
	if (CONFI(reuse_secondary_lights) == 0)
	{
		for (int i = 0; i < subLightCount; i++)
			subLights[i].valid = false;
	}

	cullUnseenLights(rt);
	REPORT_TIME("cull");

	/* Remove some secondary light sources. */

	updateDelaunay();
	REPORT_TIME("triangulation");
  
	int validSubLights = 0;
	for (int i = 0; i < subLightCount; i++)
		if (subLights[i].valid)
			validSubLights++;

	int unusedSubLights = subLightCount - validSubLights;

	const int recalcMinLights = CONFI(reuse_secondary_lights) ? CONFI(recalc_lights_min) : 0;
	const int recalcMaxLights = (!initial && CONFI(reuse_secondary_lights)) ? CONFI(recalc_lights_max) : subLightCount;

	int remove = std::max(0, recalcMinLights - unusedSubLights);

	int removed = 0;

	if (type == SPOT)
	{
		for (int k = 0; k < remove; k++)
		{
			Triangulation2::Vertex_handle best1 = NULL;
			Triangulation2::Vertex_handle best2 = NULL;
			float d = 100.0f;

			for (Triangulation2::Edge_iterator iter = tri2.edges_begin();
				 iter != tri2.edges_end();
				 iter++)
			{
				Triangulation2::Vertex_handle v1 = iter->first->vertex(tri2.cw(iter->second));
				Triangulation2::Vertex_handle v2 = iter->first->vertex(tri2.ccw(iter->second));

				float d2 = CGAL::squared_distance(v1->point(), v2->point());
				if (d2 < d)
				{
					d = d2;
					best1 = v1;
					best2 = v2;
				}
			}

			if (best1 != NULL)
			{
				float minD1 = 1000.0f;
				float minD2 = 1000.0f;

				Triangulation2::Vertex_circulator vc = tri2.incident_vertices(best1), done(vc);
				assert(vc != 0);

				do
				{
					if (vc != best2)
					{
						float d = CGAL::squared_distance(best1->point(), vc->point());
						if (d < minD1)
							minD1 = d;
					}
				} while(++vc != done);

				vc = tri2.incident_vertices(best2);
				done = vc;
				assert(vc != 0);

				do
				{
					if (vc != best1)
					{
						float d = CGAL::squared_distance(best2->point(), vc->point());
						if (d < minD2)
							minD2 = d;
					}
				} while(++vc != done);

				if (minD1 < minD2)
				{
					subLights[indices2[best1]].valid = false;
					tri2.remove(best1);
					indices2.erase(indices2.find(best1));
				}
				else
				{
					subLights[indices2[best2]].valid = false;
					tri2.remove(best2);
					indices2.erase(indices2.find(best2));
				}

				removed++;
			}
		}
	}
	else
	{
		for (int k = 0; k < remove; k++)
		{
			Triangulation3::Vertex_handle best1 = NULL;
			Triangulation3::Vertex_handle best2 = NULL;
			float d = 100.0f;

			for (Triangulation3::Finite_edges_iterator iter = tri3.finite_edges_begin();
				 iter != tri3.finite_edges_end();
				 iter++)
			{
				Triangulation3::Vertex_handle v1 = iter->first->vertex(iter->second);
				Triangulation3::Vertex_handle v2 = iter->first->vertex(iter->third);

				float d2 = CGAL::squared_distance(v1->point(), v2->point());
				if (d2 < d)
				{
					d = d2;
					best1 = v1;
					best2 = v2;
				}
			}

			if (best1 != NULL)
			{
				float minD1 = 1000.0f;
				float minD2 = 1000.0f;

				std::vector<Triangulation3::Vertex_handle> vertices;
				tri3.incident_vertices(best1, std::back_inserter(vertices));

				for (unsigned int i = 0; i < vertices.size(); i++)
				{
					if (vertices[i] == best2)
						continue;

					float d2 = CGAL::squared_distance(best1->point(), vertices[i]->point());
					if (d2 < minD1)
						minD1 = d2;
				}

				vertices.clear();
				tri3.incident_vertices(best2, std::back_inserter(vertices));

				for (unsigned int i = 0; i < vertices.size(); i++)
				{
					if (vertices[i] == best1)
						continue;

					float d2 = CGAL::squared_distance(best1->point(), vertices[i]->point());
					if (d2 < minD2)
						minD2 = d2;
				}

				if (minD1 < minD2)
				{
					subLights[indices3[best1]].valid = false;
					tri3.remove(best1);
					indices3.erase(indices3.find(best1));
				}
				else
				{
					subLights[indices3[best2]].valid = false;
					tri3.remove(best2);
					indices3.erase(indices3.find(best2));
				}

				removed++;
			}
		}
	}
	REPORT_TIME("removing");

	printf("Removed light: %d   \n", removed);

	/* Insert new secondary lights. */

	int shadowMapsRendered = 0;
	int b = recalcMaxLights;
	int reuse = CONFI(reuse_secondary_lights);

	double startTime = timer.time();
	double shadowMapRenderTime = 0.0;

	if (reuse)
		updateDispersion();

	for (int i = 0; i < subLightCount && b > 0; i++)
	{
		m::Vector3 pos;
		m::Vector3 normal;
		m::Vector3 dir;
		m::Vector3 color;

		if (subLights[i].valid)
			continue;

		m::Vector3 bestP;

		if (!reuse || disp == 0.0f || initial)
		{
			float sx = (i + 0.5f) / (float)subLightCount;
			float sy = halton2(i + 1);

			if (type == SPOT)
			{
				double a = sx * 3.14159265 * 2.0;
				double l = sqrt(sy);
				bestP.x = cos(a) * l;
				bestP.y = sin(a) * l;
			}
			else
			{
				bestP.y = sy * 2.0f - 1.0f;
				double a = sx * 3.14159265 * 2.0;
				double r = sqrt(1.0 - bestP.y * bestP.y);
				bestP.x = cos(a) * r;
				bestP.z = sin(a) * r;
			}

			disp = 1.0f;
		}
		else
		{
			bestP = dispP;
			if (type == SPOT && bestP.length() > 0.99f)
				bestP = normalize(bestP) * 0.99f;
		}

		if (type == OMNI)
		{
			dir = bestP;
		}
		else
		{
			float l = dot(bestP, bestP);
			float z;
			if (l >= 1.0f)
				z = 0.0f;
			else
				z = sqrtf(1.0f - l);

			dir = right * bestP.x + up * bestP.y + direction * z;
		}

		bool ret = rt->intersect(position, dir, pos, normal, color);
		if (!ret)
		{
			printf("A ray missed. This should be really rare in our test scenes!\n");
			subLights[i].valid = false;
			disp = 0.0f;
			continue;
		}

		if (reuse)
			updateDispersion();

		double startTime = timer.time();

		subLights[i].position = pos;
		subLights[i].p = bestP;
		subLights[i].direction = normal;
		subLights[i].nonWeightedIntensity = scale(color, intensity);// * (1.0f / subLightCount);

		subLights[i].shadowMap.setPosition(subLights[i].position + normal * 0.05f);
		subLights[i].shadowMap.setDirection(subLights[i].direction);

		subLights[i].shadowMap.render(srender, suser);

		if (CONFI(timer_individuals))
		{
			glFlush();
			glFinish();
			shadowMapRenderTime += (timer.time() - startTime) * 1000.0;
		}

		shadowMapsRendered++;

		if (reuse)
		{
			if (type == SPOT)
			{
				Triangulation2::Vertex_handle h = tri2.insert(Triangulation2::Point(bestP.x, bestP.y));
				indices2[h] = i;
			}
			else
			{
				Triangulation3::Vertex_handle h = tri3.insert(Triangulation3::Point(bestP.x, bestP.y, bestP.z));
				indices3[h] = i;
			}
		}

		subLights[i].valid = true;
		b--;
	}

	double tim = timer.time() * 1000.0 - startTime * 1000.0;

	REPORT_TIME("insert");

	if (timer_individuals)
	{
		static AverageTiming t1, t2;
		t1.insert(tim - shadowMapRenderTime);
		t2.insert(shadowMapRenderTime);
		printf("=> cpu %.1f  gpu %.1f    \n", t1.getAverage(), t2.getAverage());
	}
	
	printf("Rendered shadow maps: %d   \n", shadowMapsRendered);

	if (reuse)
		updateVoronoi();

	for (int i = 0; i < subLightCount; i++)
	{
		if (subLights[i].valid)
		{
			subLights[i].intensity =
				subLights[i].nonWeightedIntensity * subLights[i].area * ((type == SPOT) ? 1.0f : 2.0f);

			if (subLights[i].intensity.length() < 0.001f)
				subLights[i].valid = false;
			//printf("%d: %f\n", i, subLights[i].intensity.x + subLights[i].intensity.y +subLights[i].intensity.z);
		}
	}
	REPORT_TIME("voronoi");

	if (!reuse)
	{
		for (int i = 0; i < subLightCount; i++)
		{
			subLights[i].intensity =
				subLights[i].nonWeightedIntensity * (1.0f / subLightCount) * ((type == SPOT) ? 1.0f : 2.0f);
		}
	}

	//printf("AREA SUM: %f\n", A);

	int inUse = 0;
	for (int i = 0; i < subLightCount; i++)
		if (subLights[i].valid)
			inUse++;
	printf("Sublights in use: %d\n", inUse);
}

void Light::outputPS()
{
	FILE* fp = fopen((type == Light::SPOT) ? "circlevoronoi.ps" : "spherevoronoi.ps", "w");
	fprintf(fp, "%%!PS\n");
	fprintf(fp, "newpath 100 100 99.5 0 360 arc stroke\n");

	if (type == Light::SPOT)
	{
		for (int i = 0; i < subLightCount; i++)
		{
			if (subLights[i].valid)
				fprintf(fp, "newpath %f %f 1.0 0 360 arc fill\n",
					100.0f + subLights[i].p.x * 99.5f,
					100.0f + subLights[i].p.y * 99.5f);
		}
		fprintf(fp, "newpath\n");
		for (unsigned int i = 0; i < edges2.size(); i++)
		{
			const Light::VoronoiEdge2& e = edges2[i];
			fprintf(fp, "%f %f moveto %f %f lineto\n",
				100.0f + e.p[0].x * 99.5f,
				100.0f + e.p[0].y * 99.5f,
				100.0f + e.p[1].x * 99.5f,
				100.0f + e.p[1].y * 99.5f);
		}
		fprintf(fp, "stroke\n");
	}
	else
	{
		for (int i = 0; i < subLightCount; i++)
		{
			if (subLights[i].p.z < 0 || !subLights[i].valid)
				continue;

			fprintf(fp, "newpath %f %f 1 0 360 arc fill\n",
				100.0f + subLights[i].p.x * 99.5f,
				100.0f + subLights[i].p.y * 99.5f);

			m::Vector3 p = subLights[i].p * 1.1f;
			fprintf(fp, "newpath %f %f moveto %f %f lineto stroke\n",
				100.0f + subLights[i].p.x * 99.5f,
				100.0f + subLights[i].p.y * 99.5f,
				100.0f + p.x * 99.5f,
				100.0f + p.y * 99.5f);
		}

		fprintf(fp, "newpath\n");
		for (unsigned int i = 0; i < debugVoronoiEdges3.size(); i++)
		{
			const m::Vector3& a = debugVoronoiEdges3[i].first;
			const m::Vector3& b = debugVoronoiEdges3[i].second;

			if (a.z <= 0.0f && b.z <= 0.0f)
				continue;

			fprintf(fp, "%f %f moveto",
				100.0f + a.x * 99.5f,
				100.0f + a.y * 99.5f);

			for (int j = 1; j <= 8; j++)
			{
				m::Vector3 p = normalize(a + (b - a) * (j / 8.0f));
				if (p.z < 0.0f)
				{
					p.z = 0;
					p.normalize();
				}
				fprintf(fp, " %f %f %s",
					100.0f + p.x * 99.5f,
					100.0f + p.y * 99.5f,
					(p.z < 0.0f) ? "moveto" : "lineto");
			}
			fprintf(fp, "\n");

		}
		fprintf(fp, "stroke\n");
	}

	fprintf(fp, "showpage\n");
	fclose(fp);

	if (type == Light::SPOT)
	{
		FILE* fp = fopen("hemivoronoi.ps", "w");
		fprintf(fp, "%%!PS\n");
		fprintf(fp, "newpath 100 100 99.5 0 180 arc stroke\n");

		float a = 3.14159265f * 2.0f * 20.0f / 360.0f;
		float dy = sinf(a);
		float dz = cosf(a);

		for (int i = 0; i < subLightCount; i++)
			for (int j = i+1; i < subLightCount; i++)
			{
				if (subLights[i].p.y < subLights[j].p.y)
					std::swap(subLights[i], subLights[j]);
			}

		for (int i = 0; i < subLightCount; i++)
		{
			if (!subLights[i].valid)
				continue;

			float z = subLights[i].p.length();
			z = sqrtf(1.0f - z * z);
			float y0 = subLights[i].p.y * dy;
			float y1 = subLights[i].p.y * dy + z * dz;
			float y2 = subLights[i].p.y * 1.1f * dy + z * 1.1f * dz;

			if (y0 * dz - z * dy > 0.0f)
				continue;

			fprintf(fp, "newpath %f %f 1.0 0 360 arc fill\n",
				100.0f + subLights[i].p.x * 99.5f,
				100.0f + y1 * 99.5f);

			m::Vector3 p = subLights[i].p * 1.1f;
			fprintf(fp, "newpath %f %f moveto %f %f lineto stroke\n",
				100.0f + subLights[i].p.x * 99.5f,
				100.0f + y1 * 99.5f,
				100.0f + subLights[i].p.x * 1.1f * 99.5f,
				100.0f + y2 * 99.5f);
		}

		fprintf(fp, "newpath\n");
		for (unsigned int i = 0; i < edges2.size(); i++)
		{
			m::Vector3 a = m::Vector3(edges2[i].p[0].x, edges2[i].p[0].y, 0.0f);
			m::Vector3 b = m::Vector3(edges2[i].p[1].x, edges2[i].p[1].y, 0.0f);

			a.z = sqrtf(1.00001f - dot(a, a));
			b.z = sqrtf(1.00001f - dot(b, b));

			if ((a.y * dz - a.z * dy > 0.0f) && (b.y * dz - b.z * dy > 0.0f))
				continue;

			fprintf(fp, "%f %f moveto",
				100.0f + a.x * 99.5f,
				100.0f + (a.y * dy  + a.z * dz) * 99.5f);

			for (int j = 1; j <= 8; j++)
			{
				m::Vector3 p = normalize(a + (b - a) * (j / 8.0f));
				float y = p.y * dy + p.z * dz;

				fprintf(fp, " %f %f %s",
					100.0f + p.x * 99.5f,
					100.0f + y * 99.5f,
					(p.y * dz - p.z * dy > 0.0f) ? "moveto" : "lineto");
			}
			fprintf(fp, "\n");
		}
		fprintf(fp, "stroke\n");

		fprintf(fp, "100 100 translate\n");
		fprintf(fp, "99.5 %f scale\n", dy * 99.5);
		fprintf(fp, "0.01 setlinewidth\n");
		fprintf(fp, "newpath 0 0 1.0 180 360 arc stroke\n");
		fclose(fp);
	}
}


