#include "glutil.h"
#include "configuration.h"
#include <string>
#include <iostream>

void setUniform(GLuint program, const std::string& name, const m::Vector3& value)
{
  glUniform3fARB(glGetUniformLocationARB(program,name.c_str()),value[0],value[1],value[2]);
}

void setUniform(GLuint program, const std::string& name, float value)
{
  glUniform1fARB(glGetUniformLocationARB(program,name.c_str()),value);
}

void setUniform(GLuint program, const std::string& name, unsigned int value)
{
  glUniform1iARB(glGetUniformLocationARB(program,name.c_str()),value);
}

void setUniform4x4Mat(GLuint program, const std::string& name, const double *value)
{
  GLfloat mat[16];
  for(int i = 0; i < 16; i++) mat[i] = (GLfloat) value[i];
  glUniformMatrix4fvARB(glGetUniformLocationARB(program,name.c_str()),1,false,mat);
}

void setUniform4x4Mat(GLuint program, const std::string& name, const float *value)
{
  glUniformMatrix4fvARB(glGetUniformLocationARB(program,name.c_str()),1,false,value);
}

static void substitute_configurations(const char* src, char* dst)
{
	while (*src)
	{
		if (*src == '$')
		{
			int type;
			src++;

			if (*src == 'i')
				type = 0;
			else if (*src == 'f')
				type = 1;
			else if (*src == 's')
				type = 2;
			else
			{
				fprintf(stderr, "bad configuration type '%c'\n", *src);
				exit(EXIT_FAILURE);
			}

			src++;
			if (*src != ':')
			{
				fprintf(stderr, "expecting ':' in configuration variable\n");
				exit(EXIT_FAILURE);
			}
			src++;

			const char* ks = src;

			while (isalnum(*src) || *src == '_')
				src++;

			char key[128];
			strncpy(key, ks, src - ks);
			key[src - ks] = '\0';

			switch (type)
			{
			case 0: sprintf(dst, "%d", config.getInt(key)); break;
			case 1: sprintf(dst, "%f", config.getFloat(key)); break;
			case 2: sprintf(dst, "%s", config.getString(key)); break;
			}

			// TODO: check overflow
			dst += strlen(dst);
		}
		else
		{
			*dst++ = *src++;
		}
	}

	*dst = '\0';
}

GLuint makeGLSLProgram(const char*fragSrc2, const char*vtxSrc2, const char* fname, const char* vname)
{
	char fragSrcBuf[1024*8];
	char vtxSrcBuf[1024*8];

	const char* fragSrc = NULL;
	const char* vtxSrc = NULL;

	// TODO: fix overflow

	if (fragSrc2)
	{
		substitute_configurations(fragSrc2, fragSrcBuf);
		fragSrc = fragSrcBuf;
	}
	if (vtxSrc2)
	{
		substitute_configurations(vtxSrc2, vtxSrcBuf);
		vtxSrc = vtxSrcBuf;
	}

      GLuint program;
      GLint stat;
      char log[1001];
      GLsizei len;
      program = glCreateProgramObjectARB();

      if(fragSrc)
      {
        GLuint fragPrg = glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB);
        glShaderSourceARB(fragPrg,1,&fragSrc,NULL);
        glCompileShaderARB(fragPrg);
        glGetObjectParameterivARB(fragPrg,GL_OBJECT_COMPILE_STATUS_ARB,&stat);
        glGetInfoLogARB(fragPrg,1000,&len,log);
		log[len] = '\0';
        std::cout << "FRAGMENT PROGRAM " << fname << ": " << log << std::endl;
		if (strstr(log, "error"))// || strstr(log, "warning"))
			exit(EXIT_FAILURE);
        glAttachObjectARB(program,fragPrg);
      }

      if(vtxSrc)
      {
        GLuint vtxPrg = glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB);
        glShaderSourceARB(vtxPrg,1,&vtxSrc,NULL);
        glCompileShaderARB(vtxPrg);
        glGetObjectParameterivARB(vtxPrg,GL_OBJECT_COMPILE_STATUS_ARB,&stat);
        glGetInfoLogARB(vtxPrg,1000,&len,log);   
		log[len] = '\0';
        std::cout << "VERTEX PROGRAM " << vname << ": " << log << std::endl;
		if (strstr(log, "error"))// || strstr(log, "warning"))
			exit(EXIT_FAILURE);
        glAttachObjectARB(program,vtxPrg);
      }
      glLinkProgramARB(program);
      return program;
}

void makeDepthBufferObject(GLuint& shadowMapFB, GLuint& shadowTexId, int shadowMapSize)
{
    glGenTextures(1,&shadowTexId);
    glBindTexture(GL_TEXTURE_2D,shadowTexId);

    glTexImage2D(GL_TEXTURE_2D,0,GL_DEPTH_COMPONENT16,shadowMapSize,shadowMapSize,0,GL_DEPTH_COMPONENT,GL_UNSIGNED_BYTE,NULL);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);   
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);

    glGenFramebuffersEXT(1,&shadowMapFB);
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,shadowMapFB);
    glFramebufferTexture2DEXT( GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, shadowTexId, 0 );
    glDrawBuffer(GL_NONE);
    glReadBuffer(GL_NONE);
    // verify all is well and restore state

		GLenum status = glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);

		switch(status)
		{
		case GL_FRAMEBUFFER_COMPLETE_EXT:
			break;
		case GL_FRAMEBUFFER_UNSUPPORTED_EXT:
			// Choose different format
			std::cout << "Warning: Unsupported framebuffer object format. Try another format." << std::endl;
			break;
		case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT:
			std::cout << "Error: Framebuffer object incomplete - attachment." << std::endl;
			break;
		case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT:
			std::cout << "Error: Framebuffer object incomplete - missing attachment." << std::endl;
			break;
#if 0
		case GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT:
			std::cout << "Error: Framebuffer object incomplete - duplicate attachment." << std::endl;
			break;
#endif
		case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT:
			std::cout << "Error: Framebuffer object incomplete - dimensions." << std::endl;
			break;
		case GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT:
			std::cout << "Error: Framebuffer object incomplete - formats." << std::endl;
			break;
		case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT:
			std::cout << "Error: Framebuffer object incomplete - draw buffer." << std::endl;
			break;
		case GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT:
			std::cout << "Error: Framebuffer object incomplete - read buffer." << std::endl;
			break;
		default:
			// Programming error; will fail on all hardware
			assert(0);
			break;
		}
    glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, 0);
}

#define PI_ 3.14159265358979323846

/* accFrustum()
 * The first 6 arguments are identical to the glFrustum() call.
 *  
 * pixdx and pixdy are anti-alias jitter in pixels. 
 * Set both equal to 0.0 for no anti-alias jitter.
 * eyedx and eyedy are depth-of field jitter in pixels. 
 * Set both equal to 0.0 for no depth of field effects.
 *
 * focus is distance from eye to plane in focus. 
 * focus must be greater than, but not equal to 0.0.
 *
 * Note that accFrustum() calls glTranslatef().  You will 
 * probably want to insure that your ModelView matrix has been 
 * initialized to identity before calling accFrustum().
 */
void accFrustum(GLdouble left, GLdouble right, GLdouble bottom, 
   GLdouble top, GLdouble near, GLdouble far, GLdouble pixdx, 
   GLdouble pixdy, GLdouble eyedx, GLdouble eyedy, GLdouble focus)
{
   GLdouble xwsize, ywsize; 
   GLdouble dx, dy;
   GLint viewport[4];

   glGetIntegerv (GL_VIEWPORT, viewport);
	
   xwsize = right - left;
   ywsize = top - bottom;
	
   dx = -(pixdx*xwsize/(GLdouble) viewport[2] + eyedx*near/focus);
   dy = -(pixdy*ywsize/(GLdouble) viewport[3] + eyedy*near/focus);
	
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glFrustum (left + dx, right + dx, bottom + dy, top + dy, near, far);
   //glMatrixMode(GL_MODELVIEW);
   //glLoadIdentity();
   //glTranslatef (-eyedx, -eyedy, 0.0);
}

/* accPerspective()
 * 
 * The first 4 arguments are identical to the gluPerspective() call.
 * pixdx and pixdy are anti-alias jitter in pixels. 
 * Set both equal to 0.0 for no anti-alias jitter.
 * eyedx and eyedy are depth-of field jitter in pixels. 
 * Set both equal to 0.0 for no depth of field effects.
 *
 * focus is distance from eye to plane in focus. 
 * focus must be greater than, but not equal to 0.0.
 *
 * Note that accPerspective() calls accFrustum().
 */
void accPerspective(GLdouble fovy, GLdouble aspect, 
   GLdouble near, GLdouble far, GLdouble pixdx, GLdouble pixdy, 
   GLdouble eyedx, GLdouble eyedy, GLdouble focus)
{
   GLdouble fov2,left,right,bottom,top;

   fov2 = ((fovy*PI_) / 180.0) / 2.0;

   top = near / (cos(fov2) / sin(fov2));
   bottom = -top;

   right = top * aspect;
   left = -right;

   accFrustum (left, right, bottom, top, near, far,
               pixdx, pixdy, eyedx, eyedy, focus);
}
