#ifndef M_VECTOR_H
#define M_VECTOR_H

#include <cassert>
#include <cmath>

#include "Defs.h"

/// The m namespace contains the various math classes.
namespace m
{
/** @cond DOCUMENT_TEMPLATES **/

template<class T, int N, class DV, class DVT, bool R>
class VectorBase
{
public:
	// Operators
	inline T&			operator[](int i)						{ assert(i >= 0 && i < N); return ((T*)this)[i]; }
	inline const T&		operator[](int i) const					{ assert(i >= 0 && i < N); return ((T*)this)[i]; }
	inline DV&			operator+= (const VectorBase& v)		{ for(int i = 0; i < N; i++) (*this)[i] += v[i]; return *((DV*)this); }
	inline DV&			operator-= (const VectorBase& v)		{ for(int i = 0; i < N; i++) (*this)[i] -= v[i]; return *((DV*)this); }
	inline DV&			operator*= (T f)						{ for(int i = 0; i < N; i++) (*this)[i] *= f; return *((DV*)this); }
	inline const DV		operator- () const						{ return T(-1) * (*this); }
	inline const DV		operator+ () const						{ return *((DV*)this); }

	// Methods
	inline const T		length() const							{ T lensq = 0; for(int i = 0; i < N; i++) lensq += (*this)[i] * (*this)[i]; return T(sqrt(lensq)); }
	inline void			negate()								{ for(int i = 0; i < N; i++) *this[i] = -(*this)[i]; }
	inline void			normalize()								{ T s = length(); assert(s != 0); s = T(1) / s; for(int i = 0; i < N; i++) (*this)[i] *= s; }
	inline void			normalize0()							{ T s = length(); if(s == 0) return; s = T(1) / s; for(int i = 0; i < N; i++) (*this)[i] *= s; }
	inline void			zero()									{ for(int i = 0; i < N; i++) (*this)[i] = T(0); }

protected:
	// Protected default constructor to prevent VectorBase instantiation
	inline				VectorBase(void) {}
};

template<class T> class RVector2T;

template<class T>
class Vector2T : public VectorBase<T, 2, Vector2T<T>, RVector2T<T>, false>
{
public:
	T				x, y;

	inline			Vector2T(void)						{}
	inline			Vector2T(T x_, T y_)				: x(x_), y(y_) {}
	inline void		set(T x_, T y_)						{ x = x_; y = y_; }
};

template<class T>
class RVector2T : public VectorBase<T, 2, RVector2T<T>, Vector2T<T>, true>
{
public:
	T				x, y;

	inline			RVector2T(void)						{}
	inline			RVector2T(T x_, T y_)				: x(x_), y(y_) {}
	inline void		set(T x_, T y_)						{ x = x_; y = y_; }
};

template<class T> class RVector3T;

template<class T> 
class Vector3T : public VectorBase<T, 3, Vector3T<T>, RVector3T<T>, false>
{
public:
	T				x, y, z;
	inline			Vector3T(void)						{}
	inline			Vector3T(T x_, T y_, T z_)			: x(x_), y(y_), z(z_) {}
	inline void		set(T x_, T y_, T z_)				{ x = x_; y = y_; z = z_; }
};

template<class T> 
class RVector3T : public VectorBase<T, 3, RVector3T<T>, Vector3T<T>, true>
{
public:
	T				x, y, z;
	inline			RVector3T(void)						{}
	inline			RVector3T(T x_, T y_, T z_)			: x(x_), y(y_), z(z_) {}
	inline void		set(T x_, T y_, T z_)				{ x = x_; y = y_; z = z_; }
};

template<class T> class RVector4T;

template<class T> 
class Vector4T : public VectorBase<T, 4, Vector4T<T>, RVector4T<T>, false>
{
public:
	T				x, y, z, w;
	inline			Vector4T(void)						{}
	inline			Vector4T(T x_, T y_, T z_, T w_)	: x(x_), y(y_), z(z_), w(w_) {}
	inline void		set(T x_, T y_, T z_, T w_)			{ x = x_; y = y_; z = z_; w = w_; }
};

template<class T> 
class RVector4T : public VectorBase<T, 4, RVector4T<T>, Vector4T<T>, true>
{
public:
	T				x, y, z, w;
	inline			RVector4T(void)						{}
	inline			RVector4T(T x_, T y_, T z_, T w_)	: x(x_), y(y_), z(z_), w(w_) {}
	inline void		set(T x_, T y_, T z_, T w_)			{ x = x_; y = y_; z = z_; w = w_; }
};

// Global operators
template<class T, int N, class DV, class DVT, bool R>
inline bool operator==(const VectorBase<T, N, DV, DVT, R>& a, const VectorBase<T, N, DV, DVT, R>& b)
{
	for(int i = 0; i < N; i++)
		if(a[i] != b[i]) return false;

	return true;
}

template<class T, int N, class DV, class DVT, bool R>
inline bool operator!=(const VectorBase<T, N, DV, DVT, R>& a, const VectorBase<T, N, DV, DVT, R>& b)
{
	for(int i = 0; i < N; i++)
		if(a[i] != b[i]) return true;

	return false;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV operator*(T f, const VectorBase<T, N, DV, DVT, R>& v)
{
	DV result;
	for(int i = 0; i < N; i++)
		result[i] = v[i] * f;
    
	return result;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV operator*(const VectorBase<T, N, DV, DVT, R>& v, T f)
{
	DV result;
	for(int i = 0; i < N; i++)
		result[i] = v[i] * f;
    
	return result;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV operator/(const VectorBase<T, N, DV, DVT, R>& v, T f)
{
	T div = T(1) / f;
	return v * div;
}

template<class T, int N, class DCV, class DRV>
inline T operator*(const VectorBase<T, N, DRV, DCV, true>& a, const VectorBase<T, N, DCV, DRV, false>& b)
{
	float d = 0.f;
	for (int i=0; i < N; i++)
		d += a[i]*b[i];
	return d;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV operator+(const VectorBase<T, N, DV, DVT, R>& a, const VectorBase<T, N, DV, DVT, R>& b)
{
	DV result;
	for (int i=0; i < N; i++)
		result[i] = a[i] + b[i];
	return result;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV operator-(const VectorBase<T, N, DV, DVT, R>& a, const VectorBase<T, N, DV, DVT, R>& b)
{
	DV result;
	for (int i=0; i < N; i++)
		result[i] = a[i] - b[i];
	return result;
}

// Global methods
template<class T, int N, class DV, class DVT>
inline const T dot(const VectorBase<T, N, DV, DVT, false>& a, const VectorBase<T, N, DV, DVT, false>& b)
{
	T result = 0.f;
	for (int i=0; i < N; i++)
		result += a[i] * b[i];
	return result;
}

template<class T, int N, class DV, class DVT>
inline const DV vecmin(const VectorBase<T, N, DV, DVT, false>& a, const VectorBase<T, N, DV, DVT, false>& b)
{
	DV result;
	for (int i=0; i < N; i++)
		result[i] = a[i] < b[i] ? a[i] : b[i];
	return result;
}

template<class T, int N, class DV, class DVT>
inline const DV vecmax(const VectorBase<T, N, DV, DVT, false>& a, const VectorBase<T, N, DV, DVT, false>& b)
{
	DV result;
	for (int i=0; i < N; i++)
		result[i] = a[i] > b[i] ? a[i] : b[i];
	return result;
}

template<class T, int N, class DV, class DVT>
inline const DV fabs(const VectorBase<T, N, DV, DVT, false>& a)
{
	DV result;
	for (int i=0; i < N; i++)
		result[i] = ::fabs( a[i] );
	return result;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV scale(const VectorBase<T, N, DV, DVT, R>& a, const VectorBase<T, N, DV, DVT, R>& b)
{
	DV result;
	for (int i=0; i < N; i++)
		result[i] = a[i] * b[i];
	return result;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV normalize(const VectorBase<T, N, DV, DVT, R>& v)
{
	T s = v.length();
	assert(s != T(0));
	
	s = T(1) / s;

	return s * v;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV normalize0(const VectorBase<T, N, DV, DVT, R>& v)
{
	T s = v.length();
	if(s == T(0)) return *((const DV*)&v);
	
	s = T(1) / s;

	return s * v;
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV project ( const VectorBase<T, N, DV, DVT, R>& v, const VectorBase<T, N, DV, DVT, R>& onThis ) 
{
	T l = dot(onThis, onThis); 

	if( l != T(0) ) l = dot(v, onThis) / l; 
	return onThis * l; 
}

template<class T, int N, class DV, class DVT, bool R>
inline const DV lerp ( const VectorBase<T, N, DV, DVT, R>& v1, const VectorBase<T, N, DV, DVT, R>& v2, T ratio ) 
{
	return v1 + ratio * (v2 - v1); 
}

// This seems to work but it would be nice if we could prove it
template<class T, int N, class DV, class DVT, bool B>
const DVT& transpose(const VectorBase<T, N, DV, DVT, B>& v)
{
	return *((const DVT*)&v);
}

template<class T>
Vector3T<T> cross(const Vector3T<T>& a, const Vector3T<T>& b)
{
	return Vector3T<T>(	a[1] * b[2] - a[2] * b[1],
						a[2] * b[0] - a[0] * b[2],
						a[0] * b[1] - a[1] * b[0]); 
}

/** @endcond DOCUMENT_TEMPLATES **/

/// A two-dimensional column vector of floats
typedef Vector2T<float>		Vector2;
/// A three-dimensional column vector of floats
typedef Vector3T<float>		Vector3;
/// A four-dimensional column vector of floats
typedef Vector4T<float>		Vector4;

/// A two-dimensional column vector of doubles
typedef Vector2T<double>	Vector2d;
/// A three-dimensional column vector of doubles
typedef Vector3T<double>	Vector3d;
/// A four-dimensional column vector of doubles
typedef Vector4T<double>	Vector4d;

/// A two-dimensional column vector of ints
typedef Vector2T<int>		Vector2i;
/// A three-dimensional column vector of ints
typedef Vector3T<int>		Vector3i;
/// A four-dimensional column vector of ints
typedef Vector4T<int>		Vector4i;

/// A two-dimensional row vector of floats
typedef RVector2T<float>	RVector2;
/// A three-dimensional row vector of floats
typedef RVector3T<float>	RVector3;
/// A four-dimensional row vector of floats
typedef RVector4T<float>	RVector4;

/// A two-dimensional row vector of doubles
typedef RVector2T<double>	RVector2d;
/// A three-dimensional row vector of doubles
typedef RVector3T<double>	RVector3d;
/// A four-dimensional row vector of doubles
typedef RVector4T<double>	RVector4d;

/// A two-dimensional row vector of ints
typedef RVector2T<int>		RVector2i;
/// A three-dimensional row vector of ints
typedef RVector3T<int>		RVector3i;
/// A four-dimensional row vector of ints
typedef RVector4T<int>		RVector4i;

}

#endif // !M_VECTOR_H
