%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Demonstration of the Variational Bayesian Adaptive Kalman Filter
% with simple linear resonator model with 2D measurement.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Copyright (C) 2006 Simo Srkk
%
% $Id: vbakf_demo3.m,v 1.1 2006/12/13 22:48:07 ssarkka Exp $
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

  %
  % Model is
  %
  %   dx1/dt = w1(t)
  %  dx2/dt = - 0.05 x2 + w2(t),
  %      y_k = x1(k*dt) + x2(k*dt) + v_k
  %
  % where w(t) is a white noise process and
  % v_k is a Gaussian random variable.
  %
  dt = 0.1;
  m0 = [0;0;1];
  P0 = 0.01*eye(3);
  F = [0 0 0;
       0 0 1;
       0 -1/50 0];
  L = [1 0;
       0 0;
       0 1];
  H = [1 1 0;
       1 0 0];
  Qc = diag([0.1 0.001]);
  
  [A,Q] = lti_disc(F,L,Qc,dt);
  
  %
  % Simulate
  %
  steps = round(400/dt);
  x = gauss_rnd(m0,P0);
  t = 0;

  X = zeros(size(m0,1),steps);
  Y = zeros(2,steps);
  V = zeros(2,steps);
  T = zeros(1,steps);
  for k=1:steps
    if k<1.5*steps/4
      r1 = 0.2 + 0.4*(1+tanh(0.01*(k-steps/4)));
    else
      r1 = 0.2 + 0.4*(1+tanh(-0.01*(k-2*steps/4)));
    end
    if k<(1/2+2/3)/2*steps
      r2 = 0.1 + 0.5*(1+tanh(0.01*(k-steps/2)));
    else
      r2 = 0.1 + 0.5*(1+tanh(-0.01*(k-2*steps/3)));
    end
    R = diag([r1;r2]);
    x = gauss_rnd(A*x,Q);
    y = gauss_rnd(H*x,R);
    t = t + dt;
    X(:,k) = x;
    Y(:,k) = y;
    V(:,k) = [r1;r2];
    T(k)   = t;    
  end

  clf;
  plot(T,H*X,T,Y,'o');
  legend('x','y');
  fprintf('The data. press enter.\n');
  pause;

  vb_m = m0;
  vb_P = P0;
  vb_a = [1;1];
  vb_b = [1;1];
  
  akf_MM = zeros(size(m0,1),size(Y,2));
  akf_VV = zeros(2,size(Y,2));

  akf_Q_05 = zeros(2,size(Y,2));
  akf_Q_95 = zeros(2,size(Y,2));
  
  clf;
  
  for k=1:size(Y,2)
    [vb_m,vb_P,vb_a,vb_b] = vbakf_predict_diag(vb_m,vb_P,A,Q,vb_a,vb_b,0.99);

    [vb_m,vb_P,vb_a,vb_b] = vbakf_update_diag(vb_m,vb_P,...
                                              Y(:,k),H, ...
                                              vb_a,vb_b);

    akf_MM(:,k) = vb_m;
    akf_VV(:,k) = vb_b./(vb_a-1);
      
    for i=1:2
      theta = (0.05:0.01:2);
      p = -(vb_a(i)+1)*log(theta) - vb_b(i)./theta;
      p = p - median(p);
      p = exp(p);
      p = p./sum(p);

      c = cumsum(p);
      akf_Q_05(i,k) = theta(max([find(c < 0.05) 1]));
      akf_Q_95(i,k) = theta(min([find(c > 0.95) length(c)]));
    end

    %
    % Plot
    %
    if rem(k,50) == 0
      for i=1:2
        subplot(2,3,1+3*(i-1));

        theta = (0.05:0.01:2);
        p = -(vb_a(i)+1)*log(theta) - vb_b(i)./theta;
        p = p - median(p);
        p = exp(p);
        p = p./sum(p);

        r = V(i,k);
        plot(theta,p,[r r],[0 1.1*max(p)]);
        title('vbakf p(\theta)');
      
        subplot(2,3,2+3*(i-1));
        plot(T,H*X,'g--',T,Y,'.',T(1:k),H*akf_MM(:,1:k),'k-');
        title('vbakf estimate');

        subplot(2,3,3+3*(i-1));
        plot(T,V(i,:),'--',T(1:k),akf_VV(i,1:k),'-',...
             T(1:k),akf_Q_05(i,1:k),':',T(1:k),akf_Q_95(i,1:k),':');
        axis([0 T(end) 0 2*max(V(i,:))]);
        title('vbakf variance');
        legend('True','Estimated mean');
      end
      
      drawnow;
      pause(0.01);
    end
  end
  
  rmse_akf = mean(sum((X-akf_MM).^2,1));
  fprintf('AKF-RMSE = %.4f\n',rmse_akf);

