%EQ_DESIGN_COOKBOOK  Design discrete-time EQ w/ Audio Cookbook formulas
%
% Syntax:
%   [B,A] = eq_design_cookbook(type,fs,f,q,g_db,use_ee_q)
%
% In:
%   type - Type of the filter as string 'lowpass', 'highpass',
%          'bandpass', 'notch', 'allpass', 'peak', 'lowshelf',
%          'highself'
%     fs - Sampling frequency
%      f - Working frequency
%      q - The quality factor
%   g_db - Gain in decibels (for 'bandpass', 'peak', 'lowshelf'
%          and 'highshelf')
%   use_ee_q - Use EE's Q definition (1) or cookbook's (0)
%          in peak EQ (default 0)
%
% Out:
%      B - 1xN numerator filter polynomial coefficients
%      A - 1xN denumerator filter polynomial coefficients
%   
% Description:
%   Design a discrete-time biquad filter of the given type
%   based on direct Audio QE Cookbook formulas in
%
%   http://www.musicdsp.org/files/Audio-EQ-Cookbook.txt

% References:
%   http://www.musicdsp.org/files/Audio-EQ-Cookbook.txt
%
% Copyright (C) Simo Srkk, 2008-2009

function [B,A] = eq_design_cookbook(type,fs,f,q,g_db,use_ee_q)

  if nargin < 6
      use_ee_q = [];
  end
  if isempty(use_ee_q)
      use_ee_q = 0;
  end

  a = 10^(g_db/40);
  w0 = 2*pi*f/fs;
  
  cw0 = cos(w0);
  sw0 = sin(w0);

  if use_ee_q && strcmp(type, 'peak')
      q = q/a;
  end
  
  alpha = sw0/(2*q);
  tmp = 2*sqrt(a)*alpha;
  
  switch lower(type)
   case 'lowpass'      
      b0 = (1-cw0)/2;
      b1 = 1-cw0;
      b2 = (1-cw0)/2;
      a0 = 1+alpha;
      a1 = -2*cw0;
      a2 = 1-alpha;
    case 'highpass'
      b0 = (1+cw0)/2;
      b1 = -(1+cw0);
      b2 = (1+cw0)/2;
      a0 = 1+alpha;
      a1 = -2*cw0;
      a2 = 1-alpha;
    case 'bandpass'
      b0 = a*alpha;  % Gain a added
      b1 = 0;
      b2 = -a*alpha; % Gain a added
      a0 = 1+alpha;
      a1 = -2*cw0;
      a2 = 1-alpha;
    case 'notch'
      b0 = 1;
      b1 = -2*cw0;
      b2 = 1;
      a0 = 1+alpha;
      a1 = -2*cw0;
      a2 = 1-alpha;
    case 'allpass'
      b0 = 1-alpha;
      b1 = -2*cw0;
      b2 = 1+alpha;
      a0 = 1+alpha;
      a1 = -2*cw0;
      a2 = 1-alpha;
    case 'peak'
      b0 = 1+alpha*a;
      b1 = -2*cw0;
      b2 = 1-alpha*a;
      a0 = 1+alpha/a;
      a1 = -2*cw0;
      a2 = 1-alpha/a;
    case 'lowshelf'
      b0 =   a*((a+1)-(a-1)*cw0+tmp);
      b1 = 2*a*((a-1)-(a+1)*cw0);
      b2 =   a*((a+1)-(a-1)*cw0-tmp);
      a0 =      (a+1)+(a-1)*cw0+tmp;
      a1 =  -2*((a-1)+(a+1)*cw0);
      a2 =      (a+1)+(a-1)*cw0-tmp;
    case 'highshelf'
      b0 =   a*((a+1)+(a-1)*cw0+tmp);
      b1 =-2*a*((a-1)+(a+1)*cw0);
      b2 =   a*((a+1)+(a-1)*cw0-tmp);
      a0 =      (a+1)-(a-1)*cw0+tmp;
      a1 =   2*((a-1)-(a+1)*cw0);
      a2 =      (a+1)-(a-1)*cw0-tmp;
   otherwise
      error(sprintf('Unknown filter type "%s"',type));
  end
  
  B = [b2 b1 b0];
  A = [a2 a1 a0];
  
