%EQ_DESIGN_ANALOG  Design continuous-time EQ w/ Audio Cookbook formulas
%
% Syntax:
%   [BC,AC] = eq_design_analog(type,f,q,g_db,use_ee_q)
%
% In:
%   type - Type of the filter as string 'lowpass', 'highpass',
%          'bandpass', 'notch', 'allpass', 'peak', 'lowshelf',
%          'highself'
%      f - Working frequency
%      q - The quality factor
%   g_db - Gain in decibels (for 'bandpass', 'peak', 'lowshelf'
%          and 'highshelf')
% use_ee_q - Use EE definition for q (1) or not (0). Default 0.
%
% Out:
%      B - 1xN numerator filter polynomial coefficients
%      A - 1xN denumerator filter polynomial coefficients
%   
% Description:
%   Design a analog equalizer filter of the given type
%   based on direct Audio QE Cookbook formulas in
%
%   http://www.musicdsp.org/files/Audio-EQ-Cookbook.txt
%
%   Uses non-EE definition for Q by default.

% References:
%   http://www.musicdsp.org/files/Audio-EQ-Cookbook.txt
%
% Copyright (C) Simo Srkk, 2009

function [BC,AC] = eq_design_analog(type,f,Q,g_db,use_ee_q)

    if nargin < 5
        use_ee_q = 0;
    end

    K  = 10^(g_db/40);
    w0 = 2*pi*f;
      
    if ~use_ee_q && strcmp(type, 'peak')
        Q = K*Q;
    end
  
    switch lower(type)
    case 'lowpass'      
        % H(s) = w0^2 / (s^2 + w0 s/Q + w0^2)
        BC = w0^2;
        AC = [1 w0/Q w0^2];
    case 'highpass'
        % H(s) = s^2 / (s^2 + w0 s/Q + w0^2)
        BC = [1 0 0];
        AC = [1 w0/Q w0^2];
    case 'bandpass'
        % H(s) = (K w0 s/Q) / (s^2 + w0 s/Q + w0^2)
        BC = [K*w0/Q 0];
        AC = [1 w0/Q w0^2];
    case 'notch'
        % H(s) = (s^2 + w0^2) / (s^2 + w0 s/Q + w0^2)
        BC = [1 0 w0^2];
        AC = [1 w0/Q w0^2];
    case 'allpass'
        % H(s) = (s^2 - w0 s/Q + w0^2) / (s^2 + w0 s/Q + w0^2)
        BC = [1 -w0/Q w0^2];
        AC = [1 w0/Q w0^2];
    case 'peak'
        % H(s) = (s^2 + (K^2 w0/Q)*s + w0^2) / (s^2 + w0 s/Q + w0^2)
        BC = [1 K^2*w0/Q w0^2];
        AC = [1 w0/Q w0^2];
    case 'lowshelf'
        % H(s) = (s^2 + (sqrt(K)/Q)*w0*s + K w0^2) / (s^2 + (w0/sqrt(K)/Q)*s + w0^2/K)
        BC = [1 sqrt(K)*w0/Q K*w0^2];
        AC = [1 w0/sqrt(K)/Q w0^2/K];
    case 'highshelf'
        % H(s) = (K^2*s^2 + (K*sqrt(K)/Q)*w0*s + K w0^2) / (s^2 + (w0 sqrt(K)/Q)*s + K w0^2)
        BC = [K^2 K*sqrt(K)*w0/Q K*w0^2];
        AC = [1 w0*sqrt(K)/Q K*w0^2];
    otherwise
        error(sprintf('Unknown filter type "%s"',type));
    end

