/*
 * cpp_sym.C
 * @(#) Symmetry Code Generation
 *
 * Copyright (C) 1992 - 1999 by the Board of Trustees of
 * Leland Stanford Junior University.
 *
 * License to use, copy, modify, sell and/or distribute this software
 * and its documentation any purpose is hereby granted without royalty,
 * subject to the following terms and conditions:
 *
 * 1.  The above copyright notice and this permission notice must
 * appear in all copies of the software and related documentation.
 *
 * 2.  The name of Stanford University may not be used in advertising or
 * publicity pertaining to distribution of the software without the
 * specific, prior written permission of Stanford.
 *
 * 3.  This software may not be called "Murphi" if it has been modified
 * in any way, without the specific prior written permission of David L.
 * Dill.
 *
 * 4.  THE SOFTWARE IS PROVIDED "AS-IS" AND STANFORD MAKES NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION.  STANFORD MAKES NO REPRESENTATIONS OR WARRANTIES
 * OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE
 * USE OF THE SOFTWARE WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS
 * TRADEMARKS OR OTHER RIGHTS. STANFORD SHALL NOT BE LIABLE FOR ANY
 * LIABILITY OR DAMAGES WITH RESPECT TO ANY CLAIM BY LICENSEE OR ANY
 * THIRD PARTY ON ACCOUNT OF, OR ARISING FROM THE LICENSE, OR ANY
 * SUBLICENSE OR USE OF THE SOFTWARE OR ANY SERVICE OR SUPPORT.
 *
 * LICENSEE shall indemnify, hold harmless and defend STANFORD and its
 * trustees, officers, employees, students and agents against any and all
 * claims arising out of the exercise of any rights under this Agreement,
 * including, without limiting the generality of the foregoing, against
 * any damages, losses or liabilities whatsoever with respect to death or
 * injury to person or damage to property arising from or out of the
 * possession, use, or operation of Software or Licensed Program(s) by
 * LICENSEE or its customers.
 *
 * Read the file "license" distributed with these sources, or call
 * Murphi without the -l switch for additional information.
 * 
 */

/* 
 * Original Author: C. Norris Ip
 * Symmetry extension to Rel2.6
 * Created: 18 Nov 1993
 * 
 * Major Update:
 *
 * Denis Leroy
 * Subject: update to deal with resident working state
 * First updated: May 13, 95
 *
 * For details in creating and updating the file,
 * please check at the end.
 */ 

#include "mu.h"

void symmetryclass::generate_symmetry_function_decl()
{
  fprintf(codefile,
	  "  virtual void Permute(PermSet& Perm, int i);\n"
	  "  virtual void SimpleCanonicalize(PermSet& Perm);\n"
	  "  virtual void Canonicalize(PermSet& Perm);\n"
	  "  virtual void SimpleLimit(PermSet& Perm);\n"
	  "  virtual void ArrayLimit(PermSet& Perm);\n"
	  "  virtual void Limit(PermSet& Perm);\n"
	  "  virtual void MultisetLimit(PermSet& Perm);\n"
	  );

  /*
   * TJ Normalization things
   */
  fprintf(codefile,
	  "  virtual unsigned int TJ_hash(const int st, const int sv);\n"
	  );

  fprintf(codefile,
	  "  virtual unsigned int TJ_dfs_invariant(unsigned int dfs_num);\n"
	  );
 
  /*
   * TJ Nauty canon things
   */
  fprintf(codefile,
	  "#if defined(TJ_NAUTY)\n"
	  "  int TJ_nauty_v;\n"
	  "  virtual int TJ_nauty_init_graph(int first_free_v);\n"
	  "  virtual int TJ_nauty_make_graph_full();\n"
	  "  virtual int TJ_nauty_make_graph();\n"
	  "  virtual int TJ_nauty_unmake_graph();\n"
	  "  virtual void TJ_nauty_init_weights(int *w_num);\n"
	  "#endif\n"
	  );
}

/********************
  main procedure to generate code for symmetry
  ********************/

void symmetryclass::generate_code(ste * globals)
{
  stelist * var;

  // output comment on overall scalarset types on all highest level variables
  // not necessary, just to help you analyse your variables 
  fprintf(codefile, "/*\n");
  set_var_list(globals);
  for ( var = varlist; var!= NULL; var=var->next )
      fprintf(codefile, "%s:%s\n",
              var->s->getname()->getname(),
              var->s->getvalue()->gettype()->structurestring()
              );
  fprintf(codefile, "*/\n");

  // classify scalarset types as useful or useless
  // it is not useful if 
  //    1) it was not given an explicit name
  //    2) it is used in one of the state variables
  for ( var = varlist; var!= NULL; var=var->next )
    var->s->getvalue()->gettype()->setusefulness();

  // remove useless scalarset from scalarset list stored in symmetryclass
  stelist ** entry = &scalarsetlist;
  stelist * member;
  for (member = scalarsetlist;
       member != NULL;  
       member = member->next)
    if (!((scalarsettypedecl *) member->s->getvalue())->isuseless())
      {
	*entry = member;
	entry = &(member->next);
      }
  *entry = NULL;

  // generate symmetry code
  generate_symmetry_code();
}

/********************
  generate canonicalization algorithm
  ********************/

void symmetryclass::generate_symmetry_code()
{
  fprintf(codefile,
          "\n/********************\n"
          "Code for symmetry\n"
          " ********************/\n");

  setup();
  generate_permutation_class();
  generate_symmetry_class();
  
  generate_TJ_normalization();
  generate_TJ_nauty();

  fprintf(codefile,
	  "\n/********************\n"
	  " Permute and Canonicalize function for different types\n"
	  " ********************/\n");
  if (typedecl::origin != NULL) 
    typedecl::origin->generate_all_symmetry_decl(*this);

  generate_match_function();

  generate_exhaustive_fast_canonicalization();
  generate_heuristic_fast_canonicalization();
  generate_heuristic_small_mem_canonicalization();
  generate_heuristic_fast_normalization();
}

/********************
  generate symmetry class
  ********************/
void symmetryclass::generate_symmetry_class()
{
  fprintf(codefile,
	  "\n/********************\n"
	  " Symmetry Class\n"
	  " ********************/\n");
  fprintf(codefile,
	  "class SymmetryClass\n"

	  "{\n"
	  "  PermSet Perm;\n"
	  "  bool BestInitialized;\n"
	  "  state BestPermutedState;\n"
	  "\n"
	  "  // utilities\n"
	  "  void SetBestResult(int i, state* temp);\n"
	  "  void ResetBestResult() {BestInitialized = FALSE;};\n"
	  "\n"
	  "public:\n"
	  "  // initializer\n"
	  "  SymmetryClass() : Perm(), BestInitialized(FALSE) {};\n"
	  "  ~SymmetryClass() {};\n"
	  "\n"
	  "  void Normalize(state* s);\n"
	  "\n"  
	  "  void Exhaustive_Fast_Canonicalize(state *s);\n"
	  "  void Heuristic_Fast_Canonicalize(state *s);\n"
	  "  void Heuristic_Small_Mem_Canonicalize(state *s);\n"
	  "  void Heuristic_Fast_Normalize(state *s);\n"
	  "  void TJ_IPG(state *s);\n"
	  "  void TJ_Normalize1(state *s);\n"
	  "  void TJ_Normalize2(state *s);\n"
	  "#if defined(TJ_NAUTY)\n"
	  "  void TJ_Nauty_Canonicalize(state *s);\n"
	  "#endif\n"
	  "\n"
	  "  void MultisetSort(state* s);\n"
	  "};\n"
	  "\n"
	  );

  fprintf(codefile,
	  "\n/********************\n"
	  " Symmetry Class Members\n"
	  " ********************/\n");
  fprintf(codefile,
	  "void SymmetryClass::MultisetSort(state* s)\n"
	  "{\n"
	  );
  for ( stelist * var = varlist; var != NULL; var = var->next )
    fprintf( codefile, 
	     "        %s.MultisetSort();\n",
             var->s->getvalue()->mu_name
             );
  fprintf(codefile,
	  "}\n"
	  );

  fprintf(codefile,
	  "void SymmetryClass::Normalize(state* s)\n"
	  "{\n"
	  "  switch (args->sym_alg.mode) {\n"
	  "  case argsym_alg::Exhaustive_Fast_Canonicalize:\n"
	  "    Exhaustive_Fast_Canonicalize(s);\n"
	  "    break;\n"
	  "  case argsym_alg::Heuristic_Fast_Canonicalize:\n"
	  "    Heuristic_Fast_Canonicalize(s);\n"
	  "    break;\n"
	  "  case argsym_alg::Heuristic_Small_Mem_Canonicalize:\n"
	  "    Heuristic_Small_Mem_Canonicalize(s);\n"
	  "    break;\n"
	  "  case argsym_alg::Heuristic_Fast_Normalize:\n"
	  "    Heuristic_Fast_Normalize(s);\n"
	  "    break;\n"
	  "  case argsym_alg::TJ_Normalize1:\n"
	  "    TJ_Normalize1(s);\n"
	  "    break;\n"
	  "  case argsym_alg::TJ_Normalize2:\n"
	  "    TJ_Normalize2(s);\n"
	  "    break;\n"
	  "#if defined(TJ_NAUTY)\n"
	  "  case argsym_alg::TJ_Nauty_Canonicalize:\n"
	  "    TJ_Nauty_Canonicalize(s);\n"
	  "    break;\n"
	  "#endif\n"
	  "  default:\n"
	  "    Heuristic_Fast_Canonicalize(s);\n"
	  "  }\n"
	  "}\n"
	  );
}

/********************
  generate old algorithm I :
  simple exhaustive canonicalization
  ********************/

void symmetryclass::generate_match_function()   // changes by Uli
{
  stelist * var;

  fprintf(codefile,
          "\n/********************\n"
          " Auxiliary function for error trace printing\n"
          " ********************/\n");

  fprintf( codefile, 
           "bool match(state* ns, StatePtr p)\n"
           "{\n"
           "  int i;\n"
	   "  static PermSet Perm;\n"
	   "  static state temp;\n"
	   "  StateCopy(&temp, ns);\n"
	   );

  fprintf( codefile, 
	   "  if (args->symmetry_reduction.value)\n"
	   "    {\n"
	   );

  // ********************
  // matching by permuting
  // ********************
  if (no_need_for_perm)
    fprintf( codefile, 
	     "      if (  args->sym_alg.mode == argsym_alg::Exhaustive_Fast_Canonicalize) {\n"
	     );
  else 
    fprintf( codefile, 
	     "      if (  args->sym_alg.mode == argsym_alg::Exhaustive_Fast_Canonicalize\n"
	     "         || args->sym_alg.mode == argsym_alg::Heuristic_Fast_Canonicalize) {\n"
	     );

  // matching by explicit data  
  fprintf( codefile, 
	   "        Perm.ResetToExplicit();\n"
           "        for (i=0; i<Perm.count; i++)\n"
           "          if (Perm.In(i))\n"
           "            {\n"
	   "              if (ns != workingstate)\n"
	   "                  StateCopy(workingstate, ns);\n"
           "              \n"
           );
  for ( var = varlist; var != NULL; var = var->next )
    fprintf( codefile, 
	     "              %s.Permute(Perm,i);\n"
	     "              if (args->multiset_reduction.value)\n"
	     "                %s.MultisetSort();\n",
             var->s->getvalue()->mu_name,
             var->s->getvalue()->mu_name
             );
  fprintf( codefile,
	   "            if (p.compare(workingstate)) {\n"
                          // changed by Uli
	   "              StateCopy(workingstate,&temp); return TRUE; }\n"
           "          }\n"
	   "        StateCopy(workingstate,&temp);\n"
           "        return FALSE;\n"
	   "      }\n"
	   );

  // matching by generating the permutation one by one
  fprintf( codefile, 
	   "      else {\n"
	   "        Perm.ResetToSimple();\n"
	   "        Perm.SimpleToOne();\n"
	   );

  // first iteration -- may be skipped? 
  fprintf( codefile, 
	   "        if (ns != workingstate)\n"
	   "          StateCopy(workingstate, ns);\n"
           "\n"
           );
  for ( var = varlist; var != NULL; var = var->next )
    fprintf( codefile, 
	     "          %s.Permute(Perm,0);\n"
	     "          if (args->multiset_reduction.value)\n"
	     "            %s.MultisetSort();\n",
             var->s->getvalue()->mu_name,
             var->s->getvalue()->mu_name
             );
  fprintf( codefile, 
	   "        if (p.compare(workingstate)) {\n"   // changed by Uli
	   "          StateCopy(workingstate,&temp); return TRUE; }\n"
	   "\n"
	   );

  // all other iterations
  fprintf( codefile, 
	   "        while (Perm.NextPermutation())\n"
           "          {\n"
	   "            if (ns != workingstate)\n"
	   "              StateCopy(workingstate, ns);\n"
           "              \n"
           );
  for ( var = varlist; var != NULL; var = var->next )
    fprintf( codefile, 
	     "              %s.Permute(Perm,0);\n"
	     "              if (args->multiset_reduction.value)\n"
	     "                %s.MultisetSort();\n",
             var->s->getvalue()->mu_name,
             var->s->getvalue()->mu_name
             );
  fprintf( codefile, 
	   "            if (p.compare(workingstate)) {\n"   // changed by Uli
	   "              StateCopy(workingstate,&temp); return TRUE; }\n"
           "          }\n"
	   "        StateCopy(workingstate,&temp);\n"
           "        return FALSE;\n"
	   "      }\n"
	   );
	   
  // end matching by permuting
  fprintf( codefile, 
	   "    }\n"
	   );

  // matching by multisesort
  fprintf( codefile, 
	   "  if (!args->symmetry_reduction.value\n"
	   "      && args->multiset_reduction.value)\n"
	   "    {\n"
	   "      if (ns != workingstate)\n"
	   "          StateCopy(workingstate, ns);\n"
	   );
  for ( var = varlist; var != NULL; var = var->next )
    fprintf( codefile, 
	     "      %s.MultisetSort();\n",
             var->s->getvalue()->mu_name
             );
  fprintf( codefile, 
	   "      if (p.compare(workingstate)) {\n"   // changed by Uli
	   "        StateCopy(workingstate,&temp); return TRUE; }\n"
	   "      StateCopy(workingstate,&temp);\n"
           "      return FALSE;\n"
	   "    }\n"
	   );

  // matching by direct comparsion
  fprintf( codefile, 
	   "  return (p.compare(ns));\n"   // changed by Uli
           "}\n"
           );
}

/********************
  generate old algorithm :
  fast exhaustive canonicalization
  ********************/

void symmetryclass::generate_exhaustive_fast_canonicalization()
{
  fprintf(codefile,
          "\n/********************\n"
          " Canonicalization by fast exhaustive generation of\n"
          " all permutations\n"
          " ********************/\n");

  fprintf(codefile, 
	  "void SymmetryClass::Exhaustive_Fast_Canonicalize(state* s)\n"
	  "{\n"
	  "  int i;\n"
	  "  static state temp;\n"
	  "  Perm.ResetToExplicit();\n"
          "\n"
	  );

  for ( stelist * var = varlist; var != NULL; var = var->next )
    fprintf(codefile, 
	    "  StateCopy(&temp, workingstate);\n"
            "  ResetBestResult();\n"
            "  for (i=0; i<Perm.count; i++)\n"
            "    if (Perm.In(i))\n"
            "      {\n"
            "        StateCopy(workingstate, &temp);\n"
            "        %s.Permute(Perm,i);\n"
	    "        if (args->multiset_reduction.value)\n"
	    "          %s.MultisetSort();\n" 
            "        SetBestResult(i, workingstate);\n"
            "      }\n"
            "  StateCopy(workingstate, &BestPermutedState);\n"
            "\n",
            var->s->getvalue()->mu_name,
            var->s->getvalue()->mu_name
            );
  fprintf(codefile,
	  "};\n"); 
}

/********************
  generate normalization algorithm
  ********************/
void symmetryclass::generate_heuristic_fast_normalization()
{
  stelist * var;
  bool ToOne;
  
  fprintf(codefile,
          "\n/********************\n"
          " Normalization by fast simple variable canonicalization,\n"
	  " fast simple scalarset array canonicalization,\n"
	  " fast restriction on permutation set with simple scalarset array of scalarset,\n"
	  " and for all other variables, pick any remaining permutation\n" 
          " ********************/\n");
  
  fprintf(codefile, 
	  "void SymmetryClass::Heuristic_Fast_Normalize(state* s)\n"
	  "{\n"
	  "  int i;\n"
	  "  static state temp;\n"
	  "\n"
          "  Perm.ResetToSimple();\n"
          "\n"
	  );
  
  // simple variable
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetVariable()
	&& (var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetArrayOfFree
	    || var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetVariable
 	    || var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree) )
      fprintf(codefile, 
	      "  %s.SimpleCanonicalize(Perm);\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );
  
  
  // simple scalarset array
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetArrayOfFree()
        && ( var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetArrayOfFree
	     || var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree) )
      fprintf(codefile, 
	      "  %s.Canonicalize(Perm);\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // multiset
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  %s.MultisetSort();\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // if they are inside more complicated structure, only do a limit
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetVariable()
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.SimpleLimit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetArrayOfFree()
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.ArrayLimit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // refine Permutation set by checking graph structure of scalarset array of Scalarset
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetArrayOfS())
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.Limit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // checking if we need to change simple to one
  ToOne = FALSE;
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() != typedecl::NoScalarset
        && var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      ToOne = TRUE;
  if (ToOne)
    fprintf(codefile, 
	    "  Perm.SimpleToOne();\n"
	    "\n"
	    );

  // others
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() != typedecl::NoScalarset
        && var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  %s.Permute(Perm,0);\n"
	      "  if (args->multiset_reduction.value)\n"
	      "    %s.MultisetSort();\n" 
	      "\n",
	      var->s->getvalue()->mu_name,
	      var->s->getvalue()->mu_name,
	      var->s->getvalue()->mu_name,
	      var->s->getvalue()->mu_name
	      );

  fprintf(codefile,
	  "};\n"); 
}



/********************
  generate TJ normalization algorithm
  ********************/
void symmetryclass::generate_TJ_normalization()
{
  stelist *ssiter, *ssiter2, *variter;
  scalarsettypedecl *ss, *ss2;
  decl *var;



  /*
   * Generate a partition class for each scalar set
   */
  fprintf(codefile,
	  "class TJ_partition {\n"
	  "public:\n"
	  "  const int size;\n"
	  "  class Cell {\n"
	  "  public:\n"
	  "    int first, length;\n"
	  "    unsigned int magicnum;\n"
	  "    Cell *next;\n"
	  "  };\n"
	  "  int *elements;\n"
	  "  unsigned int *invariant_values;\n"
	  "  Cell *cells;\n"
	  "  Cell **in_cell;\n"
	  "  int *in_pos;\n"
	  "  Cell *first_cell, *free_cells;\n"
	  "  TJ_partition(int s);\n"
	  "  void reset();\n"
	  "  void clear_invariants();\n"
	  "  bool is_discrete();\n"
	  "  bool refine(unsigned int (*invariant)(int));\n"
	  "  bool split_cell(Cell *cell);\n"
	  "  void print(char* names[]);\n"
	  "};\n"
	  "TJ_partition::TJ_partition(int s) : size(s) {\n"
	  "  int i;\n"
	  "  unsigned int magicnum = 324;\n"
	  "  if(size <= 0)\n"
	  "    Error.Error(\"Illegal partition size\");\n"
	  "  elements = (int*)malloc(size*sizeof(int));\n"
	  "  invariant_values = (unsigned int*)malloc(size*sizeof(unsigned int));\n"
	  "  cells = (Cell*)malloc(size*sizeof(Cell));\n"
	  "  in_cell = (Cell**)malloc(size*sizeof(Cell*));\n"
	  "  in_pos = (int*)malloc(size*sizeof(int));\n"
	  "  for(i = 0; i < size; i++) {\n"
	  "    cells[i].magicnum = magicnum;\n"
	  "    magicnum = (magicnum * 0x03d2821 + 1) %% 0x07b3c9e3;\n"
	  "  }\n"
	  "  reset();\n"
	  "}\n"
	  "void TJ_partition::reset() {\n"
	  "  int i;\n"
	  "  cells[0].first = 0;\n"
	  "  cells[0].length = size;\n"
	  "  cells[0].next = 0;\n"
	  "  first_cell = &cells[0];\n"
	  "  for(i = 1; i < size; i++) {\n"
	  "    cells[i].first = 0;\n"
	  "    cells[i].length = 0;\n"
	  "    cells[i].next = &cells[i+1];\n"
	  "  }\n"
	  "  if(size > 1) {\n"
	  "    cells[size-1].next = 0;\n"
	  "    free_cells = &cells[1];\n"
	  "  } else {\n"
	  "    free_cells = 0;\n"
	  "  }\n"
	  "  for(i = 0; i < size; i++) {\n"
	  "    elements[i] = i;\n"
	  "    invariant_values[i] = 0;\n"
	  "    in_cell[i] = first_cell;\n"
	  "    in_pos[i] = i;\n"
	  "  }\n"
	  "}\n"
	  "\n"
	  "void TJ_partition::clear_invariants() {\n"
	  "  int i;\n"
	  "  for(i = 0; i < size; i++)\n"
	  "    invariant_values[i] = 0;\n"
	  "}\n"
	  "\n"
	  "bool TJ_partition::is_discrete() {\n"
	  "  Cell *cell = first_cell;\n"
	  "  while(cell) {\n"
	  "    if(cell->length > 1)\n"
	  "      return false;\n"
	  "    cell = cell->next;\n"
	  "  }\n"
	  "  return true;\n"
	  "}\n"
	  "\n"
	  "bool TJ_partition::refine(unsigned int (*invariant)(int)) {\n"
	  "  Cell *cell = first_cell;\n"
	  "  bool refined = false;\n"
	  "  int i;\n"
	  "  while(cell) {\n"
	  "    if(cell->length > 1) {\n"
	  "      /* compute invariants */\n"
	  "      for(i = cell->first; i < cell->first + cell->length; i++) {\n"
	  "        assert(in_pos[elements[i]] == i);\n"
	  "        invariant_values[i] = invariant(elements[i]);\n"
	  "      }\n"
	  "      refined |= split_cell(cell);\n"
	  "    }\n"
	  "    cell = cell->next;\n"
	  "  }\n"
	  "  return refined;\n"
	    "}\n"
	  "\n"
	  "bool TJ_partition::split_cell(Cell *cell) {\n"
	  "  int i,j;\n"
	  "  bool splitted = false;\n"
	  "  /* sort the elements in the cell */\n"
	  "  for(i = cell->first; i < cell->first + cell->length - 1; i++) {\n"
	  "    for(j = i + 1; j < cell->first + cell->length; j++) {\n"
	  "      if(invariant_values[j] < invariant_values[i]) {\n"
	  "        unsigned int temp1 = invariant_values[i];\n"
	  "        invariant_values[i] = invariant_values[j];\n"
	  "        invariant_values[j] = temp1;\n"
	  "        int temp2 = elements[i];\n"
	  "        elements[i] = elements[j];\n"
	  "        elements[j] = temp2;\n"
	  "        in_pos[elements[i]] = i;\n"
	  "        in_pos[elements[j]] = j;\n"
	  "      }\n"
	  "    }\n"
	  "  }\n"
	  " splitem:\n"
	  "  for(i = cell->first+1; i < cell->first+cell->length; i++) {\n"
	  "    if(invariant_values[i] != invariant_values[cell->first]) {\n"
	  "      Cell *new_cell = free_cells;\n"
	  "      if(!new_cell) Error.Error(\"Invalid partition splitting...\");\n"
	  "      free_cells = new_cell->next;\n"
	  "      new_cell->first = i;\n"
	  "      new_cell->length =  cell->length - (i - cell->first);\n"
	  "      new_cell->next = cell->next;\n"
	  "      cell->next = new_cell;\n"
	  "      cell->length = i - cell->first;\n"
	  "      cell = new_cell;\n"
	  "      splitted = true;\n"
	  "      for(j = cell->first; j < cell->first + cell->length; j++)\n"
	  "	   in_cell[elements[j]] = cell;\n"
	  "      goto splitem;\n"
	  "    }\n"
	  "  }\n"
	  "  return splitted;\n"
	  "}\n"
	  "\n"
	  "void TJ_partition::print(char *names[]) {\n"
	  "  int i;\n"
	  "  Cell *cell;\n"
	  "  fprintf(stderr, \"[\");\n"
	  "  for(cell = first_cell; cell; cell = cell->next) {\n"
	  "    fprintf(stderr, \"{\");\n"
	  "    for(i = 0; i < cell->length; i++) {\n"
	  "      fprintf(stderr, \"%%s\", names[elements[cell->first+i]]);\n"
	  "      if(i < cell->length-1) fprintf(stderr, \",\");\n"
	  "    }\n"
	  "    fprintf(stderr, \"}\");\n"
	  "    if(cell->next) fprintf(stderr, \",\");\n"
	  "  }\n"
	  "  fprintf(stderr, \"]\");\n"
	  "}\n"
	  "\n"
	  );


  /*
   * Make partitions for each scalar set
   */
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    fprintf(codefile,
	    "static TJ_partition TJ_p_%s(%d);\n",
	    ss->mu_name,
	    ss->getsize()
	    );
  }
  fprintf(codefile, "\n");

  /*
   * Find variables that are of ordered type
   */
  for (variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    /* //for debugging purposes
    if(var->gettype()->TJ_is_ordered())
      fprintf(stderr, "%s is ordered\n", var->mu_name);
    else
      fprintf(stderr, "%s is unordered\n", var->mu_name);
    */
  }

  /*
   * Make hash invariant stubs for each (variable,scalar set) pair
   */
  for (variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
      ss = (scalarsettypedecl *)ssiter->s->getvalue();
      fprintf(codefile,
	      "static unsigned int TJ_hash_%s___%s(const int i) {\n"
	      "  return %s.TJ_hash(%d,i+%d);\n"
	      "}\n",
	      var->mu_name,
	      ss->mu_name,
	      var->mu_name,
	      ss->tNum,
	      ss->getleft()
	      );
    }
  }


  /*******************************************************************
   * Print code for invariant partition refiner
   ******************************************************************/
  fprintf(codefile, 
          "\n"
	  "/*\n"
          " * Invariant partition refiner\n"
          " */\n"
	  "void SymmetryClass::TJ_IPG(state* s)\n"
	  "{\n"
	  "  int i;\n"
	  "  bool changed;\n"
          "\n"
	  );
  /*
   * Reset scalar set partitions
   */
  fprintf(codefile, "\n");
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    fprintf(codefile,
	    "  TJ_p_%s.reset();\n",
	    ss->mu_name
	    );
  }
  fprintf(codefile, "\n");
#define TJ_ORDERED
  /*
   * Refine partitions by exploiting ordered types and dfs invariants
   */
#ifdef TJ_ORDERED
  fprintf(codefile,
	  "  /*\n"
	  "   * Refine partitions, part 1 - vars of ordered types\n"
	  "   */\n"
	  );
  for (variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    if(!var->gettype()->TJ_is_ordered())
      continue;
    /* reset invariant values */
    for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
      ss = (scalarsettypedecl *)ssiter->s->getvalue();
      fprintf(codefile,
	      "  TJ_p_%s.clear_invariants();\n",
	      ss->mu_name
	      );
    }
    fprintf(codefile,
	    "  %s.TJ_dfs_invariant(1);\n",
	    var->mu_name
	    );
    for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
      ss = (scalarsettypedecl *)ssiter->s->getvalue();
      fprintf(codefile,
	      "  {\n"
	      "    TJ_partition::Cell *cell = TJ_p_%s.first_cell;\n"
	      "    while(cell) {\n"
	      "      TJ_partition::Cell *next_cell = cell->next;\n"
	      "      if(cell->length > 1)\n"
	      "        TJ_p_%s.split_cell(cell);\n"
	      "      cell = next_cell;\n"
	      "    }\n"
	      "  }\n",
	      ss->mu_name,
	      ss->mu_name
	      );
    }
  }
  fprintf(codefile, "\n");
#endif
  /*
   * Refine partitions by applying the hash invariant
   */
  fprintf(codefile,
	  "  /*\n"
	  "   * Refine partitions\n"
	  "   */\n"
	  "  changed = true;\n"
	  "  while(changed) {\n"
	  "    changed = false;\n"
	  );
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    for (variter = varlist; variter; variter = variter->next ) {
      var = variter->s->getvalue();
#ifdef TJ_ORDERED
      if(var->gettype()->TJ_is_ordered())
	continue;
#endif
      fprintf(codefile,
	      "    if(!TJ_p_%s.is_discrete())\n"
	      "      changed |= TJ_p_%s.refine(&TJ_hash_%s___%s);\n",
	      ss->mu_name,
	      ss->mu_name,
	      var->mu_name,
	      ss->mu_name
	      );
    }
  }
  fprintf(codefile,
	  "  }\n"
	  );
  fprintf(codefile,
	  "}\n\n");




  /*******************************************************************
   * Print code for algorithm 1
   ******************************************************************/
  fprintf(codefile, 
          "\n"
	  "/*\n"
          " * Normalization by partitioning,\n"
	  " * pick any compatible permutation\n" 
          " */\n"
	  "void SymmetryClass::TJ_Normalize1(state* s)\n"
	  "{\n"
	  "  int i;\n"
	  "  bool changed;\n"
	  "  static state temp;\n"
	  "\n"
          "  Perm.ResetToSimple();\n"
          "\n"
	  //	  "  fprintf(stderr, \"Original state:\\n\");\n"
	  //	  "  workingstate->print();\n"
	  );
  /*
   * Compute partition
   */
  fprintf(codefile,
	  "  /* Compute partition */\n"
	  "  TJ_IPG(s);\n"
	  "\n"
	  );

  fprintf(codefile,
	  "  /*\n"
	  "   * Make permutation\n"
	  "   */\n"
	  "  int c;\n"
	  "  TJ_partition::Cell *cell;\n"
	  "  c = 0;\n"
	  );
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    fprintf(codefile,
	    "  for(cell = TJ_p_%s.first_cell; cell; cell = cell->next) {\n"
	    "    for(i = cell->first; i < cell->first+cell->length; i++)\n"
	    "      Perm.class_%s[TJ_p_%s.elements[i]] = c;\n"
	    "    c++;\n"
	    "  }\n"
	    "  Perm.undefined_class_%s = c;\n",
	    ss->mu_name,
	    ss->mu_name,
	    ss->mu_name,
	    ss->mu_name
	    );
  }
  fprintf(codefile,
	  "  Perm.SimpleToOne();\n"
	  );
  fprintf(codefile,
	  "  /*\n"
	  "   * Apply permutation\n"
	  "   */\n"
	  );
  for (variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  %s.Permute(Perm, 0);\n",
	    var->mu_name
	    );
  }
  /*
  fprintf(codefile,
      	  "  fprintf(stderr, \"Normalized state:\\n\");\n"
	  "  workingstate->print();\n"
	  );
  */

  fprintf(codefile,
	  "};\n\n"); 




  /*******************************************************************
   * Print code for algorithm 2
   ******************************************************************/
  fprintf(codefile, 
          "\n"
	  "/*\n"
          " * Normalization by partitioning and search trees\n"
          " */\n"
	  "void SymmetryClass::TJ_Normalize2(state* s)\n"
	  "{\n"
	  "  int i;\n"
	  "  static state temp;\n"
	  "\n"
          "  Perm.ResetToSimple();\n"
          "\n"
	  //	  "  fprintf(stderr, \"Original state:\\n\");\n"
	  //	  "  workingstate->print();\n"
	  );
  /*
   * Compute partition
   */
  fprintf(codefile,
	  "  /* Compute partition */\n"
	  "  TJ_IPG(s);\n"
	  "\n"
	  );
  /*
   * Split and refine partitions by applying the hash invariant
   */
  fprintf(codefile,
	  "  /*\n"
	  "   * Split and refine partitions\n"
	  "   */\n"
	  );
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    fprintf(codefile,
	    "  while(!TJ_p_%s.is_discrete()) {\n",
	    ss->mu_name
	    );
    fprintf(codefile,
	    "    /* Find first non-discrete cell */\n"
	    "    TJ_partition::Cell *cell = TJ_p_%s.first_cell;\n"
	    "    while(cell && cell->length == 1)\n"
	    "      cell = cell->next;\n"
	    "    assert(cell);\n"
	    "    assert(cell->length > 1);\n"
	    "    TJ_partition::Cell *new_cell = TJ_p_%s.free_cells;\n"
	    "    assert(new_cell);\n"
	    "    TJ_p_%s.free_cells = new_cell->next;\n"
	    "    new_cell->first = cell->first+1;\n"
	    "    new_cell->length = cell->length - 1;\n"
	    "    new_cell->next = cell->next;\n"
	    "    cell->next = new_cell;\n"
	    "    cell->length = 1;\n"
	    "    for(int j = new_cell->first; j < new_cell->first + new_cell->length; j++)\n"
	    "      TJ_p_%s.in_cell[TJ_p_%s.elements[j]] = new_cell;\n",
	    ss->mu_name,
	    ss->mu_name,
	    ss->mu_name,
	    ss->mu_name,
	    ss->mu_name);
    fprintf(codefile,
	    "    /* Refine partitions until nothing changes */\n"
	    "    bool changed = true;\n"
	    "    while(changed) {\n"
	    "      changed = false;\n"
	    );
    for(ssiter2 = scalarsetlist; ssiter2; ssiter2 = ssiter2->next) {
      ss2 = (scalarsettypedecl *)ssiter2->s->getvalue();
      for(variter = varlist; variter; variter = variter->next ) {
	var = variter->s->getvalue();
#ifdef TJ_ORDERED
	if(var->gettype()->TJ_is_ordered())
	  continue;
#endif
	fprintf(codefile,
		"      if(!TJ_p_%s.is_discrete())\n"
		"        changed |= TJ_p_%s.refine(&TJ_hash_%s___%s);\n",
		ss2->mu_name,
		ss2->mu_name,
		var->mu_name,
		ss2->mu_name
		);
      }
    }
    fprintf(codefile,
	    "    }\n"
	    );
    fprintf(codefile,
	    "  }\n"
	    );
  }

  fprintf(codefile,
	  "  /*\n"
	  "   * Make permutation\n"
	  "   */\n"
	  "  int c;\n"
	  "  TJ_partition::Cell *cell;\n"
	  "  c = 0;\n"
	  );
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    fprintf(codefile,
	    "  for(cell = TJ_p_%s.first_cell; cell; cell = cell->next) {\n"
	    "    for(i = cell->first; i < cell->first+cell->length; i++)\n"
	    "      Perm.class_%s[TJ_p_%s.elements[i]] = c;\n"
	    "    c++;\n"
	    "  }\n"
	    "  Perm.undefined_class_%s = c;\n",
	    ss->mu_name,
	    ss->mu_name,
	    ss->mu_name,
	    ss->mu_name
	    );
  }
  fprintf(codefile,
	  "  Perm.SimpleToOne();\n"
	  );
  for (variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  %s.Permute(Perm, 0);\n",
	    var->mu_name
	    );
  }
  fprintf(codefile,
	  "}\n\n");
}



void symmetryclass::generate_TJ_nauty()
{
  decl *var;
  typedecl *t;
  stelist *variter;
  charlist *typeiter;
  charlist *simpletype_list = NULL;
  stelist *ssiter;
  scalarsettypedecl *ss;

  fprintf(codefile, "#if defined(TJ_NAUTY)\n");

  /*
   * Get primitive types that appear in the variables of the system
   */
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    simpletype_list =
      var->gettype()->generate_simpletype_list(simpletype_list);
  }


  /*
   * Pring global Nauty variables
   */
  fprintf(codefile,
	  "DYNALLSTAT(nvector,TJ_lab,TJ_lab_sz);\n"
	  "DYNALLSTAT(nvector,TJ_ptn,TJ_ptn_sz);\n"
	  "DYNALLSTAT(graph,TJ_cg,TJ_cg_sz);\n"
	  "DYNALLSTAT(graph,TJ_cg_canon,TJ_cg_canon_sz);\n"
	  "DYNALLSTAT(nvector,TJ_orbits,TJ_orbits_sz);\n"
	  "DYNALLSTAT(setword,TJ_workspace,TJ_workspace_sz);\n"
	  "int TJ_cg_size = 0;\n"
	  "int TJ_cg_words = 0;\n"
	  );
  fprintf(codefile, "\n");

  /* print a debug routine */
  fprintf(codefile,
	  "void print_cg() {\n"
	  "  FILE *fp = fopen(\"debug_cg.dot\", \"w\");\n"
	  "  fprintf(fp, \"digraph muu {\\n\");\n"
	  "  fprintf(fp, \"\\\"v0\\\" [label=\\\"null\\\"];\\n\");\n"
	  "  set *gv;\n"
	  "  for(int i = 0; i < TJ_cg_size; i++) {\n"
	  "    gv = GRAPHROW(TJ_cg,i,TJ_cg_words);\n"
	  "    for(int j = 0; j < TJ_cg_size; j++) {\n"
	  "      if(ISELEMENT(gv,j))\n"
	  "	fprintf(fp, \"\\\"v%%d\\\" -> \\\"v%%d\\\";\\n\", i,j);\n"
	  "    }\n"
	  "  }\n"
	  "  fprintf(fp, \"}\\n\");\n"
	  "  fclose(fp);\n"
	  "}\n");


  /*
   * Print nauty init
   */
  fprintf(codefile,
	  "void TJ_Nauty_init() {\n"
	  "  int i;\n"
	  );
  fprintf(codefile,
	  "  TJ_cg_size = 1;\n" /* vertex 0 is for null */
	  );
  int nof_vars = 0;
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    nof_vars++;
  }
  fprintf(codefile,
	  "  TJ_cg_size += %d;\n", /* one vertex for each var */
	  nof_vars);
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  TJ_cg_size = %s.TJ_nauty_init_graph(TJ_cg_size);\n",
	    var->mu_name);
  }
  //fprintf(codefile,
  //	  "  fprintf(stderr, \"CG size = %%d\\n\", TJ_cg_size);\n");
  fprintf(codefile,
	  "  TJ_cg_words = ((TJ_cg_size + WORDSIZE - 1) / WORDSIZE);\n");
  //fprintf(codefile,
  //	  "  fprintf(stderr, \"CG words = %%d\\n\", TJ_cg_words);\n");

  fprintf(codefile,
	  "  DYNALLOC1(nvector,TJ_lab,TJ_lab_sz,TJ_cg_size,\"kuukkeli\");\n"
	  "  DYNALLOC1(nvector,TJ_ptn,TJ_ptn_sz,TJ_cg_size,\"kuukkeli\");\n"
	  "  DYNALLOC1(nvector,TJ_orbits,TJ_orbits_sz,TJ_cg_size,\"kuukkeli\");\n"
	  "  DYNALLOC1(setword,TJ_workspace,TJ_workspace_sz,50*TJ_cg_words,\"kuukkeli\");\n"
	  "  DYNALLOC2(graph,TJ_cg,TJ_cg_sz,TJ_cg_size,TJ_cg_words,\"kuukkeli\");\n"
	  "  DYNALLOC2(graph,TJ_cg_canon,TJ_cg_canon_sz,TJ_cg_size,TJ_cg_words,\"kuukkeli\");\n"
 	  );
  fprintf(codefile,
	  "  for(i = 0; i < TJ_cg_size; i++)\n"
	  "    TJ_lab[i] = i;\n"
	  );
  /* make vertex weights */
  fprintf(codefile,
	  "  TJ_ptn[0] = 0;\n"
	  "  int w_num = 1;\n"
	  );
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  %s.TJ_nauty_init_weights(&w_num);\n",
	    var->mu_name);
  }
  int var_num = 1;
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  TJ_ptn[%d] = w_num++;\n",
	    var_num++);
  }
  for(typeiter = simpletype_list; typeiter; typeiter = typeiter->next ) {
    t = (typedecl*)typeiter->e;
    //fprintf(stderr, "%s:%d ", t->name, t->getsize());
    if(t->HasScalarsetLeaf()) {
      fprintf(codefile,
	      "  extern int TJ_nauty_first_v_%s;\n"
	      "  for(i = TJ_nauty_first_v_%s; i<TJ_nauty_first_v_%s+%d; i++)\n"
	      "    TJ_ptn[i] = w_num;\n"
	      "  w_num++;\n",
	      t->mu_name,
	      t->mu_name,
	      t->mu_name,
	      t->getsize()
	      );
    } else {
      fprintf(codefile,
	      "  extern int TJ_nauty_first_v_%s;\n"
	      "  for(i = TJ_nauty_first_v_%s; i<TJ_nauty_first_v_%s+%d; i++)\n"
	      "    TJ_ptn[i] = w_num++;\n",
	      t->mu_name,
	      t->mu_name,
	      t->mu_name,
	      t->getsize()
	      );
    }
  }
  //  fprintf(codefile,
  //	  "  fprintf(stderr, \"vertex colors = %%d\\n\", w_num);\n");
  fprintf(codefile,
	  "  for(i = 0; i < TJ_cg_size; i++){\n"
	  "    int smallest = INT_MAX;\n"
	  "    int smallest_index = i;\n"
	  "    for(int j = i; j < TJ_cg_size; j++){\n"
	  "      if(TJ_ptn[j] < smallest) {\n"
	  "        smallest = TJ_ptn[j]; smallest_index = j;}\n"
	  "    }\n"
	  "    assert(smallest < INT_MAX);\n"
	  "    TJ_ptn[smallest_index] = TJ_ptn[i];\n"
	  "    TJ_ptn[i] = smallest;\n"
	  "    int temp = TJ_lab[i];\n"
	  "    TJ_lab[i] = TJ_lab[smallest_index];\n"
	  "    TJ_lab[smallest_index] = temp;\n"
	  "  }\n"
	  );
  fprintf(codefile,
	  "  i = 0;\n"
	  "  int w = TJ_ptn[i];\n"
	  "  while(i < TJ_cg_size-1) {\n"
	  "    if(TJ_ptn[i+1] != w) {\n"
	  "      w = TJ_ptn[i+1];\n"
	  "      TJ_ptn[i] = 0;\n"
	  "    } else\n"
	  "      TJ_ptn[i] = INFINITY;\n"
	  "    i++;\n"
	  "  }\n"
	  "  TJ_ptn[TJ_cg_size-1] = 0;\n"
	  );
  /*
  fprintf(codefile,
	  "  for(i = 0; i < TJ_cg_size; i++)\n"
	  "    fprintf(stderr,\"%%d \", TJ_ptn[i]);\n"
	  "  fprintf(stderr, \"\n\");\n"
	  );
  */
  /* init graph */
  fprintf(codefile,
	  "  int v;\n"
	  "  set *gv;\n"
	  );
  var_num = 1;
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  v = %s.TJ_nauty_make_graph_full();\n"
	    "  gv = GRAPHROW(TJ_cg,%d,TJ_cg_words);\n"
	    "  ADDELEMENT(gv,v);\n"
	    "  gv = GRAPHROW(TJ_cg,v,TJ_cg_words);\n"
	    "  ADDELEMENT(gv,%d);\n",
	    var->mu_name,
	    var_num,
	    var_num);
    var_num++;
  }
  fprintf(codefile,
	  "#ifdef TJ_DEBUG\n"
	  "  print_cg();\n"
	  "#endif\n"
	  );
  /* unmake graph */
  var_num = 1;
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  v = %s.TJ_nauty_unmake_graph();\n"
	    "  gv = GRAPHROW(TJ_cg,%d,TJ_cg_words);\n"
	    "  DELELEMENT(gv,v);\n"
	    "  gv = GRAPHROW(TJ_cg,v,TJ_cg_words);\n"
	    "  DELELEMENT(gv,%d);\n",
	    var->mu_name,
	    var_num,
	    var_num);
    var_num++;
  }
  fprintf(codefile,
	  "}\n\n");


  fprintf(codefile,
	  "bool TJ_Nauty_initialized = false;\n"
	  );
  fprintf(codefile, "\n");

  fprintf(codefile, 
	  "void SymmetryClass::TJ_Nauty_Canonicalize(state* s)\n"
	  "{\n"
	  "  Perm.ResetToSimple();\n"
	  "  static DEFAULTOPTIONS(options);\n"
	  "  statsblk(stats);\n"
	  "  options.getcanon = TRUE;\n"
	  "  options.defaultptn = FALSE;\n"
	  "  options.writeautoms = FALSE;\n"
	  "  options.writemarkers = FALSE;\n"
	  "  if(!TJ_Nauty_initialized) {\n"
	  "    TJ_Nauty_init();\n"
	  "    TJ_Nauty_initialized = true;\n"
	  "  }\n"
	  );
  /* make graph */
  var_num = 1;
  fprintf(codefile,
	  "  int v;\n"
	  "  set *gv;\n");
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  v = %s.TJ_nauty_make_graph();\n"
	    "  gv = GRAPHROW(TJ_cg,%d,TJ_cg_words);\n"
	    "  ADDELEMENT(gv,v);\n"
	    "  gv = GRAPHROW(TJ_cg,v,TJ_cg_words);\n"
	    "  ADDELEMENT(gv,%d);\n",
	    var->mu_name,
	    var_num,
	    var_num);
    var_num++;
  }
  fprintf(codefile,
	  "  nauty(TJ_cg,TJ_lab,TJ_ptn,NILSET,TJ_orbits,&options,&stats,TJ_workspace,50*TJ_cg_words,TJ_cg_words,TJ_cg_size,TJ_cg_canon);\n"
	  );
  /*
  fprintf(codefile,
	  "  fprintf(stderr, \"%%f %%d %%d bad=%%d\\n\", stats.grpsize1, stats.numnodes, stats.maxlevel, stats.numbadleaves);\n"
	  );
  */
  /*
  fprintf(codefile,
	  "  if (stats.grpsize2 == 0)\n"
	  "    fprintf(stderr,\"grpsize=%%.0f\",stats.grpsize1+0.1);\n"
	  "  else {\n"
	  "    while (stats.grpsize1 >= 10.0) {\n"
	  "      stats.grpsize1 /= 10.0;\n"
	  "      ++stats.grpsize2;\n"
	  "    }\n"
	  "    fprintf(stderr,\"grpsize=%%12.10fe%%d\", stats.grpsize1,stats.grpsize2);\n"
	  "  }\n"
	  "  fprintf(stderr, \"\\n\");\n"
	  );
  */
  /* unmake graph */
  var_num = 1;
  for(variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  v = %s.TJ_nauty_unmake_graph();\n"
	    "  gv = GRAPHROW(TJ_cg,%d,TJ_cg_words);\n"
	    "  DELELEMENT(gv,v);\n"
	    "  gv = GRAPHROW(TJ_cg,v,TJ_cg_words);\n"
	    "  DELELEMENT(gv,%d);\n",
	    var->mu_name,
	    var_num,
	    var_num);
    var_num++;
  }

  /* make the canonical state */
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    fprintf(codefile,
	    "  extern int TJ_nauty_first_v_%s;\n",
	    ss->mu_name);
  }
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    fprintf(codefile,
	    "  Perm.undefined_class_%s = 0;\n",
	    ss->mu_name);
  }
  fprintf(codefile,
	  "  for(int i = 0; i < TJ_cg_size; i++) {\n"
	  "    int v = TJ_lab[i];\n");
  for(ssiter = scalarsetlist; ssiter; ssiter = ssiter->next) {
    ss = (scalarsettypedecl *)ssiter->s->getvalue();
    fprintf(codefile,
	    "    if(v >= TJ_nauty_first_v_%s &&\n"
	    "       v < TJ_nauty_first_v_%s + %d)\n"
	    "      Perm.class_%s[v - TJ_nauty_first_v_%s] =\n"
	    "       Perm.undefined_class_%s++;\n",
	    ss->mu_name, ss->mu_name, ss->getsize(),
	    ss->mu_name, ss->mu_name, ss->mu_name);
  }
  fprintf(codefile,
	  "  }\n");
  fprintf(codefile,
	  "  Perm.SimpleToOne();\n"
	  );
  for (variter = varlist; variter; variter = variter->next ) {
    var = variter->s->getvalue();
    fprintf(codefile,
	    "  %s.Permute(Perm, 0);\n",
	    var->mu_name
	    );
  }

  fprintf(codefile, "}\n");

  fprintf(codefile, "#endif //TJ_NAUTY\n\n");
}

/********************
  generate new algorithm IV :
  fast canonicalization for simple variable and simple scalarset array
  and also fast restriction in permutation set with simple scalarset array of scalarset
  and fast restriction for multiset
  ********************/
void symmetryclass::generate_heuristic_fast_canonicalization()
{
  stelist * var;
  bool ToExplicit;

  fprintf(codefile,
          "\n/********************\n"
          " Canonicalization by fast simple variable canonicalization,\n"
	  " fast simple scalarset array canonicalization,\n"
	  " fast restriction on permutation set with simple scalarset array of scalarset,\n"
	  " and fast exhaustive generation of\n"
          " all permutations for other variables\n"
          " ********************/\n");

  fprintf(codefile, 
	  "void SymmetryClass::Heuristic_Fast_Canonicalize(state* s)\n"
	  "{\n"
	  "  int i;\n"
	  "  static state temp;\n"
	  "\n"
          "  Perm.ResetToSimple();\n"
          "\n"
	  );

  // simple variable
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetVariable()
 	&& (var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetArrayOfFree
 	    || var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetVariable
 	    || var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree) )
      fprintf(codefile, 
	      "  %s.SimpleCanonicalize(Perm);\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // simple scalarset array
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetArrayOfFree()
        && ( var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetArrayOfFree
	     || var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree) )
      fprintf(codefile, 
	      "  %s.Canonicalize(Perm);\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // multiset
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  %s.MultisetSort();\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // if they are inside more complicated structure, only do a limit
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetVariable()
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.SimpleLimit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetArrayOfFree()
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.ArrayLimit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // refine Permutation set by checking graph structure of multiset array of Scalarset
  for ( var = varlist; var != NULL; var = var->next )
     if (var->s->getvalue()->gettype()->HasMultisetOfScalarset())
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.MultisetLimit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // refine Permutation set by checking graph structure of scalarset array of Scalarset
  for ( var = varlist; var != NULL; var = var->next )
     if (var->s->getvalue()->gettype()->HasScalarsetArrayOfS())
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.Limit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // checking if we need to change simple to explicit
  if (!no_need_for_perm)
    fprintf(codefile, 
	    "  Perm.SimpleToExplicit();\n"
	    "\n"
	    );

  // handle all other cases by explicit/exhaustive enumeration  
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() != typedecl::NoScalarset
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  StateCopy(&temp, workingstate);\n"
	      "  ResetBestResult();\n"
	      "  for (i=0; i<Perm.count; i++)\n"
	      "    if (Perm.In(i))\n"
	      "      {\n"
	      "        StateCopy(workingstate, &temp);\n"
	      "        %s.Permute(Perm,i);\n"
	      "        if (args->multiset_reduction.value)\n"
	      "          %s.MultisetSort();\n" 
	      "        SetBestResult(i, workingstate);\n"
	      "      }\n"
	      "  StateCopy(workingstate, &BestPermutedState);\n"
	      "\n",
	      var->s->getvalue()->mu_name,
	      var->s->getvalue()->mu_name
	      );
      
  fprintf(codefile,
	  "};\n"); 
}

/********************
  generate new algorithm V :
  fast canonicalization for simple variable and simple scalarset array
  and also fast restriction in permutation set with simple scalarset array of scalarset
  and fast restriction for multiset
  -- use less local memory
  ********************/
void symmetryclass::generate_heuristic_small_mem_canonicalization()
{
  stelist * var;
  bool ToExplicit;
  
  fprintf(codefile,
          "\n/********************\n"
          " Canonicalization by fast simple variable canonicalization,\n"
	  " fast simple scalarset array canonicalization,\n"
	  " fast restriction on permutation set with simple scalarset array of scalarset,\n"
	  " and fast exhaustive generation of\n"
          " all permutations for other variables\n"
	  " and use less local memory\n" 
          " ********************/\n");
  
  fprintf(codefile, 
	  "void SymmetryClass::Heuristic_Small_Mem_Canonicalize(state* s)\n"
	  "{\n"
	  "  unsigned long cycle;\n"
	  "  static state temp;\n"
	  "\n"
          "  Perm.ResetToSimple();\n"
          "\n"
	  );

  // simple variable
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetVariable()
 	&& (var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetArrayOfFree
 	    || var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetVariable
 	    || var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree) )
      fprintf(codefile, 
	      "  %s.SimpleCanonicalize(Perm);\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // simple scalarset array
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetArrayOfFree()
        && ( var->s->getvalue()->gettype()->getstructure() == typedecl::ScalarsetArrayOfFree
	     || var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree) )
      fprintf(codefile, 
	      "  %s.Canonicalize(Perm);\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // multiset
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() == typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  %s.MultisetSort();\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // if they are inside more complicated structure, only do a limit
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetVariable()
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.SimpleLimit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->HasScalarsetArrayOfFree()
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.ArrayLimit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // refine Permutation set by checking graph structure of multiset array of Scalarset
  for ( var = varlist; var != NULL; var = var->next )
     if (var->s->getvalue()->gettype()->HasMultisetOfScalarset())
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.MultisetLimit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // refine Permutation set by checking graph structure of scalarset array of Scalarset
  for ( var = varlist; var != NULL; var = var->next )
     if (var->s->getvalue()->gettype()->HasScalarsetArrayOfS())
      fprintf(codefile, 
	      "  if (Perm.MoreThanOneRemain()) {\n"
	      "    %s.Limit(Perm);\n"
	      "  }\n"
	      "\n",
	      var->s->getvalue()->mu_name
	      );

  // checking if we need to change simple to explicit
  ToExplicit = FALSE;
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() != typedecl::NoScalarset
        && var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      ToExplicit = TRUE;
  if (ToExplicit)
    fprintf(codefile, 
	    "  Perm.SimpleToOne();\n"
	    "\n"
	    "  StateCopy(&temp, workingstate);\n"
	    "  ResetBestResult();\n"
	    );

  // handle all other cases by explicit/exhaustive enumeration  
  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() != typedecl::NoScalarset
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "  %s.Permute(Perm,0);\n"
	      "  if (args->multiset_reduction.value)\n"
	      "    %s.MultisetSort();\n",
	      var->s->getvalue()->mu_name,
	      var->s->getvalue()->mu_name
	      );
      
  if (!no_need_for_perm)
    fprintf(codefile, 
	    "  BestPermutedState = *workingstate;\n"
	    "  BestInitialized = TRUE;\n"
	    "\n"
	    "  cycle=1;\n"
	    "  while (Perm.NextPermutation())\n"
	    "    {\n"
	    "      if (args->perm_limit.value != 0\n"
	    "          && cycle++ >= args->perm_limit.value) break;\n"
	    "      StateCopy(workingstate, &temp);\n"
	    );

  for ( var = varlist; var != NULL; var = var->next )
    if (var->s->getvalue()->gettype()->getstructure() != typedecl::NoScalarset
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetVariable
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::ScalarsetArrayOfFree
	&& var->s->getvalue()->gettype()->getstructure() != typedecl::MultisetOfFree)
      fprintf(codefile, 
	      "      %s.Permute(Perm,0);\n"
	      "      if (args->multiset_reduction.value)\n"
	      "        %s.MultisetSort();\n",
	      var->s->getvalue()->mu_name,
	      var->s->getvalue()->mu_name
	      );
      
  if (!no_need_for_perm)
    fprintf(codefile,
	    "      switch (StateCmp(workingstate, &BestPermutedState)) {\n"
	    "      case -1:\n"
	    "        BestPermutedState = *workingstate;\n"
	    "        break;\n"
	    "      case 1:\n"
	    "        break;\n"
	    "      case 0:\n"
	    "        break;\n"
	    "      default:\n"
	    "        Error.Error(\"funny return value from StateCmp\");\n"
	    "      }\n"
	    "    }\n"
	    "  StateCopy(workingstate, &BestPermutedState);\n"
	    "\n");
  fprintf(codefile,
	  "};\n"); 
}

/****************************************
  * 18 Nov 93 Norris Ip: 
  basic structures
  * 26 Nov 93 Norris Ip: 
  form varlist in class -> set_var_list(globals);
  rearrange varlist according to complexity of type and
  order of declaration in program.
  * 2 Dec 93 Norris Ip:
  generate perm class
  * 3 Dec 93 Norris Ip:
  generate simple exhaustive canonicalization
  * 7 Dec 93 Norris Ip:
  added typedecl::generate_permute_function()
  to complete the simple exhaustive canonicalization
  *** completed the old simple/fast exhaustive canonicalization
  *** for one scalarset
  * 8 Dec 93 Norris Ip:
  added typedecl::generate_canonicalize_function()
  for simple varible only
           void symmetryclass::generate_fast_canonicalization_1()
  * 13 Dec 93 Norris Ip:
  added typedecl::generate_canonicalize_function()
  for simple scalarset array (inefficient)
           void symmetryclass::generate_fast_canonicalization_2()
  * 13 Dec 93 Norris Ip:
  generate class set for easy manipulation of set
  * 14 Dec 93 Norris Ip:
  added typedecl::generate_canonicalize_function()
  for simple scalarset array
  using fast simple variable/array canonicalization and other
  variables using fast exhaustive
  * 20 Dec 93 Norris Ip:
  move auxiliary code to cpp_sym_aux.C and cpp_sym_decl.C
  created scalarset array of scalarset permutation set restriction
           void symmetryclass::generate_fast_canonicalization_3()
  * 31 Jan 94 Norris Ip:
  removed useless scalarset types from scalarsetlist
  * 22 Feb 94 Norris Ip:
  add fast normalize code
  * 25 Feb 94 Norris Ip:
  Fixes bugs about sym alg executed in the wrong order
  Added hasScalarsetVariable() and HasScalarsetArrayOfFree()
  * 6 April 94 Norris Ip:
  generate code for Match() for correct error trace printing
beta2.9S released
  * 14 July 95 Norris Ip:
  Tidying/removing old algorithm
  adding small memory algorithm for larger scalarsets
****************************************/

/********************
 $Log: cpp_sym.C,v $
 Revision 1.2  1999/01/29 07:49:12  uli
 bugfixes

 Revision 1.4  1996/08/07 18:54:00  ip
 last bug fix on NextRule/SetNextEnabledRule has a bug; fixed this turn

 Revision 1.3  1996/08/07 00:59:13  ip
 Fixed bug on what_rule setting during guard evaluation; otherwise, bad diagnoistic message on undefine error on guard

 Revision 1.2  1996/08/06 23:57:39  ip
 fixed while code generation bug

 Revision 1.1  1996/08/06 23:56:55  ip
 Initial revision

 ********************/
